/*
 * ====================================================================
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 *
 */
package org.apache.http.benchmark;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileOutputStream;
import java.io.FileReader;
import java.io.IOException;
import java.net.URL;
import java.security.cert.CertificateException;
import java.security.cert.X509Certificate;
import java.util.UUID;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

import javax.net.SocketFactory;
import javax.net.ssl.SSLContext;

import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.CommandLineParser;
import org.apache.commons.cli.Options;
import org.apache.commons.cli.PosixParser;
import org.apache.http.Header;
import org.apache.http.HttpEntity;
import org.apache.http.HttpHost;
import org.apache.http.HttpRequest;
import org.apache.http.HttpVersion;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.FileEntity;
import org.apache.http.entity.StringEntity;
import org.apache.http.entity.mime.MultipartRelatedEntity;
import org.apache.http.entity.mime.content.FileBody;
import org.apache.http.entity.mime.content.FileBodyTemplate;
import org.apache.http.message.BasicHttpEntityEnclosingRequest;
import org.apache.http.message.BasicHttpRequest;
import org.apache.http.protocol.HTTP;
import org.apache.http.ssl.SSLContextBuilder;
import org.apache.http.ssl.TrustStrategy;

/**
 * Main program of the HTTP benchmark.
 *
 *
 * @since 4.0
 */
public class HttpBenchmark {

    private final Config config;

    public static void main(final String[] args) throws Exception {

        final Options options = CommandLineUtils.getOptions();
        final CommandLineParser parser = new PosixParser();
        final CommandLine cmd = parser.parse(options, args);

        if (args.length == 0 || cmd.hasOption('h') || cmd.getArgs().length != 1) {
            CommandLineUtils.showUsage(options);
            System.exit(1);
        }

        final Config config = new Config();
        CommandLineUtils.parseCommandLine(cmd, config);

        if (config.getUrl() == null) {
            CommandLineUtils.showUsage(options);
            System.exit(1);
        }

        final HttpBenchmark httpBenchmark = new HttpBenchmark(config);
        final String risultato = httpBenchmark.execute();
        if (cmd.hasOption('f')) {
            final FileOutputStream fout = new FileOutputStream(config.getReportFile());
            fout.write(risultato.getBytes());
            fout.flush();
            fout.close();
        } else {
            System.out.println(risultato);
        }

    }

    private StringBuffer header = null;
    private FileEntity payload = null;
    BustaContentProducer cp  = null;

    private StringBuffer xmlTemplate = null;
    private XmlTemplateContentProducer xmlP;

    public HttpBenchmark(final Config config) {
        super();
        this.config = config != null ? config : new Config();
    }

    private HttpRequest createRequest() {
        final URL url = config.getUrl();
        HttpEntity entity = null;

        // Prepare requests for each thread
        if (config.getPayloadFile() != null) {

            if (config.getBustaFileHeader()!=null) {
                try {
                    this.loadfilesBusta(config);
                } catch (final IOException e) {
                    // TODO Auto-generated catch block
                    e.printStackTrace();
                    System.exit(1);
                }
                this.cp = new BustaContentProducer(header, payload);

                if (config.getAttachmentsDir() == null) {
                    entity = new XMLEntityTemplate(this.cp, config.getContentType());
                } else {
                    final FileBodyTemplate root = new FileBodyTemplate(this.cp, config.getContentType());
                    final MultipartRelatedEntity mentity = new MultipartRelatedEntity(root);
                    File f = config.getAttachmentsDir();
                    if (!f.isAbsolute()) {
                        f = f.getAbsoluteFile();
                    }
                    for (final File filename : f.listFiles()) {
                        if (filename.isDirectory())
                            continue;

                        String mime = null;
                        try {
                            final org.apache.http.entity.mime.MimeTypes mimeTypes = org.apache.http.entity.mime.MimeTypes.getInstance();
                            mime = "application/octet-stream";
                            if(filename.getName().contains(".")){
                                final String ext = filename.getName().substring(filename.getName().lastIndexOf('.')+1, filename.getName().length());
                                mime = mimeTypes.getMimeType(ext);
                                if(mime==null){
                                    mime = "application/octet-stream";
                                }
                            }
                        } catch (final Exception e) {
                            // TODO Auto-generated catch block
                            e.printStackTrace();
                            System.exit(1);
                        }

                        final FileBody contentBody = new FileBody(filename, org.apache.http.entity.ContentType.create(mime, "UTF8"));
                        mentity.addPart(UUID.randomUUID().toString(), contentBody);
                    }

                    entity = mentity;

                    // Debug contenuto inviato
//                    try {
//                        mentity.writeTo(new java.io.FileOutputStream("/tmp/pippo.txt"));
//                    } catch (final java.io.FileNotFoundException e) {
//                        // TODO Auto-generated catch block
//                        e.printStackTrace();
//                    } catch (final java.io.IOException e) {
//                        // TODO Auto-generated catch block
//                        e.printStackTrace();
//                    }

                }
            } else {

                if (config.isPayloadFileTemplate()) {
                    try {
                        this.loadfilesXmlTemplate(config);
                    } catch (final IOException e) {
                        // TODO Auto-generated catch block
                        e.printStackTrace();
                        System.exit(1);
                    }
                    this.xmlP = new XmlTemplateContentProducer(xmlTemplate);
                }

                if (config.getAttachmentsDir() == null) {
                    if (config.isPayloadFileTemplate()) {
                        entity = new XMLEntityTemplate(this.xmlP, config.getContentType());
                    }
                    else{

            final FileEntity fe = new FileEntity(config.getPayloadFile());
            fe.setContentType(config.getContentType());
            fe.setChunked(config.isUseChunking());
            entity = fe;

                    }
                } else {

                    // Ottengo ContentType senza ';'
                    final String contentType = config.getContentType().contains(";") ? config.getContentType().split(";")[0] : config.getContentType();

                    MultipartRelatedEntity mentity = null;
                    if (config.isPayloadFileTemplate()) {
                        final FileBodyTemplate root = new FileBodyTemplate(this.xmlP, contentType);
                        mentity = new MultipartRelatedEntity(root);
                    }
                    else{
                        @SuppressWarnings("deprecation")
                        final FileBody root = new FileBody(config.getPayloadFile(), contentType);
                        mentity = new MultipartRelatedEntity(root);
                    }

                    File f = config.getAttachmentsDir();
                    if (!f.isAbsolute()) {
                        f = f.getAbsoluteFile();
                    }
                    for (final File filename : f.listFiles()) {
                        if (filename.isDirectory())
                            continue;

                        String mime = null;
                        try {
                            final org.apache.http.entity.mime.MimeTypes mimeTypes = org.apache.http.entity.mime.MimeTypes.getInstance();
                            mime = "application/octet-stream";
                            if(filename.getName().contains(".")){
                                final String ext = filename.getName().substring(filename.getName().lastIndexOf('.')+1, filename.getName().length());
                                mime = mimeTypes.getMimeType(ext);
                                if(mime==null){
                                    mime = "application/octet-stream";
                                }
                            }
                        } catch (final Exception e) {
                            // TODO Auto-generated catch block
                            e.printStackTrace();
                            System.exit(1);
                        }

                        final FileBody contentBody = new FileBody(filename, org.apache.http.entity.ContentType.create(mime, "UTF8"));
                        mentity.addPart(UUID.randomUUID().toString(), contentBody);

                    }

                    entity = mentity;

                }

            }

        } else if (config.getPayloadText() != null) {
            final StringEntity se = new StringEntity(config.getPayloadText(),
                    ContentType.parse(config.getContentType()));
            se.setChunked(config.isUseChunking());
            entity = se;
        }
        final HttpVersion ver = config.isUseHttp1_0() ? HttpVersion.HTTP_1_0 : HttpVersion.HTTP_1_1;
        final HttpRequest request;
        if ("POST".equals(config.getMethod())) {
            final BasicHttpEntityEnclosingRequest httppost =
                    new BasicHttpEntityEnclosingRequest("POST", url.getPath(), ver);
            httppost.setEntity(entity);
            request = httppost;
        } else if ("PUT".equals(config.getMethod())) {
            final BasicHttpEntityEnclosingRequest httpput =
                    new BasicHttpEntityEnclosingRequest("PUT", url.getPath(), ver);
            httpput.setEntity(entity);
            request = httpput;
        } else if ("PATCH".equals(config.getMethod()) || "LINK".equals(config.getMethod()) || "UNLINK".equals(config.getMethod())) {
            final BasicHttpEntityEnclosingRequest httpput =
                    new BasicHttpEntityEnclosingRequest(config.getMethod(), url.getPath(), ver);
            httpput.setEntity(entity);
            request = httpput;
        } else {
            String path = url.getPath();
            if (url.getQuery() != null && url.getQuery().length() > 0) {
                path += "?" + url.getQuery();
            } else if (path.trim().isEmpty()) {
                path = "/";
            }
            request = new BasicHttpRequest(config.getMethod(), path, ver);
        }

        if (!config.isKeepAlive()) {
            request.addHeader(new DefaultHeader(HTTP.CONN_DIRECTIVE, HTTP.CONN_CLOSE));
        }

        final String[] headers = config.getHeaders();
        if (headers != null) {
            for (final String s : headers) {
                final int pos = s.indexOf(':');
                if (pos != -1) {
                    final Header header = new DefaultHeader(s.substring(0, pos).trim(), s.substring(pos + 1));
                    request.addHeader(header);
                }
            }
        }

        if (config.isUseAcceptGZip()) {
            request.addHeader(new DefaultHeader("Accept-Encoding", "gzip"));
        }

        if (config.getSoapAction() != null && config.getSoapAction().length() > 0) {
            request.addHeader(new DefaultHeader("SOAPAction", config.getSoapAction()));
        }
        return request;
    }

    public String execute() throws Exception {
        final Results results = doExecute();
        return ResultProcessor.printResults(results);
    }

    public Results doExecute() throws Exception {

        final URL url = config.getUrl();
        final HttpHost host = new HttpHost(url.getHost(), url.getPort(), url.getProtocol());

        final ThreadPoolExecutor workerPool = new ThreadPoolExecutor(
                config.getThreads(), config.getThreads(), 5, TimeUnit.SECONDS,
            new LinkedBlockingQueue<Runnable>(),
            new ThreadFactory() {

                @Override
                public Thread newThread(final Runnable r) {
                    return new Thread(r, "ClientPool");
                }

            });
        workerPool.prestartAllCoreThreads();

        SocketFactory socketFactory = null;
        if ("https".equals(host.getSchemeName())) {
            final SSLContextBuilder sslContextBuilder = new SSLContextBuilder();
            sslContextBuilder.setProtocol("SSL");
            if (config.isDisableSSLVerification()) {
                sslContextBuilder.loadTrustMaterial(null, new TrustStrategy() {

                    @Override
                    public boolean isTrusted(
                            final X509Certificate[] chain, final String authType) throws CertificateException {
                        return true;
                    }

                });
            } else if (config.getTrustStorePath() != null) {
                sslContextBuilder.loadTrustMaterial(
                        new File(config.getTrustStorePath()),
                        config.getTrustStorePassword() != null ? config.getTrustStorePassword().toCharArray() : null);
            }
            if (config.getIdentityStorePath() != null) {
                sslContextBuilder.loadKeyMaterial(
                        new File(config.getIdentityStorePath()),
                        config.getIdentityStorePassword() != null ? config.getIdentityStorePassword().toCharArray() : null,
                        config.getIdentityStorePassword() != null ? config.getIdentityStorePassword().toCharArray() : null);
            }
            final SSLContext sslContext = sslContextBuilder.build();
            socketFactory = sslContext.getSocketFactory();
        }

        final BenchmarkWorker[] workers = new BenchmarkWorker[config.getThreads()];
        for (int i = 0; i < workers.length; i++) {
            workers[i] = new BenchmarkWorker(
                    createRequest(),
                    host,
                    socketFactory,
                    config);
            workerPool.execute(workers[i]);
        }

        if (config.getDurationSec() > 0) {
            // wait duration seconds
            // System.out.println("wait "+duration_sec);
            try {
                Thread.sleep(config.getDurationSec()  * 1000);
            } catch (final InterruptedException ie) {
            }
            // Signal timeout to workers
            for (int i = 0; i < config.getThreads(); i++) {
                workers[i].setTimedOut();
            }
        }

        while (workerPool.getCompletedTaskCount() < config.getThreads()) {
            Thread.yield();
            try {
                Thread.sleep(1000);
            } catch (final InterruptedException ignore) {
            }
        }

        workerPool.shutdown();

        if(this.cp!=null)
            this.cp.stop();

        if(this.xmlP!=null)
            this.xmlP.stop();

        return ResultProcessor.collectResults(workers, host, config.getUrl().toString());
    }

    private void loadfilesBusta(final Config config)  throws IOException {

        //String dir = config.getPayloadFile().getParent();
        final File headerfile = config.getBustaFileHeader();
        //        if (dir != null)
        //            headerfile = new File(new File(dir),"headeregov.xml");
        //        else
        //            headerfile = new File("headeregov.xml");

        final BufferedReader br;
        String line;


        //System.out.println(".header:"+header);
        if (this.header == null){
            header=new StringBuffer();

            br = new BufferedReader(new FileReader(headerfile));

            while ((line = br.readLine()) != null)
                header.append(line).append('\n');
            br.close();

        }
        if (payload == null) {
            payload=new FileEntity(config.getPayloadFile());
            payload.setContentType(config.getContentType());
        }

    }

    private void loadfilesXmlTemplate(final Config config)  throws IOException {

        //String dir = config.getPayloadFile().getParent();
        final File payloadFile = config.getPayloadFile();
        //        if (dir != null)
        //            headerfile = new File(new File(dir),"headeregov.xml");
        //        else
        //            headerfile = new File("headeregov.xml");

        final BufferedReader br;
        String line;

        //System.out.println(".header:"+header);
        if (this.xmlTemplate == null){
            xmlTemplate=new StringBuffer();

            br = new BufferedReader(new FileReader(payloadFile));

            while ((line = br.readLine()) != null)
                xmlTemplate.append(line).append('\n');
            br.close();

        }

    }

}
