/*
 * ====================================================================
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 *
 */
package org.apache.http.benchmark;

import java.util.ArrayList;
import java.util.Collections;

/**
 * Helper to gather statistics for an {@link HttpBenchmark HttpBenchmark}.
 *
 *
 * @since 4.0
 */
public class Stats {

    private long startTime = -1;    // nano seconds - does not represent an actual time
    private long finishTime = -1;   // nano seconds - does not represent an actual time
    private int successCount = 0;
    private int failureCount = 0;
    private int writeErrors = 0;
    private int keepAliveCount = 0;
    private String serverName = null;
    private long totalBytesRecv = 0;
    private long totalBytesSent = 0;
    private long contentLength = -1;
    private ArrayList<Long> times = new ArrayList<Long>();
    private boolean ordered = false;

    public Stats() {
        super();
    }

    public void start() {
        this.startTime = System.nanoTime();
    }

    public void finish() {
        this.finishTime = System.nanoTime();
    }

    public long getFinishTime() {
        return this.finishTime;
    }

    public long getStartTime() {
        return this.startTime;
    }

    /**
     * Total execution time measured in nano seconds
     *
     * @return duration in nanoseconds
     */
    public long getDuration() {
        // we are using System.nanoTime() and the return values could be negative
        // but its only the difference that we are concerned about
        return this.finishTime - this.startTime;
    }

    public void incSuccessCount() {
        this.successCount++;
    }

    public int getSuccessCount() {
        return this.successCount;
    }

    public void incFailureCount() {
        this.failureCount++;
    }

    public int getFailureCount() {
        return this.failureCount;
    }

    public void incWriteErrors() {
        this.writeErrors++;
    }

    public int getWriteErrors() {
        return this.writeErrors;
    }

    public void incKeepAliveCount() {
        this.keepAliveCount++;
    }

    public int getKeepAliveCount() {
        return this.keepAliveCount;
    }

    public long getTotalBytesRecv() {
        return this.totalBytesRecv;
    }

    public void incTotalBytesRecv(final long n) {
        this.totalBytesRecv += n;
    }

    public long getTotalBytesSent() {
        return this.totalBytesSent;
    }

    public void incTotalBytesSent(final long n) {
        this.totalBytesSent += n;
    }

    public long getContentLength() {
        return this.contentLength;
    }

    public void setContentLength(final long contentLength) {
        this.contentLength = contentLength;
    }

    public String getServerName() {
        return this.serverName;
    }

    public void setServerName(final String serverName) {
        this.serverName = serverName;
    }

    public void addInvocationTime(final long duration) {
        this.times.add(duration);
        ordered = false;
    }

    public long getPercentile(final int perc) {
        // http://www.allinterview.com/showanswers/61469.html
        // The 90th percentile tells you the value for which 90% of
        // the data points are smaller and 10% are bigger.
        // Statistically, to calculate the 90th percentile value:
        // 1. Sort the transaction instances by their value.
        // 2. Remove the top 10% instances.
        // 3. The highest value left is the 90th percentile.
        //
        // Example:
        // There are ten instances of transaction "t1" with the values
        // 1,3,2,4,5,20,7,8,9,6 (in sec).
        // 1. Sort by value -- 1,2,3,4,5,6,7,8,9,20.
        // 2. Remove top 10 % -- remove the value "20."
        // 3. The highest value left is the 90th percentile -- 9 is
        // the 90th percentile value.
        long perc_time = 0;
        if (times.size() == 0)
            return perc_time;
        if (!ordered) {
            Collections.sort(times);
            ordered = true;
        }
        final double percentile_position = perc * (times.size() + 1) / 100;

        final int i = (int) Math.floor(percentile_position);
        // System.out.println(i);
        // ArrayList are 0-indexed
        try {
            perc_time = times.get(i - 1);
        } catch (final ArrayIndexOutOfBoundsException aiobe) {
            aiobe.printStackTrace();
        }
        return perc_time;
    }

    public long getNinetyPercentile() {
        return getPercentile(90);
    }

    public ArrayList<Long> getAllTimes() {
        return times;
    }

    public double getRPSfromTimes() {
        double sum = 0;
        if (times.size() == 0)
            return sum;
        for (final long x : times)
            sum += x;
        // Times are in ns; must be converted in sec.
        final double res = times.size() / (sum / 1000000000);
        return res;
    }

    public long getMinInvocationTime() {
        if (!ordered) {
            Collections.sort(times);
            ordered = true;
        }
        return (times.size() > 0) ? times.get(0) : 0L;
    }

    public long getMaxInvocationTime() {
        if (!ordered) {
            Collections.sort(times);
            ordered = true;
        }
        return (times.size() > 0) ? times.get(times.size() - 1) : 0L;
    }

    public long getFrequencyIn(final long start, final long end) {
        long sum = 0;
        if (start < end) {
            for (final long x : times)
                if (x > start && x <= end)
                    sum++;
        } else if (start == end)
            for (final long x : times)
                if (x == start)
                    sum++;
        return sum;
    }

}
