/*
 * ====================================================================
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 *
 */

package org.apache.http.entity.mime;

import java.io.BufferedReader;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.util.Hashtable;

public class MimeTypes {

    private Hashtable<String, Object> mapMimeToExt = new Hashtable<String, Object>();
    private Hashtable<String, String> mapExtToMime = new Hashtable<String, String>();

    MimeTypes() throws Exception{

        InputStream is =null;
        BufferedReader br = null;
        InputStreamReader ir = null;
        try{
            final String file = "/org/apache/http/entity/mime/mime.types";
            is = MimeTypes.class.getResourceAsStream(file);
            if(is==null){
                throw new Exception("File ["+file+"] in classpath not found");
            }

            ir = new InputStreamReader(is);
            br = new BufferedReader(ir);
            String line;
            while ((line = br.readLine()) != null) {
                line = line.trim();
                if(!line.startsWith("#") && !line.startsWith("/*") && !line.startsWith("*/") && !line.startsWith(" *")){
                    if(line.contains("\t")){
                        throw new Exception("Line["+line+"] contains tabs");
                    }
                    final String [] tmp = line.split(" ");
                    if(tmp.length<2){
                        //System.out.println("TYPE["+tmp[0]+"] without exts");
                        this.mapMimeToExt.put(tmp[0].trim(), "NULL");
                    }else{
                        final StringBuffer bf = new StringBuffer();
                        for (int i = 1; i < tmp.length; i++) {
                            bf.append(" EXT-"+i+"=["+tmp[i].trim()+"]");
                            this.mapExtToMime.put(tmp[i].trim(),tmp[0].trim());
                        }
                        //System.out.println("TYPE["+tmp[0]+"]"+bf.toString());
                        this.mapMimeToExt.put(tmp[0].trim(), tmp[1].trim());
                    }
                }
            }

        }catch(final Exception e){
            throw new Exception(e.getMessage(),e);
        }finally{
            try{
                if(br!=null){
                    br.close();
                }
            }catch(final Exception eClose){}
            try{
                if(ir!=null){
                    ir.close();
                }
            }catch(final Exception eClose){}
            try{
                if(is!=null){
                    is.close();
                }
            }catch(final Exception eClose){}
        }
    }

    public String getMimeType(final String ext){
        return this.mapExtToMime.get(ext.trim().toLowerCase());
    }
    public String getExtension(final String mime){
        final Object o = this.mapMimeToExt.get(mime.trim().toLowerCase());
        if(o==null){
            return null;
        }
        if(o instanceof String && ((String)o).equals("NULL")){
            return null;
        }
        return (String) o;
    }

    public boolean existsExtension(final String ext){
        return this.mapExtToMime.containsKey(ext.trim().toLowerCase());
    }
    public boolean existsMimeType(final String mime, final boolean checkExistsExtension){
        if(this.mapMimeToExt.containsKey(mime.trim().toLowerCase())){
            if(checkExistsExtension){
                final String ext = this.getExtension(mime);
                return ext!=null;
            }
            else{
                return true;
            }
        }else{
            return false;
        }
    }

    // static

    private static MimeTypes mimeTypes = null;
    private static synchronized void init() throws Exception{
        if(mimeTypes==null){
            mimeTypes = new MimeTypes();
        }
    }
    public static MimeTypes getInstance() throws Exception{
        if(mimeTypes==null){
            init();
        }
        return mimeTypes;
    }

}
