/*
 * ====================================================================
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 *
 */
package org.apache.http.entity.mime;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.Iterator;
import java.util.List;
import java.util.Random;

import org.apache.http.Header;
import org.apache.http.HttpEntity;
import org.apache.http.entity.mime.content.ContentBody;
import org.apache.http.message.BasicHeader;
import org.apache.http.protocol.HTTP;
import org.apache.james.mime4j.field.Fields;
import org.apache.james.mime4j.message.MessageImpl;
import org.apache.james.mime4j.dom.Message;

public class MultipartRelatedEntity implements HttpEntity {

    /**
     * The pool of ASCII chars to be used for generating a multipart boundary.
     */
    private final static char[] MULTIPART_CHARS = "-_1234567890abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ"
            .toCharArray();

    private final Message message;
    private final HttpMultipartRelated multipart;
    private final Header contentType;

    private long length;
    private volatile boolean dirty; // used to decide whether to recalculate
    // length

    public MultipartRelatedEntity(final ContentBody rootPart) {
        this(rootPart, null, HttpMultipartMode.STRICT);
    }

    public MultipartRelatedEntity(final ContentBody rootPart, final String boundary) {
        this(rootPart, boundary, HttpMultipartMode.STRICT);
    }

    public MultipartRelatedEntity(final ContentBody rootPart,
            final String boundary, final HttpMultipartMode modeParam) {
        super();

        HttpMultipartMode mode = modeParam;
        if (mode == null) {
            mode = HttpMultipartMode.STRICT;
        }

        this.multipart = new HttpMultipartRelated("related",null,boundary,mode);

        this.contentType = new BasicHeader(HTTP.CONTENT_TYPE,generateContentType(boundary, rootPart));
        this.dirty = true;

        this.message = new MessageImpl();
        final org.apache.james.mime4j.message.HeaderImpl header = new org.apache.james.mime4j.message.HeaderImpl();
        this.message.setHeader(header);



        //this.multipart.setParent(message);

        this.message.getHeader().addField(
                Fields.contentType(this.contentType.getValue()));
    }

    public void addPart(final String name, final ContentBody contentBody) {
        this.multipart.addBodyPart(new RelatedBodyPart(name, contentBody));
        this.dirty = true;
    }

    // RFC238 state:
    // The Multipart/Related content-type addresses the MIME representation
    // of compound objects. The object is categorized by a "type"
    // parameter. Additional parameters are provided to indicate a specific
    // starting body part or root and auxiliary information which may be
    // required when unpacking or processing the object.

    protected String generateContentType(final String boundary,final ContentBody rootpart) {
        final StringBuilder buffer = new StringBuilder();
        buffer.append("multipart/related; boundary=\"");
        if (boundary != null) {
            buffer.append(boundary);
        } else {
            final Random rand = new Random();
            final int count = rand.nextInt(11) + 30; // a random size from 30 to 40
            final int start_boundary = buffer.length();
            for (int i = 0; i < count; i++) {
                buffer.append(MULTIPART_CHARS[rand
                                              .nextInt(MULTIPART_CHARS.length)]);
            }
            this.multipart.setBoundary(buffer.substring(start_boundary, start_boundary+count));
        }

        final RelatedBodyPart rbp = new RelatedBodyPart("root", rootpart);

        this.multipart.addBodyPart(rbp);
        buffer.append("\"; type=\"");
        String content = rootpart.getMimeType();
        if(content.contains(";")){
            // elimino eventuali charset
            final String [] tmp = content.split(";");
            //for (int i = 0; i < tmp.length; i++) {
            if(tmp.length>0)
                content = tmp[0]; // dovrebbe essere il primo tipo.
            //    break;
            //}
        }
        buffer.append(content);
        buffer.append('"');

        //buffer.append("; start=\"");
        //buffer.append(rbp.getContentId());
        //buffer.append('"');

        return buffer.toString();

    }

    //@Override
    public void consumeContent() throws IOException {
        if (isStreaming()) {
            throw new UnsupportedOperationException(
                    "Streaming entity does not implement #consumeContent()");
        }

    }

    //@Override
    public InputStream getContent() throws IOException, IllegalStateException {
        //        throw new UnsupportedOperationException(
        //                "Multipart form entity does not implement #getContent()");
        final ByteArrayOutputStream bout = new ByteArrayOutputStream();
        this.writeTo(bout);
        bout.flush();
        bout.close();
        return new ByteArrayInputStream(bout.toByteArray());
    }

    //@Override
    public Header getContentEncoding() {
        return null;
    }

    //@Override
    public long getContentLength() {
        if (this.dirty) {
            this.length = this.multipart.getTotalLength();
            this.dirty = false;
        }
        return this.length;
    }

    public void setDirty(final boolean dirty) {
        this.dirty = dirty;
    }

    //@Override
    public Header getContentType() {
        return this.contentType;

    }

    //@Override
    public boolean isChunked() {
        return !isRepeatable();
    }

    //@Override
    public boolean isRepeatable() {
        final List<?> parts = this.multipart.getBodyParts();
        for (final Iterator<?> it = parts.iterator(); it.hasNext();) {
            final RelatedBodyPart part = (RelatedBodyPart) it.next();
            final ContentBody body = (ContentBody) part.getBody();
            if (body.getContentLength() < 0) {
                return false;
            }
        }
        return true;
    }

    //@Override
    public boolean isStreaming() {
        return !isRepeatable();
    }

    //@Override
    public void writeTo(final OutputStream outstream) throws IOException {
        this.multipart.writeTo(outstream);
        //

    }

}
