

-- GovWay Config

-- **** Porte di Dominio ****

CREATE SEQUENCE seq_pdd MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE pdd
(
	nome VARCHAR2(255) NOT NULL,
	descrizione VARCHAR2(255),
	-- ip pubblico
	ip VARCHAR2(255),
	-- porta pubblico
	porta NUMBER,
	-- protocollo pubblico
	protocollo VARCHAR2(255),
	-- ip gestione
	ip_gestione VARCHAR2(255),
	-- porta gestione
	porta_gestione NUMBER,
	-- protocollo gestione
	protocollo_gestione VARCHAR2(255),
	-- Tipo della Porta
	tipo VARCHAR2(255),
	implementazione VARCHAR2(255),
	subject VARCHAR2(255),
	password VARCHAR2(255),
	-- client auth: disabilitato/abilitato
	client_auth VARCHAR2(255),
	ora_registrazione TIMESTAMP,
	superuser VARCHAR2(255),
	-- fk/pk columns
	id NUMBER NOT NULL,
	-- check constraints
	CONSTRAINT chk_pdd_1 CHECK (tipo IN ('operativo','nonoperativo','esterno')),
	-- unique constraints
	CONSTRAINT unique_pdd_1 UNIQUE (nome),
	-- fk/pk keys constraints
	CONSTRAINT pk_pdd PRIMARY KEY (id)
);


ALTER TABLE pdd MODIFY implementazione DEFAULT 'standard';
ALTER TABLE pdd MODIFY ora_registrazione DEFAULT CURRENT_TIMESTAMP;

CREATE TRIGGER trg_pdd
BEFORE
insert on pdd
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_pdd.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/




-- govway
		  		
-- **** Connettori ****

CREATE SEQUENCE seq_connettori MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE connettori
(
	-- (disabilitato,http,jms)
	endpointtype VARCHAR2(255) NOT NULL,
	nome_connettore VARCHAR2(2000) NOT NULL,
	-- url nel caso http
	url VARCHAR2(4000),
	-- nel caso di http indicazione se usare chunking
	transfer_mode VARCHAR2(255),
	transfer_mode_chunk_size NUMBER,
	-- nel caso di http indicazione se seguire il redirect o meno
	redirect_mode VARCHAR2(255),
	redirect_max_hop NUMBER,
	-- nome coda jms
	nome VARCHAR2(255),
	-- tipo coda jms (queue,topic)
	tipo VARCHAR2(255),
	-- utente di una connessione jms
	utente VARCHAR2(255),
	-- password per una connessione jms
	password VARCHAR2(255),
	enc_password CLOB,
	-- context property: initial_content
	initcont VARCHAR2(255),
	-- context property: url_pkg
	urlpkg VARCHAR2(255),
	-- context property: provider_url
	provurl VARCHAR2(255),
	-- ConnectionFactory JMS
	connection_factory VARCHAR2(255),
	-- Messaggio JMS inviato come text/byte message
	send_as VARCHAR2(255),
	-- 1/0 (true/false) abilita il debug tramite il connettore
	debug NUMBER,
	-- 1/0 (true/false) abilita l'utilizzo di un proxy tramite il connettore
	proxy NUMBER,
	proxy_type VARCHAR2(255),
	proxy_hostname VARCHAR2(255),
	proxy_port VARCHAR2(255),
	proxy_username VARCHAR2(255),
	proxy_password VARCHAR2(255),
	enc_proxy_password CLOB,
	-- Indicazione sull'intervallo massimo di tempo necessario per instaurare una connessione (intervallo in ms)
	connection_timeout NUMBER,
	-- Indicazione sull'intervallo massimo di tempo che può occorrere prima di ricevere una risposta (intervallo in ms)
	read_timeout NUMBER,
	-- Indicazione sull'intervallo massimo di tempo medio atteso prima di ricevere una risposta (intervallo in ms)
	avg_response_time NUMBER,
	-- 1/0 (true/false) indica se il connettore e' gestito tramite le proprieta' custom
	custom NUMBER,
	-- Gestione Token
	token_policy VARCHAR2(255),
	api_key CLOB,
	api_key_header VARCHAR2(255),
	app_id CLOB,
	app_id_header VARCHAR2(255),
	-- fk/pk columns
	id NUMBER NOT NULL,
	-- unique constraints
	CONSTRAINT unique_connettori_1 UNIQUE (nome_connettore),
	-- fk/pk keys constraints
	CONSTRAINT pk_connettori PRIMARY KEY (id)
);


ALTER TABLE connettori MODIFY debug DEFAULT 0;
ALTER TABLE connettori MODIFY proxy DEFAULT 0;
ALTER TABLE connettori MODIFY custom DEFAULT 0;

CREATE TRIGGER trg_connettori
BEFORE
insert on connettori
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_connettori.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/



CREATE SEQUENCE seq_connettori_custom MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE connettori_custom
(
	name VARCHAR2(255) NOT NULL,
	value VARCHAR2(4000) NOT NULL,
	enc_value CLOB,
	id_connettore NUMBER NOT NULL,
	-- fk/pk columns
	id NUMBER NOT NULL,
	-- unique constraints
	CONSTRAINT unique_connettori_custom_1 UNIQUE (id_connettore,name),
	-- fk/pk keys constraints
	CONSTRAINT fk_connettori_custom_1 FOREIGN KEY (id_connettore) REFERENCES connettori(id),
	CONSTRAINT pk_connettori_custom PRIMARY KEY (id)
);

-- index
CREATE INDEX idx_conn_custom_1 ON connettori_custom (id_connettore);
CREATE TRIGGER trg_connettori_custom
BEFORE
insert on connettori_custom
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_connettori_custom.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/



CREATE SEQUENCE seq_connettori_properties MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE connettori_properties
(
	-- nome connettore personalizzato attraverso file properties
	nome_connettore VARCHAR2(255) NOT NULL,
	-- location del file properties
	path VARCHAR2(255) NOT NULL,
	-- fk/pk columns
	id NUMBER NOT NULL,
	-- unique constraints
	CONSTRAINT unique_connettori_properties_1 UNIQUE (nome_connettore),
	-- fk/pk keys constraints
	CONSTRAINT pk_connettori_properties PRIMARY KEY (id)
);

CREATE TRIGGER trg_connettori_properties
BEFORE
insert on connettori_properties
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_connettori_properties.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/




-- govway
		  		
-- **** Connettori Gestione Errore ****

CREATE SEQUENCE seq_gestione_errore MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE gestione_errore
(
	-- accetta/rispedisci
	comportamento_default VARCHAR2(255),
	cadenza_rispedizione VARCHAR2(255),
	nome VARCHAR2(255) NOT NULL,
	-- fk/pk columns
	id NUMBER NOT NULL,
	-- unique constraints
	CONSTRAINT unique_gestione_errore_1 UNIQUE (nome),
	-- fk/pk keys constraints
	CONSTRAINT pk_gestione_errore PRIMARY KEY (id)
);

CREATE TRIGGER trg_gestione_errore
BEFORE
insert on gestione_errore
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_gestione_errore.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/



CREATE SEQUENCE seq_gestione_errore_trasporto MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE gestione_errore_trasporto
(
	id_gestione_errore NUMBER NOT NULL,
	valore_massimo NUMBER,
	valore_minimo NUMBER,
	-- accetta/rispedisci
	comportamento VARCHAR2(255),
	cadenza_rispedizione VARCHAR2(255),
	-- fk/pk columns
	id NUMBER NOT NULL,
	-- fk/pk keys constraints
	CONSTRAINT fk_gestione_errore_trasporto_1 FOREIGN KEY (id_gestione_errore) REFERENCES gestione_errore(id),
	CONSTRAINT pk_gestione_errore_trasporto PRIMARY KEY (id)
);

-- index
CREATE INDEX idx_gest_err_trasporto_1 ON gestione_errore_trasporto (id_gestione_errore);
CREATE TRIGGER trg_gestione_errore_trasporto
BEFORE
insert on gestione_errore_trasporto
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_gestione_errore_trasporto.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/



CREATE SEQUENCE seq_gestione_errore_soap MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE gestione_errore_soap
(
	id_gestione_errore NUMBER NOT NULL,
	fault_actor VARCHAR2(255),
	fault_code VARCHAR2(255),
	fault_string VARCHAR2(255),
	-- accetta/rispedisci
	comportamento VARCHAR2(255),
	cadenza_rispedizione VARCHAR2(255),
	-- fk/pk columns
	id NUMBER NOT NULL,
	-- fk/pk keys constraints
	CONSTRAINT fk_gestione_errore_soap_1 FOREIGN KEY (id_gestione_errore) REFERENCES gestione_errore(id),
	CONSTRAINT pk_gestione_errore_soap PRIMARY KEY (id)
);

-- index
CREATE INDEX idx_gest_err_soap_1 ON gestione_errore_soap (id_gestione_errore);
CREATE TRIGGER trg_gestione_errore_soap
BEFORE
insert on gestione_errore_soap
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_gestione_errore_soap.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/




-- govway
		  		
-- **** Gruppi ****

CREATE SEQUENCE seq_gruppi MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE gruppi
(
	nome VARCHAR2(255) NOT NULL,
	descrizione VARCHAR2(4000),
	service_binding VARCHAR2(255),
	superuser VARCHAR2(255),
	ora_registrazione TIMESTAMP,
	utente_richiedente VARCHAR2(255),
	data_creazione TIMESTAMP,
	utente_ultima_modifica VARCHAR2(255),
	data_ultima_modifica TIMESTAMP,
	-- fk/pk columns
	id NUMBER NOT NULL,
	-- check constraints
	CONSTRAINT chk_gruppi_1 CHECK (service_binding IN ('soap','rest')),
	-- unique constraints
	CONSTRAINT unique_gruppi_1 UNIQUE (nome),
	-- fk/pk keys constraints
	CONSTRAINT pk_gruppi PRIMARY KEY (id)
);


ALTER TABLE gruppi MODIFY ora_registrazione DEFAULT CURRENT_TIMESTAMP;

CREATE TRIGGER trg_gruppi
BEFORE
insert on gruppi
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_gruppi.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/




-- govway
		  		
-- **** Ruoli ****

CREATE SEQUENCE seq_ruoli MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE ruoli
(
	nome VARCHAR2(255) NOT NULL,
	descrizione VARCHAR2(4000),
	tipologia VARCHAR2(255) NOT NULL,
	nome_esterno VARCHAR2(255),
	contesto_utilizzo VARCHAR2(255) NOT NULL,
	superuser VARCHAR2(255),
	ora_registrazione TIMESTAMP,
	utente_richiedente VARCHAR2(255),
	data_creazione TIMESTAMP,
	utente_ultima_modifica VARCHAR2(255),
	data_ultima_modifica TIMESTAMP,
	-- fk/pk columns
	id NUMBER NOT NULL,
	-- check constraints
	CONSTRAINT chk_ruoli_1 CHECK (tipologia IN ('interno','esterno','qualsiasi')),
	CONSTRAINT chk_ruoli_2 CHECK (contesto_utilizzo IN ('portaDelegata','portaApplicativa','qualsiasi')),
	-- unique constraints
	CONSTRAINT unique_ruoli_1 UNIQUE (nome),
	-- fk/pk keys constraints
	CONSTRAINT pk_ruoli PRIMARY KEY (id)
);


ALTER TABLE ruoli MODIFY tipologia DEFAULT 'qualsiasi';
ALTER TABLE ruoli MODIFY contesto_utilizzo DEFAULT 'qualsiasi';
ALTER TABLE ruoli MODIFY ora_registrazione DEFAULT CURRENT_TIMESTAMP;

CREATE TRIGGER trg_ruoli
BEFORE
insert on ruoli
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_ruoli.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/




-- govway
		  		
-- **** Scope ****

CREATE SEQUENCE seq_scope MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE scope
(
	nome VARCHAR2(255) NOT NULL,
	descrizione VARCHAR2(4000),
	tipologia VARCHAR2(255),
	nome_esterno VARCHAR2(255),
	contesto_utilizzo VARCHAR2(255) NOT NULL,
	superuser VARCHAR2(255),
	ora_registrazione TIMESTAMP,
	utente_richiedente VARCHAR2(255),
	data_creazione TIMESTAMP,
	utente_ultima_modifica VARCHAR2(255),
	data_ultima_modifica TIMESTAMP,
	-- fk/pk columns
	id NUMBER NOT NULL,
	-- check constraints
	CONSTRAINT chk_scope_1 CHECK (contesto_utilizzo IN ('portaDelegata','portaApplicativa','qualsiasi')),
	-- unique constraints
	CONSTRAINT unique_scope_1 UNIQUE (nome),
	-- fk/pk keys constraints
	CONSTRAINT pk_scope PRIMARY KEY (id)
);


ALTER TABLE scope MODIFY contesto_utilizzo DEFAULT 'qualsiasi';
ALTER TABLE scope MODIFY ora_registrazione DEFAULT CURRENT_TIMESTAMP;

CREATE TRIGGER trg_scope
BEFORE
insert on scope
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_scope.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/




-- govway
		  		
-- **** Protocol Properties ****

CREATE SEQUENCE seq_protocol_properties MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE protocol_properties
(
	-- tipoProprietario
	tipo_proprietario VARCHAR2(255) NOT NULL,
	-- idOggettoProprietarioDocumento
	id_proprietario NUMBER NOT NULL,
	-- nome property
	name VARCHAR2(255) NOT NULL,
	-- valore come stringa
	value_string VARCHAR2(4000),
	-- valore come stringa cifrato
	enc_value_string CLOB,
	-- valore come numero
	value_number NUMBER,
	-- valore true o false
	value_boolean NUMBER,
	-- valore binario
	value_binary BLOB,
	file_name VARCHAR2(4000),
	-- fk/pk columns
	id NUMBER NOT NULL,
	-- unique constraints
	CONSTRAINT unique_protocol_properties_1 UNIQUE (tipo_proprietario,id_proprietario,name),
	-- fk/pk keys constraints
	CONSTRAINT pk_protocol_properties PRIMARY KEY (id)
);

CREATE TRIGGER trg_protocol_properties
BEFORE
insert on protocol_properties
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_protocol_properties.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/




-- govway
		  		
-- **** Soggetti ****

CREATE SEQUENCE seq_soggetti MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE soggetti
(
	nome_soggetto VARCHAR2(255) NOT NULL,
	tipo_soggetto VARCHAR2(255) NOT NULL,
	descrizione VARCHAR2(4000),
	identificativo_porta VARCHAR2(255),
	-- 1/0 (true/false) Indicazione se il soggetto svolge è quello di default per il protocollo
	is_default NUMBER,
	-- 1/0 (true/false) svolge attivita di router
	is_router NUMBER,
	id_connettore NUMBER NOT NULL,
	superuser VARCHAR2(255),
	server VARCHAR2(255),
	-- 1/0 (true/false) indica se il soggetto e' privato/pubblico
	privato NUMBER,
	ora_registrazione TIMESTAMP,
	profilo VARCHAR2(255),
	codice_ipa VARCHAR2(255) NOT NULL,
	tipoauth VARCHAR2(255),
	utente VARCHAR2(2800),
	password VARCHAR2(255),
	subject VARCHAR2(2800),
	cn_subject VARCHAR2(255),
	issuer VARCHAR2(2800),
	cn_issuer VARCHAR2(255),
	certificate BLOB,
	cert_strict_verification NUMBER,
	utente_richiedente VARCHAR2(255),
	data_creazione TIMESTAMP,
	utente_ultima_modifica VARCHAR2(255),
	data_ultima_modifica TIMESTAMP,
	pd_url_prefix_rewriter VARCHAR2(255),
	pa_url_prefix_rewriter VARCHAR2(255),
	-- fk/pk columns
	id NUMBER NOT NULL,
	-- unique constraints
	CONSTRAINT unique_soggetti_1 UNIQUE (nome_soggetto,tipo_soggetto),
	CONSTRAINT unique_soggetti_2 UNIQUE (codice_ipa),
	-- fk/pk keys constraints
	CONSTRAINT fk_soggetti_1 FOREIGN KEY (id_connettore) REFERENCES connettori(id),
	CONSTRAINT pk_soggetti PRIMARY KEY (id)
);


ALTER TABLE soggetti MODIFY is_default DEFAULT 0;
ALTER TABLE soggetti MODIFY is_router DEFAULT 0;
ALTER TABLE soggetti MODIFY privato DEFAULT 0;
ALTER TABLE soggetti MODIFY ora_registrazione DEFAULT CURRENT_TIMESTAMP;

CREATE TRIGGER trg_soggetti
BEFORE
insert on soggetti
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_soggetti.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/



CREATE SEQUENCE seq_soggetti_ruoli MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE soggetti_ruoli
(
	id_soggetto NUMBER NOT NULL,
	id_ruolo NUMBER NOT NULL,
	-- fk/pk columns
	id NUMBER NOT NULL,
	-- unique constraints
	CONSTRAINT unique_soggetti_ruoli_1 UNIQUE (id_soggetto,id_ruolo),
	-- fk/pk keys constraints
	CONSTRAINT fk_soggetti_ruoli_1 FOREIGN KEY (id_soggetto) REFERENCES soggetti(id),
	CONSTRAINT fk_soggetti_ruoli_2 FOREIGN KEY (id_ruolo) REFERENCES ruoli(id),
	CONSTRAINT pk_soggetti_ruoli PRIMARY KEY (id)
);

-- index
CREATE INDEX INDEX_SOGGETTI_RUOLI ON soggetti_ruoli (id_soggetto);
CREATE TRIGGER trg_soggetti_ruoli
BEFORE
insert on soggetti_ruoli
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_soggetti_ruoli.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/



CREATE SEQUENCE seq_soggetti_credenziali MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE soggetti_credenziali
(
	id_soggetto NUMBER NOT NULL,
	subject VARCHAR2(2800),
	cn_subject VARCHAR2(255),
	issuer VARCHAR2(2800),
	cn_issuer VARCHAR2(255),
	certificate BLOB,
	cert_strict_verification NUMBER,
	-- fk/pk columns
	id NUMBER NOT NULL,
	-- fk/pk keys constraints
	CONSTRAINT fk_soggetti_credenziali_1 FOREIGN KEY (id_soggetto) REFERENCES soggetti(id),
	CONSTRAINT pk_soggetti_credenziali PRIMARY KEY (id)
);

-- index
CREATE INDEX INDEX_SOGGETTI_CRED ON soggetti_credenziali (id_soggetto);
CREATE TRIGGER trg_soggetti_credenziali
BEFORE
insert on soggetti_credenziali
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_soggetti_credenziali.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/



CREATE SEQUENCE seq_soggetti_properties MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE soggetti_properties
(
	id_soggetto NUMBER NOT NULL,
	nome VARCHAR2(255) NOT NULL,
	valore VARCHAR2(4000) NOT NULL,
	enc_value CLOB,
	-- fk/pk columns
	id NUMBER NOT NULL,
	-- unique constraints
	CONSTRAINT uniq_soggetti_props_1 UNIQUE (id_soggetto,nome),
	-- fk/pk keys constraints
	CONSTRAINT fk_soggetti_properties_1 FOREIGN KEY (id_soggetto) REFERENCES soggetti(id),
	CONSTRAINT pk_soggetti_properties PRIMARY KEY (id)
);

-- index
CREATE INDEX INDEX_SOGGETTI_PROP ON soggetti_properties (id_soggetto);
CREATE TRIGGER trg_soggetti_properties
BEFORE
insert on soggetti_properties
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_soggetti_properties.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/




-- govway
		  		
-- **** Documenti ****

CREATE SEQUENCE seq_documenti MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE documenti
(
	ruolo VARCHAR2(255) NOT NULL,
	-- tipo (es. xsd,xml...)
	tipo VARCHAR2(255) NOT NULL,
	-- nome documento
	nome VARCHAR2(255) NOT NULL,
	-- contenuto documento
	contenuto BLOB NOT NULL,
	-- idOggettoProprietarioDocumento
	id_proprietario NUMBER NOT NULL,
	-- tipoProprietario
	tipo_proprietario VARCHAR2(255) NOT NULL,
	ora_registrazione TIMESTAMP NOT NULL,
	-- fk/pk columns
	id NUMBER NOT NULL,
	-- check constraints
	CONSTRAINT chk_documenti_1 CHECK (ruolo IN ('allegato','specificaSemiformale','specificaLivelloServizio','specificaSicurezza','specificaCoordinamento')),
	CONSTRAINT chk_documenti_2 CHECK (tipo_proprietario IN ('accordoServizio','accordoCooperazione','servizio')),
	-- unique constraints
	CONSTRAINT unique_documenti_1 UNIQUE (ruolo,tipo,nome,id_proprietario,tipo_proprietario),
	-- fk/pk keys constraints
	CONSTRAINT pk_documenti PRIMARY KEY (id)
);

-- index
CREATE INDEX INDEX_DOC_SEARCH ON documenti (id_proprietario);

ALTER TABLE documenti MODIFY ora_registrazione DEFAULT CURRENT_TIMESTAMP;

CREATE TRIGGER trg_documenti
BEFORE
insert on documenti
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_documenti.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/



-- **** Accordi di Servizio Parte Comune ****

CREATE SEQUENCE seq_accordi MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE accordi
(
	nome VARCHAR2(255) NOT NULL,
	descrizione VARCHAR2(4000),
	service_binding VARCHAR2(255) NOT NULL,
	message_type VARCHAR2(255),
	profilo_collaborazione VARCHAR2(255),
	formato_specifica VARCHAR2(255),
	wsdl_definitorio CLOB,
	wsdl_concettuale CLOB,
	wsdl_logico_erogatore CLOB,
	wsdl_logico_fruitore CLOB,
	spec_conv_concettuale CLOB,
	spec_conv_erogatore CLOB,
	spec_conv_fruitore CLOB,
	filtro_duplicati VARCHAR2(255),
	conferma_ricezione VARCHAR2(255),
	identificativo_collaborazione VARCHAR2(255),
	id_riferimento_richiesta VARCHAR2(255),
	consegna_in_ordine VARCHAR2(255),
	scadenza VARCHAR2(255),
	superuser VARCHAR2(255),
	-- id del soggetto referente
	id_referente NUMBER,
	versione NUMBER,
	-- 1/0 (vero/falso) indica se questo accordo di servizio e' utilizzabile in mancanza di azioni associate
	utilizzo_senza_azione NUMBER,
	-- 1/0 (true/false) indica se il soggetto e' privato/pubblico
	privato NUMBER,
	ora_registrazione TIMESTAMP,
	stato VARCHAR2(255) NOT NULL,
	canale VARCHAR2(255),
	utente_richiedente VARCHAR2(255),
	data_creazione TIMESTAMP,
	utente_ultima_modifica VARCHAR2(255),
	data_ultima_modifica TIMESTAMP,
	-- fk/pk columns
	id NUMBER NOT NULL,
	-- check constraints
	CONSTRAINT chk_accordi_1 CHECK (service_binding IN ('soap','rest')),
	CONSTRAINT chk_accordi_2 CHECK (stato IN ('finale','bozza','operativo')),
	-- unique constraints
	CONSTRAINT unique_accordi_1 UNIQUE (nome,id_referente,versione),
	-- fk/pk keys constraints
	CONSTRAINT pk_accordi PRIMARY KEY (id)
);

-- index
CREATE INDEX index_accordi_1 ON accordi (canale);

ALTER TABLE accordi MODIFY id_referente DEFAULT 0;
ALTER TABLE accordi MODIFY versione DEFAULT 1;
ALTER TABLE accordi MODIFY utilizzo_senza_azione DEFAULT 1;
ALTER TABLE accordi MODIFY privato DEFAULT 0;
ALTER TABLE accordi MODIFY ora_registrazione DEFAULT CURRENT_TIMESTAMP;
ALTER TABLE accordi MODIFY stato DEFAULT 'finale';

CREATE TRIGGER trg_accordi
BEFORE
insert on accordi
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_accordi.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/



CREATE SEQUENCE seq_accordi_azioni MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE accordi_azioni
(
	id_accordo NUMBER NOT NULL,
	nome VARCHAR2(255) NOT NULL,
	profilo_collaborazione VARCHAR2(255),
	filtro_duplicati VARCHAR2(255),
	conferma_ricezione VARCHAR2(255),
	identificativo_collaborazione VARCHAR2(255),
	id_riferimento_richiesta VARCHAR2(255),
	consegna_in_ordine VARCHAR2(255),
	scadenza VARCHAR2(255),
	correlata VARCHAR2(255),
	-- ridefinito/default
	profilo_azione VARCHAR2(255),
	-- fk/pk columns
	id NUMBER NOT NULL,
	-- unique constraints
	CONSTRAINT unique_accordi_azioni_1 UNIQUE (id_accordo,nome),
	-- fk/pk keys constraints
	CONSTRAINT fk_accordi_azioni_1 FOREIGN KEY (id_accordo) REFERENCES accordi(id),
	CONSTRAINT pk_accordi_azioni PRIMARY KEY (id)
);

CREATE TRIGGER trg_accordi_azioni
BEFORE
insert on accordi_azioni
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_accordi_azioni.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/



CREATE SEQUENCE seq_port_type MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE port_type
(
	id_accordo NUMBER NOT NULL,
	nome VARCHAR2(255) NOT NULL,
	descrizione VARCHAR2(255),
	profilo_collaborazione VARCHAR2(255),
	filtro_duplicati VARCHAR2(255),
	conferma_ricezione VARCHAR2(255),
	identificativo_collaborazione VARCHAR2(255),
	id_riferimento_richiesta VARCHAR2(255),
	consegna_in_ordine VARCHAR2(255),
	scadenza VARCHAR2(255),
	-- ridefinito/default
	profilo_pt VARCHAR2(255),
	-- document/RPC
	soap_style VARCHAR2(255),
	message_type VARCHAR2(255),
	-- fk/pk columns
	id NUMBER NOT NULL,
	-- unique constraints
	CONSTRAINT unique_port_type_1 UNIQUE (id_accordo,nome),
	-- fk/pk keys constraints
	CONSTRAINT fk_port_type_1 FOREIGN KEY (id_accordo) REFERENCES accordi(id),
	CONSTRAINT pk_port_type PRIMARY KEY (id)
);

CREATE TRIGGER trg_port_type
BEFORE
insert on port_type
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_port_type.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/



CREATE SEQUENCE seq_port_type_azioni MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE port_type_azioni
(
	id_port_type NUMBER NOT NULL,
	nome VARCHAR2(255) NOT NULL,
	profilo_collaborazione VARCHAR2(255),
	filtro_duplicati VARCHAR2(255),
	conferma_ricezione VARCHAR2(255),
	identificativo_collaborazione VARCHAR2(255),
	id_riferimento_richiesta VARCHAR2(255),
	consegna_in_ordine VARCHAR2(255),
	scadenza VARCHAR2(255),
	correlata_servizio VARCHAR2(255),
	correlata VARCHAR2(255),
	-- ridefinito/default
	profilo_pt_azione VARCHAR2(255),
	-- document/rpc
	soap_style VARCHAR2(255),
	soap_action VARCHAR2(255),
	-- literal/encoded
	soap_use_msg_input VARCHAR2(255),
	-- namespace utilizzato per RPC
	soap_namespace_msg_input VARCHAR2(255),
	-- literal/encoded
	soap_use_msg_output VARCHAR2(255),
	-- namespace utilizzato per RPC
	soap_namespace_msg_output VARCHAR2(255),
	-- fk/pk columns
	id NUMBER NOT NULL,
	-- unique constraints
	CONSTRAINT unique_port_type_azioni_1 UNIQUE (id_port_type,nome),
	-- fk/pk keys constraints
	CONSTRAINT fk_port_type_azioni_1 FOREIGN KEY (id_port_type) REFERENCES port_type(id),
	CONSTRAINT pk_port_type_azioni PRIMARY KEY (id)
);

CREATE TRIGGER trg_port_type_azioni
BEFORE
insert on port_type_azioni
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_port_type_azioni.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/



CREATE SEQUENCE seq_operation_messages MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE operation_messages
(
	id_port_type_azione NUMBER NOT NULL,
	-- true(1)/false(0), true indica un input-message, false un output-message
	input_message NUMBER,
	name VARCHAR2(255) NOT NULL,
	element_name VARCHAR2(255),
	element_namespace VARCHAR2(255),
	type_name VARCHAR2(255),
	type_namespace VARCHAR2(255),
	-- fk/pk columns
	id NUMBER NOT NULL,
	-- fk/pk keys constraints
	CONSTRAINT fk_operation_messages_1 FOREIGN KEY (id_port_type_azione) REFERENCES port_type_azioni(id),
	CONSTRAINT pk_operation_messages PRIMARY KEY (id)
);

-- index
CREATE INDEX INDEX_OP_MESSAGES ON operation_messages (id_port_type_azione,input_message);

ALTER TABLE operation_messages MODIFY input_message DEFAULT 1;

CREATE TRIGGER trg_operation_messages
BEFORE
insert on operation_messages
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_operation_messages.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/



CREATE SEQUENCE seq_api_resources MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE api_resources
(
	id_accordo NUMBER NOT NULL,
	nome VARCHAR2(255) NOT NULL,
	descrizione VARCHAR2(255),
	http_method VARCHAR2(255) NOT NULL,
	path VARCHAR2(255) NOT NULL,
	message_type VARCHAR2(255),
	message_type_request VARCHAR2(255),
	message_type_response VARCHAR2(255),
	filtro_duplicati VARCHAR2(255),
	conferma_ricezione VARCHAR2(255),
	identificativo_collaborazione VARCHAR2(255),
	id_riferimento_richiesta VARCHAR2(255),
	consegna_in_ordine VARCHAR2(255),
	scadenza VARCHAR2(255),
	-- ridefinito/default
	profilo_azione VARCHAR2(255),
	-- fk/pk columns
	id NUMBER NOT NULL,
	-- unique constraints
	CONSTRAINT unique_api_resources_1 UNIQUE (id_accordo,nome),
	CONSTRAINT unique_api_resources_2 UNIQUE (id_accordo,http_method,path),
	-- fk/pk keys constraints
	CONSTRAINT fk_api_resources_1 FOREIGN KEY (id_accordo) REFERENCES accordi(id),
	CONSTRAINT pk_api_resources PRIMARY KEY (id)
);

CREATE TRIGGER trg_api_resources
BEFORE
insert on api_resources
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_api_resources.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/



CREATE SEQUENCE seq_api_resources_response MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE api_resources_response
(
	id_resource NUMBER NOT NULL,
	descrizione VARCHAR2(255),
	status NUMBER NOT NULL,
	-- fk/pk columns
	id NUMBER NOT NULL,
	-- unique constraints
	CONSTRAINT uniq_api_resp_1 UNIQUE (id_resource,status),
	-- fk/pk keys constraints
	CONSTRAINT fk_api_resources_response_1 FOREIGN KEY (id_resource) REFERENCES api_resources(id),
	CONSTRAINT pk_api_resources_response PRIMARY KEY (id)
);

CREATE TRIGGER trg_api_resources_response
BEFORE
insert on api_resources_response
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_api_resources_response.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/



CREATE SEQUENCE seq_api_resources_media MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE api_resources_media
(
	id_resource_media NUMBER,
	id_resource_response_media NUMBER,
	media_type VARCHAR2(255) NOT NULL,
	message_type VARCHAR2(255),
	nome VARCHAR2(255),
	descrizione VARCHAR2(255),
	tipo VARCHAR2(255),
	xml_tipo VARCHAR2(255),
	xml_name VARCHAR2(255),
	xml_namespace VARCHAR2(255),
	json_type VARCHAR2(255),
	-- fk/pk columns
	id NUMBER NOT NULL,
	-- fk/pk keys constraints
	CONSTRAINT fk_api_resources_media_1 FOREIGN KEY (id_resource_response_media) REFERENCES api_resources_response(id),
	CONSTRAINT fk_api_resources_media_2 FOREIGN KEY (id_resource_media) REFERENCES api_resources(id),
	CONSTRAINT pk_api_resources_media PRIMARY KEY (id)
);

CREATE TRIGGER trg_api_resources_media
BEFORE
insert on api_resources_media
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_api_resources_media.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/



CREATE SEQUENCE seq_api_resources_parameter MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE api_resources_parameter
(
	id_resource_parameter NUMBER,
	id_resource_response_par NUMBER,
	nome VARCHAR2(255) NOT NULL,
	descrizione VARCHAR2(255),
	tipo_parametro VARCHAR2(255) NOT NULL,
	required NUMBER NOT NULL,
	tipo VARCHAR2(255) NOT NULL,
	restrizioni CLOB,
	-- fk/pk columns
	id NUMBER NOT NULL,
	-- fk/pk keys constraints
	CONSTRAINT fk_api_resources_parameter_1 FOREIGN KEY (id_resource_response_par) REFERENCES api_resources_response(id),
	CONSTRAINT fk_api_resources_parameter_2 FOREIGN KEY (id_resource_parameter) REFERENCES api_resources(id),
	CONSTRAINT pk_api_resources_parameter PRIMARY KEY (id)
);


ALTER TABLE api_resources_parameter MODIFY required DEFAULT 0;

CREATE TRIGGER trg_api_resources_parameter
BEFORE
insert on api_resources_parameter
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_api_resources_parameter.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/



CREATE SEQUENCE seq_accordi_gruppi MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE accordi_gruppi
(
	id_accordo NUMBER NOT NULL,
	id_gruppo NUMBER NOT NULL,
	-- fk/pk columns
	id NUMBER NOT NULL,
	-- unique constraints
	CONSTRAINT unique_acc_gruppi_1 UNIQUE (id_accordo,id_gruppo),
	-- fk/pk keys constraints
	CONSTRAINT fk_accordi_gruppi_1 FOREIGN KEY (id_gruppo) REFERENCES gruppi(id),
	CONSTRAINT fk_accordi_gruppi_2 FOREIGN KEY (id_accordo) REFERENCES accordi(id),
	CONSTRAINT pk_accordi_gruppi PRIMARY KEY (id)
);

CREATE TRIGGER trg_accordi_gruppi
BEFORE
insert on accordi_gruppi
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_accordi_gruppi.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/



-- **** Accordi di Cooperazione ****

CREATE SEQUENCE seq_accordi_cooperazione MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE accordi_cooperazione
(
	nome VARCHAR2(255) NOT NULL,
	descrizione VARCHAR2(255),
	-- id del soggetto referente
	id_referente NUMBER,
	versione NUMBER,
	stato VARCHAR2(255) NOT NULL,
	-- 1/0 (true/false) indica se il soggetto e' privato/pubblico
	privato NUMBER,
	superuser VARCHAR2(255),
	ora_registrazione TIMESTAMP,
	-- fk/pk columns
	id NUMBER NOT NULL,
	-- check constraints
	CONSTRAINT chk_accordi_cooperazione_1 CHECK (stato IN ('finale','bozza','operativo')),
	-- unique constraints
	CONSTRAINT unique_accordi_cooperazione_1 UNIQUE (nome,id_referente,versione),
	-- fk/pk keys constraints
	CONSTRAINT pk_accordi_cooperazione PRIMARY KEY (id)
);


ALTER TABLE accordi_cooperazione MODIFY id_referente DEFAULT 0;
ALTER TABLE accordi_cooperazione MODIFY versione DEFAULT 1;
ALTER TABLE accordi_cooperazione MODIFY stato DEFAULT 'finale';
ALTER TABLE accordi_cooperazione MODIFY privato DEFAULT 0;
ALTER TABLE accordi_cooperazione MODIFY ora_registrazione DEFAULT CURRENT_TIMESTAMP;

CREATE TRIGGER trg_accordi_cooperazione
BEFORE
insert on accordi_cooperazione
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_accordi_cooperazione.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/



CREATE SEQUENCE seq_accordi_coop_partecipanti MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE accordi_coop_partecipanti
(
	id_accordo_cooperazione NUMBER NOT NULL,
	id_soggetto NUMBER NOT NULL,
	-- fk/pk columns
	id NUMBER NOT NULL,
	-- unique constraints
	CONSTRAINT unique_acc_coop_part_1 UNIQUE (id_accordo_cooperazione,id_soggetto),
	-- fk/pk keys constraints
	CONSTRAINT fk_accordi_coop_partecipanti_1 FOREIGN KEY (id_soggetto) REFERENCES soggetti(id),
	CONSTRAINT fk_accordi_coop_partecipanti_2 FOREIGN KEY (id_accordo_cooperazione) REFERENCES accordi_cooperazione(id),
	CONSTRAINT pk_accordi_coop_partecipanti PRIMARY KEY (id)
);

-- index
CREATE INDEX INDEX_AC_COOP_PAR ON accordi_coop_partecipanti (id_accordo_cooperazione);
CREATE INDEX INDEX_AC_SOG ON accordi_coop_partecipanti (id_soggetto);
CREATE TRIGGER trg_accordi_coop_partecipanti
BEFORE
insert on accordi_coop_partecipanti
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_accordi_coop_partecipanti.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/



-- **** Accordi di Servizio Parte Specifica ****

CREATE SEQUENCE seq_servizi MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE servizi
(
	nome_servizio VARCHAR2(255) NOT NULL,
	tipo_servizio VARCHAR2(255) NOT NULL,
	versione_servizio NUMBER,
	id_soggetto NUMBER NOT NULL,
	id_accordo NUMBER NOT NULL,
	servizio_correlato VARCHAR2(255),
	id_connettore NUMBER NOT NULL,
	wsdl_implementativo_erogatore CLOB,
	wsdl_implementativo_fruitore CLOB,
	superuser VARCHAR2(255),
	-- 1/0 (true/false) indica se il soggetto e' privato/pubblico
	privato NUMBER,
	ora_registrazione TIMESTAMP,
	port_type VARCHAR2(255),
	profilo VARCHAR2(255),
	descrizione VARCHAR2(4000),
	stato VARCHAR2(255) NOT NULL,
	message_type VARCHAR2(255),
	utente_richiedente VARCHAR2(255),
	data_creazione TIMESTAMP,
	utente_ultima_modifica VARCHAR2(255),
	data_ultima_modifica TIMESTAMP,
	-- fk/pk columns
	id NUMBER NOT NULL,
	-- check constraints
	CONSTRAINT chk_servizi_1 CHECK (stato IN ('finale','bozza','operativo')),
	-- unique constraints
	CONSTRAINT unique_servizi_1 UNIQUE (id_soggetto,tipo_servizio,nome_servizio,versione_servizio),
	-- fk/pk keys constraints
	CONSTRAINT fk_servizi_1 FOREIGN KEY (id_connettore) REFERENCES connettori(id),
	CONSTRAINT fk_servizi_2 FOREIGN KEY (id_soggetto) REFERENCES soggetti(id),
	CONSTRAINT fk_servizi_3 FOREIGN KEY (id_accordo) REFERENCES accordi(id),
	CONSTRAINT pk_servizi PRIMARY KEY (id)
);

-- index
CREATE INDEX INDEX_SERV_ACC ON servizi (id_accordo);
CREATE INDEX INDEX_SERV_SOG ON servizi (id_soggetto);

ALTER TABLE servizi MODIFY versione_servizio DEFAULT 1;
ALTER TABLE servizi MODIFY privato DEFAULT 0;
ALTER TABLE servizi MODIFY ora_registrazione DEFAULT CURRENT_TIMESTAMP;
ALTER TABLE servizi MODIFY stato DEFAULT 'finale';

CREATE TRIGGER trg_servizi
BEFORE
insert on servizi
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_servizi.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/



CREATE SEQUENCE seq_servizi_azioni MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE servizi_azioni
(
	id_servizio NUMBER NOT NULL,
	id_connettore NUMBER NOT NULL,
	-- fk/pk columns
	id NUMBER NOT NULL,
	-- fk/pk keys constraints
	CONSTRAINT fk_servizi_azioni_1 FOREIGN KEY (id_connettore) REFERENCES connettori(id),
	CONSTRAINT fk_servizi_azioni_2 FOREIGN KEY (id_servizio) REFERENCES servizi(id),
	CONSTRAINT pk_servizi_azioni PRIMARY KEY (id)
);

-- index
CREATE INDEX index_servizi_azioni_1 ON servizi_azioni (id_servizio);
CREATE TRIGGER trg_servizi_azioni
BEFORE
insert on servizi_azioni
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_servizi_azioni.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/



CREATE SEQUENCE seq_servizi_azione MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE servizi_azione
(
	nome_azione VARCHAR2(255) NOT NULL,
	id_servizio_azioni NUMBER NOT NULL,
	-- fk/pk columns
	id NUMBER NOT NULL,
	-- unique constraints
	CONSTRAINT unique_servizi_azione_1 UNIQUE (nome_azione,id_servizio_azioni),
	-- fk/pk keys constraints
	CONSTRAINT fk_servizi_azione_1 FOREIGN KEY (id_servizio_azioni) REFERENCES servizi_azioni(id),
	CONSTRAINT pk_servizi_azione PRIMARY KEY (id)
);

CREATE TRIGGER trg_servizi_azione
BEFORE
insert on servizi_azione
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_servizi_azione.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/



CREATE SEQUENCE seq_servizi_fruitori MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE servizi_fruitori
(
	id_servizio NUMBER NOT NULL,
	id_soggetto NUMBER NOT NULL,
	id_connettore NUMBER NOT NULL,
	wsdl_implementativo_erogatore CLOB,
	wsdl_implementativo_fruitore CLOB,
	ora_registrazione TIMESTAMP,
	stato VARCHAR2(255) NOT NULL,
	descrizione VARCHAR2(4000),
	utente_richiedente VARCHAR2(255),
	data_creazione TIMESTAMP,
	utente_ultima_modifica VARCHAR2(255),
	data_ultima_modifica TIMESTAMP,
	-- fk/pk columns
	id NUMBER NOT NULL,
	-- check constraints
	CONSTRAINT chk_servizi_fruitori_1 CHECK (stato IN ('finale','bozza','operativo')),
	-- unique constraints
	CONSTRAINT unique_servizi_fruitori_1 UNIQUE (id_servizio,id_soggetto),
	-- fk/pk keys constraints
	CONSTRAINT fk_servizi_fruitori_1 FOREIGN KEY (id_connettore) REFERENCES connettori(id),
	CONSTRAINT fk_servizi_fruitori_2 FOREIGN KEY (id_soggetto) REFERENCES soggetti(id),
	CONSTRAINT fk_servizi_fruitori_3 FOREIGN KEY (id_servizio) REFERENCES servizi(id),
	CONSTRAINT pk_servizi_fruitori PRIMARY KEY (id)
);

-- index
CREATE INDEX INDEX_SERV_FRU_SOG ON servizi_fruitori (id_soggetto);

ALTER TABLE servizi_fruitori MODIFY ora_registrazione DEFAULT CURRENT_TIMESTAMP;
ALTER TABLE servizi_fruitori MODIFY stato DEFAULT 'finale';

CREATE TRIGGER trg_servizi_fruitori
BEFORE
insert on servizi_fruitori
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_servizi_fruitori.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/



CREATE SEQUENCE seq_servizi_fruitori_azioni MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE servizi_fruitori_azioni
(
	id_fruizione NUMBER NOT NULL,
	id_connettore NUMBER NOT NULL,
	-- fk/pk columns
	id NUMBER NOT NULL,
	-- fk/pk keys constraints
	CONSTRAINT fk_servizi_fruitori_azioni_1 FOREIGN KEY (id_connettore) REFERENCES connettori(id),
	CONSTRAINT fk_servizi_fruitori_azioni_2 FOREIGN KEY (id_fruizione) REFERENCES servizi_fruitori(id),
	CONSTRAINT pk_servizi_fruitori_azioni PRIMARY KEY (id)
);

-- index
CREATE INDEX index_serv_fru_azioni_1 ON servizi_fruitori_azioni (id_fruizione);
CREATE TRIGGER trg_servizi_fruitori_azioni
BEFORE
insert on servizi_fruitori_azioni
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_servizi_fruitori_azioni.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/



CREATE SEQUENCE seq_servizi_fruitori_azione MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE servizi_fruitori_azione
(
	nome_azione VARCHAR2(255) NOT NULL,
	id_fruizione_azioni NUMBER NOT NULL,
	-- fk/pk columns
	id NUMBER NOT NULL,
	-- unique constraints
	CONSTRAINT uniq_serv_fru_azione_1 UNIQUE (nome_azione,id_fruizione_azioni),
	-- fk/pk keys constraints
	CONSTRAINT fk_servizi_fruitori_azione_1 FOREIGN KEY (id_fruizione_azioni) REFERENCES servizi_fruitori_azioni(id),
	CONSTRAINT pk_servizi_fruitori_azione PRIMARY KEY (id)
);

CREATE TRIGGER trg_servizi_fruitori_azione
BEFORE
insert on servizi_fruitori_azione
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_servizi_fruitori_azione.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/



-- **** Accordi di Servizio Composti ****

CREATE SEQUENCE seq_acc_serv_composti MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE acc_serv_composti
(
	id_accordo NUMBER NOT NULL,
	id_accordo_cooperazione NUMBER NOT NULL,
	-- fk/pk columns
	id NUMBER NOT NULL,
	-- unique constraints
	CONSTRAINT unique_acc_serv_composti_1 UNIQUE (id_accordo),
	-- fk/pk keys constraints
	CONSTRAINT fk_acc_serv_composti_1 FOREIGN KEY (id_accordo_cooperazione) REFERENCES accordi_cooperazione(id),
	CONSTRAINT fk_acc_serv_composti_2 FOREIGN KEY (id_accordo) REFERENCES accordi(id),
	CONSTRAINT pk_acc_serv_composti PRIMARY KEY (id)
);

-- index
CREATE INDEX INDEX_AC_SC ON acc_serv_composti (id_accordo_cooperazione);
CREATE TRIGGER trg_acc_serv_composti
BEFORE
insert on acc_serv_composti
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_acc_serv_composti.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/



CREATE SEQUENCE seq_acc_serv_componenti MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE acc_serv_componenti
(
	id_servizio_composto NUMBER NOT NULL,
	id_servizio_componente NUMBER NOT NULL,
	azione VARCHAR2(255),
	-- fk/pk columns
	id NUMBER NOT NULL,
	-- fk/pk keys constraints
	CONSTRAINT fk_acc_serv_componenti_1 FOREIGN KEY (id_servizio_composto) REFERENCES acc_serv_composti(id),
	CONSTRAINT fk_acc_serv_componenti_2 FOREIGN KEY (id_servizio_componente) REFERENCES servizi(id),
	CONSTRAINT pk_acc_serv_componenti PRIMARY KEY (id)
);

-- index
CREATE INDEX INDEX_AC_SC_SC ON acc_serv_componenti (id_servizio_composto);
CREATE INDEX INDEX_AC_SC_SERV ON acc_serv_componenti (id_servizio_componente);
CREATE TRIGGER trg_acc_serv_componenti
BEFORE
insert on acc_serv_componenti
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_acc_serv_componenti.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/




-- govway
		  		
-- PLUGINS

CREATE SEQUENCE seq_plugins MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE plugins
(
	tipo_plugin VARCHAR2(255) NOT NULL,
	class_name VARCHAR2(255) NOT NULL,
	tipo VARCHAR2(255) NOT NULL,
	descrizione VARCHAR2(255),
	label VARCHAR2(255) NOT NULL,
	stato NUMBER,
	-- fk/pk columns
	id NUMBER NOT NULL,
	-- unique constraints
	CONSTRAINT unique_plugins_1 UNIQUE (tipo_plugin,class_name),
	CONSTRAINT unique_plugins_2 UNIQUE (tipo_plugin,tipo),
	CONSTRAINT unique_plugins_3 UNIQUE (tipo_plugin,label),
	-- fk/pk keys constraints
	CONSTRAINT pk_plugins PRIMARY KEY (id)
);


ALTER TABLE plugins MODIFY stato DEFAULT 1;

CREATE TRIGGER trg_plugins
BEFORE
insert on plugins
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_plugins.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/



CREATE SEQUENCE seq_plugins_servizi_comp MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE plugins_servizi_comp
(
	-- Accordo di Servizio
	uri_accordo VARCHAR2(255),
	-- Nome del port-type
	servizio VARCHAR2(255),
	-- fk/pk columns
	id NUMBER NOT NULL,
	id_plugin NUMBER NOT NULL,
	-- fk/pk keys constraints
	CONSTRAINT fk_plugins_servizi_comp_1 FOREIGN KEY (id_plugin) REFERENCES plugins(id) ON DELETE CASCADE,
	CONSTRAINT pk_plugins_servizi_comp PRIMARY KEY (id)
);

-- index
CREATE INDEX idx_plug_ser_com_1 ON plugins_servizi_comp (id_plugin);
CREATE TRIGGER trg_plugins_servizi_comp
BEFORE
insert on plugins_servizi_comp
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_plugins_servizi_comp.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/



CREATE SEQUENCE seq_plugins_azioni_comp MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE plugins_azioni_comp
(
	azione VARCHAR2(255) NOT NULL,
	-- fk/pk columns
	id NUMBER NOT NULL,
	id_plugin_servizio_comp NUMBER NOT NULL,
	-- unique constraints
	CONSTRAINT unique_plugins_azioni_comp_1 UNIQUE (id_plugin_servizio_comp,azione),
	-- fk/pk keys constraints
	CONSTRAINT fk_plugins_azioni_comp_1 FOREIGN KEY (id_plugin_servizio_comp) REFERENCES plugins_servizi_comp(id) ON DELETE CASCADE,
	CONSTRAINT pk_plugins_azioni_comp PRIMARY KEY (id)
);

CREATE TRIGGER trg_plugins_azioni_comp
BEFORE
insert on plugins_azioni_comp
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_plugins_azioni_comp.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/



CREATE SEQUENCE seq_plugins_props_comp MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE plugins_props_comp
(
	nome VARCHAR2(255) NOT NULL,
	valore VARCHAR2(255) NOT NULL,
	-- fk/pk columns
	id NUMBER NOT NULL,
	id_plugin NUMBER NOT NULL,
	-- fk/pk keys constraints
	CONSTRAINT fk_plugins_props_comp_1 FOREIGN KEY (id_plugin) REFERENCES plugins(id) ON DELETE CASCADE,
	CONSTRAINT pk_plugins_props_comp PRIMARY KEY (id)
);

-- index
CREATE INDEX idx_plug_prop_com_1 ON plugins_props_comp (id_plugin);
CREATE TRIGGER trg_plugins_props_comp
BEFORE
insert on plugins_props_comp
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_plugins_props_comp.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/



-- CONFIGURAZIONE DEI SERVIZI PER PERSONALIZZAZIONI

CREATE SEQUENCE seq_plugins_conf_servizi MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE plugins_conf_servizi
(
	-- Accordo di Servizio
	accordo VARCHAR2(255) NOT NULL,
	tipo_soggetto_referente VARCHAR2(255) NOT NULL,
	nome_soggetto_referente VARCHAR2(255) NOT NULL,
	versione NUMBER NOT NULL,
	-- Nome del port-type
	servizio VARCHAR2(255),
	-- fk/pk columns
	id NUMBER NOT NULL,
	-- fk/pk keys constraints
	CONSTRAINT pk_plugins_conf_servizi PRIMARY KEY (id)
);

-- index
CREATE INDEX idx_conf_servizi_1 ON plugins_conf_servizi (accordo,tipo_soggetto_referente,nome_soggetto_referente,versione,servizio);

ALTER TABLE plugins_conf_servizi MODIFY versione DEFAULT 1;

CREATE TRIGGER trg_plugins_conf_servizi
BEFORE
insert on plugins_conf_servizi
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_plugins_conf_servizi.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/



CREATE SEQUENCE seq_plugins_conf_azioni MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE plugins_conf_azioni
(
	azione VARCHAR2(255) NOT NULL,
	-- fk/pk columns
	id NUMBER NOT NULL,
	id_config_servizio NUMBER NOT NULL,
	-- unique constraints
	CONSTRAINT unique_plugins_conf_azioni_1 UNIQUE (id_config_servizio,azione),
	-- fk/pk keys constraints
	CONSTRAINT fk_plugins_conf_azioni_1 FOREIGN KEY (id_config_servizio) REFERENCES plugins_conf_servizi(id) ON DELETE CASCADE,
	CONSTRAINT pk_plugins_conf_azioni PRIMARY KEY (id)
);

CREATE TRIGGER trg_plugins_conf_azioni
BEFORE
insert on plugins_conf_azioni
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_plugins_conf_azioni.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/



-- FILTRO PER PERSONALIZZAZIONI

CREATE SEQUENCE seq_plugins_conf_filtri MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE plugins_conf_filtri
(
	-- Dati Identificativi Regola
	nome VARCHAR2(255) NOT NULL,
	descrizione VARCHAR2(255),
	-- Regole di Filtraggio
	tipo_mittente VARCHAR2(255),
	nome_mittente VARCHAR2(255),
	idporta_mittente VARCHAR2(255),
	tipo_destinatario VARCHAR2(255),
	nome_destinatario VARCHAR2(255),
	idporta_destinatario VARCHAR2(255),
	tipo_servizio VARCHAR2(255),
	nome_servizio VARCHAR2(255),
	versione_servizio NUMBER,
	azione VARCHAR2(255),
	-- fk/pk columns
	id NUMBER NOT NULL,
	-- unique constraints
	CONSTRAINT uniq_conf_filtri_1 UNIQUE (nome),
	-- fk/pk keys constraints
	CONSTRAINT pk_plugins_conf_filtri PRIMARY KEY (id)
);


ALTER TABLE plugins_conf_filtri MODIFY versione_servizio DEFAULT 1;

CREATE TRIGGER trg_plugins_conf_filtri
BEFORE
insert on plugins_conf_filtri
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_plugins_conf_filtri.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/




-- govway
		  		
-- PERSONALIZZAZIONI/PLUGINS DELLE TRANSAZIONI

CREATE SEQUENCE seq_config_transazioni MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE config_transazioni
(
	enabled NUMBER NOT NULL,
	-- fk/pk columns
	id NUMBER NOT NULL,
	id_conf_servizio_azione NUMBER NOT NULL,
	-- unique constraints
	CONSTRAINT uniq_trans_pers_1 UNIQUE (id_conf_servizio_azione),
	-- fk/pk keys constraints
	CONSTRAINT fk_config_transazioni_1 FOREIGN KEY (id_conf_servizio_azione) REFERENCES plugins_conf_azioni(id),
	CONSTRAINT pk_config_transazioni PRIMARY KEY (id)
);

CREATE TRIGGER trg_config_transazioni
BEFORE
insert on config_transazioni
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_config_transazioni.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/



CREATE SEQUENCE seq_config_tran_plugins MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE config_tran_plugins
(
	id_conf_trans_plugin VARCHAR2(255) NOT NULL,
	enabled NUMBER NOT NULL,
	-- fk/pk columns
	id NUMBER NOT NULL,
	id_configurazione_transazione NUMBER NOT NULL,
	id_plugin NUMBER NOT NULL,
	-- unique constraints
	CONSTRAINT uniq_conf_trans_plug_1 UNIQUE (id_configurazione_transazione,id_conf_trans_plugin),
	CONSTRAINT uniq_conf_trans_plug_2 UNIQUE (id_configurazione_transazione,id_plugin),
	-- fk/pk keys constraints
	CONSTRAINT fk_config_tran_plugins_1 FOREIGN KEY (id_configurazione_transazione) REFERENCES config_transazioni(id),
	CONSTRAINT fk_config_tran_plugins_2 FOREIGN KEY (id_plugin) REFERENCES plugins(id),
	CONSTRAINT pk_config_tran_plugins PRIMARY KEY (id)
);

CREATE TRIGGER trg_config_tran_plugins
BEFORE
insert on config_tran_plugins
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_config_tran_plugins.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/



CREATE SEQUENCE seq_configurazione_stati MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE configurazione_stati
(
	enabled NUMBER NOT NULL,
	nome VARCHAR2(255) NOT NULL,
	tipo_controllo VARCHAR2(255) NOT NULL,
	tipo_messaggio VARCHAR2(255) NOT NULL,
	valore VARCHAR2(255),
	xpath VARCHAR2(255) NOT NULL,
	-- fk/pk columns
	id NUMBER NOT NULL,
	id_configurazione_transazione NUMBER NOT NULL,
	-- check constraints
	CONSTRAINT chk_configurazione_stati_1 CHECK (tipo_controllo IN ('Match','Equals','Exist')),
	CONSTRAINT chk_configurazione_stati_2 CHECK (tipo_messaggio IN ('Richiesta','Risposta')),
	-- unique constraints
	CONSTRAINT uniq_trans_pers_stati_1 UNIQUE (id_configurazione_transazione,nome),
	-- fk/pk keys constraints
	CONSTRAINT fk_configurazione_stati_1 FOREIGN KEY (id_configurazione_transazione) REFERENCES config_transazioni(id),
	CONSTRAINT pk_configurazione_stati PRIMARY KEY (id)
);

CREATE TRIGGER trg_configurazione_stati
BEFORE
insert on configurazione_stati
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_configurazione_stati.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/



CREATE SEQUENCE seq_conf_risorse_contenuti MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE conf_risorse_contenuti
(
	abilita_anonimizzazione NUMBER NOT NULL,
	abilita_compressione NUMBER NOT NULL,
	tipo_compressione VARCHAR2(255),
	carattere_maschera CHAR(1),
	num_char_maschera NUMBER,
	posizionamento_maschera VARCHAR2(255),
	tipo_mascheramento VARCHAR2(255),
	enabled NUMBER NOT NULL,
	nome VARCHAR2(255) NOT NULL,
	tipo_messaggio VARCHAR2(255) NOT NULL,
	xpath VARCHAR2(255) NOT NULL,
	stat_enabled NUMBER NOT NULL,
	-- fk/pk columns
	id NUMBER NOT NULL,
	id_conf_transazione NUMBER NOT NULL,
	id_configurazione_stato NUMBER,
	-- check constraints
	CONSTRAINT chk_conf_risorse_contenuti_1 CHECK (tipo_compressione IN ('Deflater','Gzip','Zip')),
	CONSTRAINT chk_conf_risorse_contenuti_2 CHECK (posizionamento_maschera IN ('Primi','Ultimi')),
	CONSTRAINT chk_conf_risorse_contenuti_3 CHECK (tipo_mascheramento IN ('Fisico','Logico')),
	CONSTRAINT chk_conf_risorse_contenuti_4 CHECK (tipo_messaggio IN ('Richiesta','Risposta')),
	-- unique constraints
	CONSTRAINT uniq_trans_pers_risorse_1 UNIQUE (id_conf_transazione,nome),
	-- fk/pk keys constraints
	CONSTRAINT fk_conf_risorse_contenuti_1 FOREIGN KEY (id_conf_transazione) REFERENCES config_transazioni(id),
	CONSTRAINT fk_conf_risorse_contenuti_2 FOREIGN KEY (id_configurazione_stato) REFERENCES configurazione_stati(id),
	CONSTRAINT pk_conf_risorse_contenuti PRIMARY KEY (id)
);


ALTER TABLE conf_risorse_contenuti MODIFY abilita_anonimizzazione DEFAULT 0;
ALTER TABLE conf_risorse_contenuti MODIFY abilita_compressione DEFAULT 0;
ALTER TABLE conf_risorse_contenuti MODIFY stat_enabled DEFAULT 0;

CREATE TRIGGER trg_conf_risorse_contenuti
BEFORE
insert on conf_risorse_contenuti
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_conf_risorse_contenuti.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/




-- govway
		  		
-- PLUGINS PER LE RICERCHE PERSONALIZZATE

CREATE SEQUENCE seq_ricerche_personalizzate MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE ricerche_personalizzate
(
	id VARCHAR2(255) NOT NULL,
	enabled NUMBER NOT NULL,
	label VARCHAR2(255) NOT NULL,
	-- fk/pk columns
	pid NUMBER NOT NULL,
	id_configurazione NUMBER NOT NULL,
	id_plugin NUMBER NOT NULL,
	-- unique constraints
	CONSTRAINT uniq_ricerche_pers_1 UNIQUE (id_configurazione,id),
	CONSTRAINT uniq_ricerche_pers_2 UNIQUE (id_configurazione,id_plugin),
	-- fk/pk keys constraints
	CONSTRAINT fk_ricerche_personalizzate_1 FOREIGN KEY (id_configurazione) REFERENCES plugins_conf_azioni(id) ON DELETE CASCADE,
	CONSTRAINT fk_ricerche_personalizzate_2 FOREIGN KEY (id_plugin) REFERENCES plugins(id) ON DELETE CASCADE,
	CONSTRAINT pk_ricerche_personalizzate PRIMARY KEY (pid)
);

CREATE TRIGGER trg_ricerche_personalizzate
BEFORE
insert on ricerche_personalizzate
for each row
begin
   IF (:new.pid IS NULL) THEN
      SELECT seq_ricerche_personalizzate.nextval INTO :new.pid
                FROM DUAL;
   END IF;
end;
/




-- govway
		  		
-- PLUGINS PER LE STATISTICHE PERSONALIZZATE

CREATE SEQUENCE seq_stat_personalizzate MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE stat_personalizzate
(
	id VARCHAR2(255) NOT NULL,
	enabled NUMBER NOT NULL,
	label VARCHAR2(255) NOT NULL,
	-- fk/pk columns
	pid NUMBER NOT NULL,
	id_configurazione NUMBER NOT NULL,
	id_plugin NUMBER NOT NULL,
	-- unique constraints
	CONSTRAINT uniq_stat_pers_1 UNIQUE (id_configurazione,id),
	CONSTRAINT uniq_stat_pers_2 UNIQUE (id_configurazione,id_plugin),
	-- fk/pk keys constraints
	CONSTRAINT fk_stat_personalizzate_1 FOREIGN KEY (id_configurazione) REFERENCES plugins_conf_azioni(id) ON DELETE CASCADE,
	CONSTRAINT fk_stat_personalizzate_2 FOREIGN KEY (id_plugin) REFERENCES plugins(id) ON DELETE CASCADE,
	CONSTRAINT pk_stat_personalizzate PRIMARY KEY (pid)
);

CREATE TRIGGER trg_stat_personalizzate
BEFORE
insert on stat_personalizzate
for each row
begin
   IF (:new.pid IS NULL) THEN
      SELECT seq_stat_personalizzate.nextval INTO :new.pid
                FROM DUAL;
   END IF;
end;
/




-- govway
		  		
-- **** Servizi Applicativi ****

CREATE SEQUENCE seq_servizi_applicativi MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE servizi_applicativi
(
	nome VARCHAR2(2000) NOT NULL,
	tipo VARCHAR2(255),
	as_client NUMBER,
	descrizione VARCHAR2(4000),
	-- * Risposta Asincrona *
	-- valori 0/1 indicano rispettivamente FALSE/TRUE
	sbustamentorisp NUMBER,
	getmsgrisp VARCHAR2(255),
	-- FOREIGN KEY (id_gestione_errore_risp) REFERENCES gestione_errore(id) Nota: indica un eventuale tipologia di gestione dell'errore per la risposta asincrona
	id_gestione_errore_risp NUMBER,
	tipoauthrisp VARCHAR2(255),
	utenterisp VARCHAR2(255),
	passwordrisp VARCHAR2(255),
	enc_passwordrisp CLOB,
	invio_x_rif_risp VARCHAR2(255),
	risposta_x_rif_risp VARCHAR2(255),
	id_connettore_risp NUMBER NOT NULL,
	sbustamento_protocol_info_risp NUMBER,
	-- * Invocazione Servizio *
	-- valori 0/1 indicano rispettivamente FALSE/TRUE
	sbustamentoinv NUMBER,
	getmsginv VARCHAR2(255),
	-- FOREIGN KEY (id_gestione_errore_inv) REFERENCES gestione_errore(id) Nota: indica un eventuale tipologia di gestione dell'errore per l'invocazione servizio
	id_gestione_errore_inv NUMBER,
	tipoauthinv VARCHAR2(255),
	utenteinv VARCHAR2(255),
	passwordinv VARCHAR2(255),
	enc_passwordinv CLOB,
	invio_x_rif_inv VARCHAR2(255),
	risposta_x_rif_inv VARCHAR2(255),
	id_connettore_inv NUMBER NOT NULL,
	sbustamento_protocol_info_inv NUMBER,
	-- * SoggettoErogatore *
	id_soggetto NUMBER NOT NULL,
	-- * Invocazione Porta *
	fault VARCHAR2(255),
	fault_actor VARCHAR2(255),
	generic_fault_code VARCHAR2(255),
	prefix_fault_code VARCHAR2(255),
	tipoauth VARCHAR2(255),
	utente VARCHAR2(2800),
	password VARCHAR2(255),
	subject VARCHAR2(2800),
	cn_subject VARCHAR2(255),
	issuer VARCHAR2(2800),
	cn_issuer VARCHAR2(255),
	certificate BLOB,
	cert_strict_verification NUMBER,
	token_policy VARCHAR2(255),
	invio_x_rif VARCHAR2(255),
	sbustamento_protocol_info NUMBER,
	tipologia_fruizione VARCHAR2(255),
	tipologia_erogazione VARCHAR2(255),
	ora_registrazione TIMESTAMP,
	utente_richiedente VARCHAR2(255),
	data_creazione TIMESTAMP,
	utente_ultima_modifica VARCHAR2(255),
	data_ultima_modifica TIMESTAMP,
	-- fk/pk columns
	id NUMBER NOT NULL,
	-- unique constraints
	CONSTRAINT unique_servizi_applicativi_1 UNIQUE (nome,id_soggetto),
	-- fk/pk keys constraints
	CONSTRAINT fk_servizi_applicativi_1 FOREIGN KEY (id_connettore_inv) REFERENCES connettori(id),
	CONSTRAINT fk_servizi_applicativi_2 FOREIGN KEY (id_soggetto) REFERENCES soggetti(id),
	CONSTRAINT pk_servizi_applicativi PRIMARY KEY (id)
);

-- index
CREATE INDEX index_servizi_applicativi_1 ON servizi_applicativi (id_soggetto);

ALTER TABLE servizi_applicativi MODIFY sbustamentorisp DEFAULT 0;
ALTER TABLE servizi_applicativi MODIFY sbustamento_protocol_info_risp DEFAULT 1;
ALTER TABLE servizi_applicativi MODIFY sbustamentoinv DEFAULT 0;
ALTER TABLE servizi_applicativi MODIFY sbustamento_protocol_info_inv DEFAULT 1;
ALTER TABLE servizi_applicativi MODIFY sbustamento_protocol_info DEFAULT 1;
ALTER TABLE servizi_applicativi MODIFY ora_registrazione DEFAULT CURRENT_TIMESTAMP;

CREATE TRIGGER trg_servizi_applicativi
BEFORE
insert on servizi_applicativi
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_servizi_applicativi.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/



CREATE SEQUENCE seq_sa_ruoli MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE sa_ruoli
(
	id_servizio_applicativo NUMBER NOT NULL,
	ruolo VARCHAR2(255) NOT NULL,
	-- fk/pk columns
	id NUMBER NOT NULL,
	-- unique constraints
	CONSTRAINT unique_sa_ruoli_1 UNIQUE (id_servizio_applicativo,ruolo),
	-- fk/pk keys constraints
	CONSTRAINT fk_sa_ruoli_1 FOREIGN KEY (id_servizio_applicativo) REFERENCES servizi_applicativi(id),
	CONSTRAINT pk_sa_ruoli PRIMARY KEY (id)
);

-- index
CREATE INDEX INDEX_SA_RUOLI ON sa_ruoli (id_servizio_applicativo);
CREATE TRIGGER trg_sa_ruoli
BEFORE
insert on sa_ruoli
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_sa_ruoli.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/



CREATE SEQUENCE seq_sa_credenziali MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE sa_credenziali
(
	id_servizio_applicativo NUMBER NOT NULL,
	subject VARCHAR2(2800),
	cn_subject VARCHAR2(255),
	issuer VARCHAR2(2800),
	cn_issuer VARCHAR2(255),
	certificate BLOB,
	cert_strict_verification NUMBER,
	-- fk/pk columns
	id NUMBER NOT NULL,
	-- fk/pk keys constraints
	CONSTRAINT fk_sa_credenziali_1 FOREIGN KEY (id_servizio_applicativo) REFERENCES servizi_applicativi(id),
	CONSTRAINT pk_sa_credenziali PRIMARY KEY (id)
);

-- index
CREATE INDEX INDEX_SA_CRED ON sa_credenziali (id_servizio_applicativo);
CREATE TRIGGER trg_sa_credenziali
BEFORE
insert on sa_credenziali
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_sa_credenziali.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/



CREATE SEQUENCE seq_sa_properties MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE sa_properties
(
	id_servizio_applicativo NUMBER NOT NULL,
	nome VARCHAR2(255) NOT NULL,
	valore VARCHAR2(4000) NOT NULL,
	enc_value CLOB,
	-- fk/pk columns
	id NUMBER NOT NULL,
	-- unique constraints
	CONSTRAINT uniq_sa_properties_1 UNIQUE (id_servizio_applicativo,nome),
	-- fk/pk keys constraints
	CONSTRAINT fk_sa_properties_1 FOREIGN KEY (id_servizio_applicativo) REFERENCES servizi_applicativi(id),
	CONSTRAINT pk_sa_properties PRIMARY KEY (id)
);

-- index
CREATE INDEX INDEX_SA_PROP ON sa_properties (id_servizio_applicativo);
CREATE TRIGGER trg_sa_properties
BEFORE
insert on sa_properties
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_sa_properties.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/




-- govway
		  		
-- **** Porte Delegate ****

CREATE SEQUENCE seq_porte_delegate MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE porte_delegate
(
	nome_porta VARCHAR2(2000) NOT NULL,
	descrizione VARCHAR2(4000),
	-- * Soggetto Erogatore *
	-- tipo/nome per le modalita static
	-- id utilizzato in caso di registryInput
	id_soggetto_erogatore NUMBER,
	tipo_soggetto_erogatore VARCHAR2(255) NOT NULL,
	nome_soggetto_erogatore VARCHAR2(255) NOT NULL,
	-- * Servizio *
	-- tipo/nome/versione per le modalita static
	-- id utilizzato in caso di registryInput
	id_servizio NUMBER,
	tipo_servizio VARCHAR2(255) NOT NULL,
	nome_servizio VARCHAR2(255) NOT NULL,
	versione_servizio NUMBER NOT NULL,
	id_accordo NUMBER,
	id_port_type NUMBER,
	-- * Azione *
	-- tipo/nome per le modalita static
	-- tipo/pattern per la modalita contentBased/urlBased
	-- id utilizzato in caso di registryInput
	id_azione NUMBER,
	nome_azione VARCHAR2(255),
	mode_azione VARCHAR2(255),
	pattern_azione VARCHAR2(255),
	nome_porta_delegante_azione VARCHAR2(255),
	-- abilitato/disabilitato
	force_interface_based_azione VARCHAR2(255),
	-- Controllo Accessi
	autenticazione VARCHAR2(255),
	-- abilitato/disabilitato
	autenticazione_opzionale VARCHAR2(255),
	-- Gestione Token
	token_policy VARCHAR2(255),
	token_opzionale VARCHAR2(255),
	token_validazione VARCHAR2(255),
	token_introspection VARCHAR2(255),
	token_user_info VARCHAR2(255),
	token_forward VARCHAR2(255),
	token_options VARCHAR2(4000),
	token_authn_issuer VARCHAR2(255),
	token_authn_client_id VARCHAR2(255),
	token_authn_subject VARCHAR2(255),
	token_authn_username VARCHAR2(255),
	token_authn_email VARCHAR2(255),
	-- Autorizzazione
	autorizzazione VARCHAR2(255),
	autorizzazione_xacml CLOB,
	autorizzazione_contenuto VARCHAR2(255),
	-- disable/packaging/unpackaging/verify
	mtom_request_mode VARCHAR2(255),
	-- disable/packaging/unpackaging/verify
	mtom_response_mode VARCHAR2(255),
	-- abilitato/disabilitato (se abilitato le WSSproperties sono presenti nelle tabelle ...._security_request/response)
	security VARCHAR2(255),
	-- abilitato/disabilitato
	security_mtom_req VARCHAR2(255),
	-- abilitato/disabilitato
	security_mtom_res VARCHAR2(255),
	security_request_mode VARCHAR2(255),
	security_response_mode VARCHAR2(255),
	-- abilitato/disabilitato
	ricevuta_asincrona_sim VARCHAR2(255),
	-- abilitato/disabilitato
	ricevuta_asincrona_asim VARCHAR2(255),
	-- abilitato/disabilitato/warningOnly
	validazione_contenuti_stato VARCHAR2(255),
	-- wsdl/interface/xsd
	validazione_contenuti_tipo VARCHAR2(255),
	-- abilitato/disabilitato
	validazione_contenuti_mtom VARCHAR2(255),
	-- lista di tipi separati dalla virgola
	integrazione VARCHAR2(4000),
	-- scadenza correlazione applicativa
	scadenza_correlazione_appl VARCHAR2(255),
	-- abilitato/disabilitato
	allega_body VARCHAR2(255),
	-- abilitato/disabilitato
	scarta_body VARCHAR2(255),
	-- abilitato/disabilitato
	gestione_manifest VARCHAR2(255),
	-- abilitato/disabilitato
	stateless VARCHAR2(255),
	-- abilitato/disabilitato
	local_forward VARCHAR2(255),
	-- Nome della PortaApplicativa
	local_forward_pa VARCHAR2(255),
	-- all/any
	ruoli_match VARCHAR2(255),
	token_sa_stato VARCHAR2(255),
	token_ruoli_stato VARCHAR2(255),
	-- all/any
	token_ruoli_match VARCHAR2(255),
	token_ruoli_tipologia VARCHAR2(255),
	scope_stato VARCHAR2(255),
	-- all/any
	scope_match VARCHAR2(255),
	-- abilitato/disabilitato
	ricerca_porta_azione_delegata VARCHAR2(255),
	-- Livello Log Messaggi Diagnostici
	msg_diag_severita VARCHAR2(255),
	-- Tracciamento delle transazioni
	tracciamento_stato VARCHAR2(255),
	tracciamento_esiti VARCHAR2(255),
	transazioni_tempi VARCHAR2(255),
	transazioni_token VARCHAR2(255),
	-- Gestione CORS
	cors_stato VARCHAR2(255),
	cors_tipo VARCHAR2(255),
	cors_all_allow_origins VARCHAR2(255),
	cors_all_allow_methods VARCHAR2(255),
	cors_all_allow_headers VARCHAR2(255),
	cors_allow_credentials VARCHAR2(255),
	cors_allow_max_age NUMBER,
	cors_allow_max_age_seconds NUMBER,
	cors_allow_origins CLOB,
	cors_allow_headers CLOB,
	cors_allow_methods CLOB,
	cors_allow_expose_headers CLOB,
	-- Response caching
	response_cache_stato VARCHAR2(255),
	response_cache_seconds NUMBER,
	response_cache_max_msg_size NUMBER,
	response_cache_hash_url VARCHAR2(255),
	response_cache_hash_query VARCHAR2(255),
	response_cache_hash_query_list CLOB,
	response_cache_hash_headers VARCHAR2(255),
	response_cache_hash_hdr_list CLOB,
	response_cache_hash_payload VARCHAR2(255),
	response_cache_control_nocache NUMBER,
	response_cache_control_maxage NUMBER,
	response_cache_control_nostore NUMBER,
	-- Stato della porta: abilitato/disabilitato
	stato VARCHAR2(255),
	-- proprietario porta delegata (Soggetto fruitore)
	id_soggetto NUMBER NOT NULL,
	ora_registrazione TIMESTAMP,
	options VARCHAR2(4000),
	canale VARCHAR2(255),
	utente_richiedente VARCHAR2(255),
	data_creazione TIMESTAMP,
	utente_ultima_modifica VARCHAR2(255),
	data_ultima_modifica TIMESTAMP,
	-- fk/pk columns
	id NUMBER NOT NULL,
	-- unique constraints
	CONSTRAINT unique_porte_delegate_1 UNIQUE (nome_porta),
	-- fk/pk keys constraints
	CONSTRAINT fk_porte_delegate_1 FOREIGN KEY (id_soggetto) REFERENCES soggetti(id),
	CONSTRAINT pk_porte_delegate PRIMARY KEY (id)
);

-- index
CREATE INDEX index_porte_delegate_1 ON porte_delegate (id_soggetto);
CREATE INDEX index_porte_delegate_2 ON porte_delegate (canale);

ALTER TABLE porte_delegate MODIFY versione_servizio DEFAULT 1;
ALTER TABLE porte_delegate MODIFY ora_registrazione DEFAULT CURRENT_TIMESTAMP;

CREATE TRIGGER trg_porte_delegate
BEFORE
insert on porte_delegate
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_porte_delegate.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/



CREATE SEQUENCE seq_porte_delegate_sa MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE porte_delegate_sa
(
	id_porta NUMBER NOT NULL,
	id_servizio_applicativo NUMBER NOT NULL,
	-- fk/pk columns
	id NUMBER NOT NULL,
	-- unique constraints
	CONSTRAINT unique_porte_delegate_sa_1 UNIQUE (id_porta,id_servizio_applicativo),
	-- fk/pk keys constraints
	CONSTRAINT fk_porte_delegate_sa_1 FOREIGN KEY (id_servizio_applicativo) REFERENCES servizi_applicativi(id),
	CONSTRAINT fk_porte_delegate_sa_2 FOREIGN KEY (id_porta) REFERENCES porte_delegate(id),
	CONSTRAINT pk_porte_delegate_sa PRIMARY KEY (id)
);

-- index
CREATE INDEX INDEX_PD_SA ON porte_delegate_sa (id_porta);
CREATE TRIGGER trg_porte_delegate_sa
BEFORE
insert on porte_delegate_sa
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_porte_delegate_sa.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/



CREATE SEQUENCE seq_pd_auth_properties MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE pd_auth_properties
(
	id_porta NUMBER NOT NULL,
	nome VARCHAR2(255) NOT NULL,
	valore VARCHAR2(255) NOT NULL,
	enc_value CLOB,
	-- fk/pk columns
	id NUMBER NOT NULL,
	-- unique constraints
	CONSTRAINT uniq_pd_auth_props_1 UNIQUE (id_porta,nome),
	-- fk/pk keys constraints
	CONSTRAINT fk_pd_auth_properties_1 FOREIGN KEY (id_porta) REFERENCES porte_delegate(id),
	CONSTRAINT pk_pd_auth_properties PRIMARY KEY (id)
);

-- index
CREATE INDEX INDEX_PD_AUTH_PROP ON pd_auth_properties (id_porta);
CREATE TRIGGER trg_pd_auth_properties
BEFORE
insert on pd_auth_properties
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_pd_auth_properties.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/



CREATE SEQUENCE seq_pd_authz_properties MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE pd_authz_properties
(
	id_porta NUMBER NOT NULL,
	nome VARCHAR2(255) NOT NULL,
	valore VARCHAR2(255) NOT NULL,
	enc_value CLOB,
	-- fk/pk columns
	id NUMBER NOT NULL,
	-- unique constraints
	CONSTRAINT uniq_pd_authz_props_1 UNIQUE (id_porta,nome),
	-- fk/pk keys constraints
	CONSTRAINT fk_pd_authz_properties_1 FOREIGN KEY (id_porta) REFERENCES porte_delegate(id),
	CONSTRAINT pk_pd_authz_properties PRIMARY KEY (id)
);

-- index
CREATE INDEX INDEX_PD_AUTHZ_PROP ON pd_authz_properties (id_porta);
CREATE TRIGGER trg_pd_authz_properties
BEFORE
insert on pd_authz_properties
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_pd_authz_properties.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/



CREATE SEQUENCE seq_pd_authzc_properties MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE pd_authzc_properties
(
	id_porta NUMBER NOT NULL,
	nome VARCHAR2(255) NOT NULL,
	valore VARCHAR2(255) NOT NULL,
	enc_value CLOB,
	-- fk/pk columns
	id NUMBER NOT NULL,
	-- unique constraints
	CONSTRAINT uniq_pd_authzc_props_1 UNIQUE (id_porta,nome),
	-- fk/pk keys constraints
	CONSTRAINT fk_pd_authzc_properties_1 FOREIGN KEY (id_porta) REFERENCES porte_delegate(id),
	CONSTRAINT pk_pd_authzc_properties PRIMARY KEY (id)
);

-- index
CREATE INDEX INDEX_PD_AUTHZC_PROP ON pd_authzc_properties (id_porta);
CREATE TRIGGER trg_pd_authzc_properties
BEFORE
insert on pd_authzc_properties
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_pd_authzc_properties.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/



CREATE SEQUENCE seq_pd_ct_properties MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE pd_ct_properties
(
	id_porta NUMBER NOT NULL,
	nome VARCHAR2(255) NOT NULL,
	valore VARCHAR2(255) NOT NULL,
	-- fk/pk columns
	id NUMBER NOT NULL,
	-- unique constraints
	CONSTRAINT uniq_pd_ct_props_1 UNIQUE (id_porta,nome),
	-- fk/pk keys constraints
	CONSTRAINT fk_pd_ct_properties_1 FOREIGN KEY (id_porta) REFERENCES porte_delegate(id),
	CONSTRAINT pk_pd_ct_properties PRIMARY KEY (id)
);

-- index
CREATE INDEX idx_pd_ct_props_1 ON pd_ct_properties (valore);
CREATE INDEX INDEX_PD_CT_PROP ON pd_ct_properties (id_porta);
CREATE TRIGGER trg_pd_ct_properties
BEFORE
insert on pd_ct_properties
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_pd_ct_properties.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/



CREATE SEQUENCE seq_pd_properties MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE pd_properties
(
	id_porta NUMBER NOT NULL,
	nome VARCHAR2(255) NOT NULL,
	valore VARCHAR2(255) NOT NULL,
	enc_value CLOB,
	-- fk/pk columns
	id NUMBER NOT NULL,
	-- unique constraints
	CONSTRAINT uniq_pd_properties_1 UNIQUE (id_porta,nome),
	-- fk/pk keys constraints
	CONSTRAINT fk_pd_properties_1 FOREIGN KEY (id_porta) REFERENCES porte_delegate(id),
	CONSTRAINT pk_pd_properties PRIMARY KEY (id)
);

-- index
CREATE INDEX INDEX_PD_PROP ON pd_properties (id_porta);
CREATE TRIGGER trg_pd_properties
BEFORE
insert on pd_properties
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_pd_properties.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/



CREATE SEQUENCE seq_pd_mtom_request MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE pd_mtom_request
(
	id_porta NUMBER NOT NULL,
	nome VARCHAR2(255) NOT NULL,
	pattern CLOB NOT NULL,
	content_type VARCHAR2(255),
	required NUMBER NOT NULL,
	-- fk/pk columns
	id NUMBER NOT NULL,
	-- fk/pk keys constraints
	CONSTRAINT fk_pd_mtom_request_1 FOREIGN KEY (id_porta) REFERENCES porte_delegate(id),
	CONSTRAINT pk_pd_mtom_request PRIMARY KEY (id)
);

-- index
CREATE INDEX INDEX_PD_MTOMTREQ ON pd_mtom_request (id_porta);
CREATE TRIGGER trg_pd_mtom_request
BEFORE
insert on pd_mtom_request
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_pd_mtom_request.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/



CREATE SEQUENCE seq_pd_mtom_response MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE pd_mtom_response
(
	id_porta NUMBER NOT NULL,
	nome VARCHAR2(255) NOT NULL,
	pattern CLOB NOT NULL,
	content_type VARCHAR2(255),
	required NUMBER NOT NULL,
	-- fk/pk columns
	id NUMBER NOT NULL,
	-- fk/pk keys constraints
	CONSTRAINT fk_pd_mtom_response_1 FOREIGN KEY (id_porta) REFERENCES porte_delegate(id),
	CONSTRAINT pk_pd_mtom_response PRIMARY KEY (id)
);

-- index
CREATE INDEX INDEX_PD_MTOMTRES ON pd_mtom_response (id_porta);
CREATE TRIGGER trg_pd_mtom_response
BEFORE
insert on pd_mtom_response
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_pd_mtom_response.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/



CREATE SEQUENCE seq_pd_security_request MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE pd_security_request
(
	id_porta NUMBER NOT NULL,
	nome VARCHAR2(255) NOT NULL,
	valore CLOB NOT NULL,
	enc_value CLOB,
	-- fk/pk columns
	id NUMBER NOT NULL,
	-- fk/pk keys constraints
	CONSTRAINT fk_pd_security_request_1 FOREIGN KEY (id_porta) REFERENCES porte_delegate(id),
	CONSTRAINT pk_pd_security_request PRIMARY KEY (id)
);

-- index
CREATE INDEX INDEX_PD_WSSREQ ON pd_security_request (id_porta);
CREATE TRIGGER trg_pd_security_request
BEFORE
insert on pd_security_request
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_pd_security_request.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/



CREATE SEQUENCE seq_pd_security_response MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE pd_security_response
(
	id_porta NUMBER NOT NULL,
	nome VARCHAR2(255) NOT NULL,
	valore CLOB NOT NULL,
	enc_value CLOB,
	-- fk/pk columns
	id NUMBER NOT NULL,
	-- fk/pk keys constraints
	CONSTRAINT fk_pd_security_response_1 FOREIGN KEY (id_porta) REFERENCES porte_delegate(id),
	CONSTRAINT pk_pd_security_response PRIMARY KEY (id)
);

-- index
CREATE INDEX INDEX_PD_WSSRES ON pd_security_response (id_porta);
CREATE TRIGGER trg_pd_security_response
BEFORE
insert on pd_security_response
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_pd_security_response.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/



CREATE SEQUENCE seq_pd_correlazione MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE pd_correlazione
(
	id_porta NUMBER NOT NULL,
	nome_elemento VARCHAR2(255),
	-- modalita di scelta user input, content-based, url-based, disabilitato
	mode_correlazione VARCHAR2(255),
	-- pattern utilizzato solo per content/url based
	pattern CLOB,
	-- blocca/accetta
	identificazione_fallita VARCHAR2(255),
	-- abilitato/disabilitato
	riuso_id VARCHAR2(255),
	-- fk/pk columns
	id NUMBER NOT NULL,
	-- fk/pk keys constraints
	CONSTRAINT fk_pd_correlazione_1 FOREIGN KEY (id_porta) REFERENCES porte_delegate(id),
	CONSTRAINT pk_pd_correlazione PRIMARY KEY (id)
);

-- index
CREATE INDEX INDEX_PD_CORR_REQ ON pd_correlazione (id_porta);
CREATE TRIGGER trg_pd_correlazione
BEFORE
insert on pd_correlazione
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_pd_correlazione.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/



CREATE SEQUENCE seq_pd_correlazione_risposta MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE pd_correlazione_risposta
(
	id_porta NUMBER NOT NULL,
	nome_elemento VARCHAR2(255),
	-- modalita di scelta user input, content-based, url-based, disabilitato
	mode_correlazione VARCHAR2(255),
	-- pattern utilizzato solo per content/url based
	pattern CLOB,
	-- blocca/accetta
	identificazione_fallita VARCHAR2(255),
	-- fk/pk columns
	id NUMBER NOT NULL,
	-- fk/pk keys constraints
	CONSTRAINT fk_pd_correlazione_risposta_1 FOREIGN KEY (id_porta) REFERENCES porte_delegate(id),
	CONSTRAINT pk_pd_correlazione_risposta PRIMARY KEY (id)
);

-- index
CREATE INDEX INDEX_PD_CORR_RES ON pd_correlazione_risposta (id_porta);
CREATE TRIGGER trg_pd_correlazione_risposta
BEFORE
insert on pd_correlazione_risposta
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_pd_correlazione_risposta.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/



CREATE SEQUENCE seq_pd_ruoli MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE pd_ruoli
(
	id_porta NUMBER NOT NULL,
	ruolo VARCHAR2(255) NOT NULL,
	-- fk/pk columns
	id NUMBER NOT NULL,
	-- unique constraints
	CONSTRAINT unique_pd_ruoli_1 UNIQUE (id_porta,ruolo),
	-- fk/pk keys constraints
	CONSTRAINT fk_pd_ruoli_1 FOREIGN KEY (id_porta) REFERENCES porte_delegate(id),
	CONSTRAINT pk_pd_ruoli PRIMARY KEY (id)
);

CREATE TRIGGER trg_pd_ruoli
BEFORE
insert on pd_ruoli
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_pd_ruoli.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/



CREATE SEQUENCE seq_pd_scope MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE pd_scope
(
	id_porta NUMBER NOT NULL,
	scope VARCHAR2(255) NOT NULL,
	-- fk/pk columns
	id NUMBER NOT NULL,
	-- unique constraints
	CONSTRAINT unique_pd_scope_1 UNIQUE (id_porta,scope),
	-- fk/pk keys constraints
	CONSTRAINT fk_pd_scope_1 FOREIGN KEY (id_porta) REFERENCES porte_delegate(id),
	CONSTRAINT pk_pd_scope PRIMARY KEY (id)
);

CREATE TRIGGER trg_pd_scope
BEFORE
insert on pd_scope
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_pd_scope.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/



CREATE SEQUENCE seq_pd_azioni MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE pd_azioni
(
	id_porta NUMBER NOT NULL,
	azione VARCHAR2(255) NOT NULL,
	-- fk/pk columns
	id NUMBER NOT NULL,
	-- unique constraints
	CONSTRAINT unique_pd_azioni_1 UNIQUE (id_porta,azione),
	-- fk/pk keys constraints
	CONSTRAINT fk_pd_azioni_1 FOREIGN KEY (id_porta) REFERENCES porte_delegate(id),
	CONSTRAINT pk_pd_azioni PRIMARY KEY (id)
);

CREATE TRIGGER trg_pd_azioni
BEFORE
insert on pd_azioni
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_pd_azioni.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/



CREATE SEQUENCE seq_pd_cache_regole MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE pd_cache_regole
(
	id_porta NUMBER NOT NULL,
	status_min NUMBER,
	status_max NUMBER,
	fault NUMBER,
	cache_seconds NUMBER,
	-- fk/pk columns
	id NUMBER NOT NULL,
	-- fk/pk keys constraints
	CONSTRAINT fk_pd_cache_regole_1 FOREIGN KEY (id_porta) REFERENCES porte_delegate(id),
	CONSTRAINT pk_pd_cache_regole PRIMARY KEY (id)
);

-- index
CREATE INDEX INDEX_PD_CACHE ON pd_cache_regole (id_porta);

ALTER TABLE pd_cache_regole MODIFY fault DEFAULT 0;

CREATE TRIGGER trg_pd_cache_regole
BEFORE
insert on pd_cache_regole
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_pd_cache_regole.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/



CREATE SEQUENCE seq_pd_transform MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE pd_transform
(
	id_porta NUMBER NOT NULL,
	nome VARCHAR2(255) NOT NULL,
	posizione NUMBER NOT NULL,
	stato VARCHAR2(255),
	applicabilita_azioni CLOB,
	applicabilita_ct CLOB,
	applicabilita_pattern CLOB,
	applicabilita_connettori CLOB,
	req_conversione_enabled NUMBER NOT NULL,
	req_conversione_tipo VARCHAR2(255),
	req_conversione_template BLOB,
	req_content_type VARCHAR2(255),
	rest_transformation NUMBER NOT NULL,
	rest_method VARCHAR2(255),
	rest_path VARCHAR2(4000),
	soap_transformation NUMBER NOT NULL,
	soap_version VARCHAR2(255),
	soap_action VARCHAR2(255),
	soap_envelope NUMBER NOT NULL,
	soap_envelope_as_attach NUMBER NOT NULL,
	soap_envelope_tipo VARCHAR2(255),
	soap_envelope_template BLOB,
	-- fk/pk columns
	id NUMBER NOT NULL,
	-- unique constraints
	CONSTRAINT unique_pd_transform_1 UNIQUE (id_porta,nome),
	CONSTRAINT unique_pd_transform_2 UNIQUE (id_porta,posizione),
	-- fk/pk keys constraints
	CONSTRAINT fk_pd_transform_1 FOREIGN KEY (id_porta) REFERENCES porte_delegate(id),
	CONSTRAINT pk_pd_transform PRIMARY KEY (id)
);


ALTER TABLE pd_transform MODIFY req_conversione_enabled DEFAULT 0;
ALTER TABLE pd_transform MODIFY rest_transformation DEFAULT 0;
ALTER TABLE pd_transform MODIFY soap_transformation DEFAULT 0;
ALTER TABLE pd_transform MODIFY soap_envelope DEFAULT 0;
ALTER TABLE pd_transform MODIFY soap_envelope_as_attach DEFAULT 0;

CREATE TRIGGER trg_pd_transform
BEFORE
insert on pd_transform
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_pd_transform.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/



CREATE SEQUENCE seq_pd_transform_sa MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE pd_transform_sa
(
	id_trasformazione NUMBER NOT NULL,
	id_servizio_applicativo NUMBER NOT NULL,
	-- fk/pk columns
	id NUMBER NOT NULL,
	-- unique constraints
	CONSTRAINT unique_pd_transform_sa_1 UNIQUE (id_trasformazione,id_servizio_applicativo),
	-- fk/pk keys constraints
	CONSTRAINT fk_pd_transform_sa_1 FOREIGN KEY (id_servizio_applicativo) REFERENCES servizi_applicativi(id),
	CONSTRAINT fk_pd_transform_sa_2 FOREIGN KEY (id_trasformazione) REFERENCES pd_transform(id),
	CONSTRAINT pk_pd_transform_sa PRIMARY KEY (id)
);

CREATE TRIGGER trg_pd_transform_sa
BEFORE
insert on pd_transform_sa
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_pd_transform_sa.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/



CREATE SEQUENCE seq_pd_transform_hdr MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE pd_transform_hdr
(
	id_trasformazione NUMBER NOT NULL,
	tipo VARCHAR2(255) NOT NULL,
	nome VARCHAR2(255) NOT NULL,
	valore CLOB,
	identificazione_fallita VARCHAR2(255),
	-- fk/pk columns
	id NUMBER NOT NULL,
	-- fk/pk keys constraints
	CONSTRAINT fk_pd_transform_hdr_1 FOREIGN KEY (id_trasformazione) REFERENCES pd_transform(id),
	CONSTRAINT pk_pd_transform_hdr PRIMARY KEY (id)
);

-- index
CREATE INDEX idx_pd_trasf_hdr_1 ON pd_transform_hdr (id_trasformazione);
CREATE TRIGGER trg_pd_transform_hdr
BEFORE
insert on pd_transform_hdr
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_pd_transform_hdr.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/



CREATE SEQUENCE seq_pd_transform_url MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE pd_transform_url
(
	id_trasformazione NUMBER NOT NULL,
	tipo VARCHAR2(255) NOT NULL,
	nome VARCHAR2(255) NOT NULL,
	valore CLOB,
	identificazione_fallita VARCHAR2(255),
	-- fk/pk columns
	id NUMBER NOT NULL,
	-- fk/pk keys constraints
	CONSTRAINT fk_pd_transform_url_1 FOREIGN KEY (id_trasformazione) REFERENCES pd_transform(id),
	CONSTRAINT pk_pd_transform_url PRIMARY KEY (id)
);

-- index
CREATE INDEX idx_pd_trasf_url_1 ON pd_transform_url (id_trasformazione);
CREATE TRIGGER trg_pd_transform_url
BEFORE
insert on pd_transform_url
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_pd_transform_url.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/



CREATE SEQUENCE seq_pd_transform_risp MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE pd_transform_risp
(
	id_trasformazione NUMBER NOT NULL,
	nome VARCHAR2(255) NOT NULL,
	posizione NUMBER NOT NULL,
	applicabilita_status_min NUMBER,
	applicabilita_status_max NUMBER,
	applicabilita_ct CLOB,
	applicabilita_pattern CLOB,
	conversione_enabled NUMBER NOT NULL,
	conversione_tipo VARCHAR2(255),
	conversione_template BLOB,
	content_type VARCHAR2(255),
	return_code VARCHAR2(255),
	soap_envelope NUMBER NOT NULL,
	soap_envelope_as_attach NUMBER NOT NULL,
	soap_envelope_tipo VARCHAR2(255),
	soap_envelope_template BLOB,
	-- fk/pk columns
	id NUMBER NOT NULL,
	-- unique constraints
	CONSTRAINT uniq_pd_trasf_resp_1 UNIQUE (id_trasformazione,nome),
	CONSTRAINT uniq_pd_trasf_resp_2 UNIQUE (id_trasformazione,posizione),
	-- fk/pk keys constraints
	CONSTRAINT fk_pd_transform_risp_1 FOREIGN KEY (id_trasformazione) REFERENCES pd_transform(id),
	CONSTRAINT pk_pd_transform_risp PRIMARY KEY (id)
);


ALTER TABLE pd_transform_risp MODIFY conversione_enabled DEFAULT 0;
ALTER TABLE pd_transform_risp MODIFY soap_envelope DEFAULT 0;
ALTER TABLE pd_transform_risp MODIFY soap_envelope_as_attach DEFAULT 0;

CREATE TRIGGER trg_pd_transform_risp
BEFORE
insert on pd_transform_risp
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_pd_transform_risp.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/



CREATE SEQUENCE seq_pd_transform_risp_hdr MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE pd_transform_risp_hdr
(
	id_transform_risp NUMBER NOT NULL,
	tipo VARCHAR2(255) NOT NULL,
	nome VARCHAR2(255) NOT NULL,
	valore CLOB,
	identificazione_fallita VARCHAR2(255),
	-- fk/pk columns
	id NUMBER NOT NULL,
	-- fk/pk keys constraints
	CONSTRAINT fk_pd_transform_risp_hdr_1 FOREIGN KEY (id_transform_risp) REFERENCES pd_transform_risp(id),
	CONSTRAINT pk_pd_transform_risp_hdr PRIMARY KEY (id)
);

-- index
CREATE INDEX idx_pd_trasf_hdr_resp_1 ON pd_transform_risp_hdr (id_transform_risp);
CREATE TRIGGER trg_pd_transform_risp_hdr
BEFORE
insert on pd_transform_risp_hdr
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_pd_transform_risp_hdr.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/



CREATE SEQUENCE seq_pd_handlers MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE pd_handlers
(
	id_porta NUMBER NOT NULL,
	tipologia VARCHAR2(255) NOT NULL,
	tipo VARCHAR2(255) NOT NULL,
	posizione NUMBER NOT NULL,
	stato VARCHAR2(255),
	-- fk/pk columns
	id NUMBER NOT NULL,
	-- unique constraints
	CONSTRAINT unique_pd_handlers_1 UNIQUE (id_porta,tipologia,tipo),
	-- fk/pk keys constraints
	CONSTRAINT fk_pd_handlers_1 FOREIGN KEY (id_porta) REFERENCES porte_delegate(id),
	CONSTRAINT pk_pd_handlers PRIMARY KEY (id)
);

CREATE TRIGGER trg_pd_handlers
BEFORE
insert on pd_handlers
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_pd_handlers.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/



CREATE SEQUENCE seq_pd_aa MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE pd_aa
(
	id_porta NUMBER NOT NULL,
	nome VARCHAR2(255) NOT NULL,
	attributi CLOB,
	-- fk/pk columns
	id NUMBER NOT NULL,
	-- unique constraints
	CONSTRAINT unique_pd_aa_1 UNIQUE (id_porta,nome),
	-- fk/pk keys constraints
	CONSTRAINT fk_pd_aa_1 FOREIGN KEY (id_porta) REFERENCES porte_delegate(id),
	CONSTRAINT pk_pd_aa PRIMARY KEY (id)
);

CREATE TRIGGER trg_pd_aa
BEFORE
insert on pd_aa
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_pd_aa.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/



CREATE SEQUENCE seq_pd_token_sa MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE pd_token_sa
(
	id_porta NUMBER NOT NULL,
	id_servizio_applicativo NUMBER NOT NULL,
	-- fk/pk columns
	id NUMBER NOT NULL,
	-- unique constraints
	CONSTRAINT uniq_pd_token_sa_1 UNIQUE (id_porta,id_servizio_applicativo),
	-- fk/pk keys constraints
	CONSTRAINT fk_pd_token_sa_1 FOREIGN KEY (id_servizio_applicativo) REFERENCES servizi_applicativi(id),
	CONSTRAINT fk_pd_token_sa_2 FOREIGN KEY (id_porta) REFERENCES porte_delegate(id),
	CONSTRAINT pk_pd_token_sa PRIMARY KEY (id)
);

-- index
CREATE INDEX INDEX_PD_TOKEN_SA ON pd_token_sa (id_porta);
CREATE TRIGGER trg_pd_token_sa
BEFORE
insert on pd_token_sa
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_pd_token_sa.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/



CREATE SEQUENCE seq_pd_token_ruoli MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE pd_token_ruoli
(
	id_porta NUMBER NOT NULL,
	ruolo VARCHAR2(255) NOT NULL,
	-- fk/pk columns
	id NUMBER NOT NULL,
	-- unique constraints
	CONSTRAINT unique_pd_token_ruoli_1 UNIQUE (id_porta,ruolo),
	-- fk/pk keys constraints
	CONSTRAINT fk_pd_token_ruoli_1 FOREIGN KEY (id_porta) REFERENCES porte_delegate(id),
	CONSTRAINT pk_pd_token_ruoli PRIMARY KEY (id)
);

CREATE TRIGGER trg_pd_token_ruoli
BEFORE
insert on pd_token_ruoli
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_pd_token_ruoli.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/




-- govway
		  		
-- **** Porte Applicative ****

CREATE SEQUENCE seq_porte_applicative MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE porte_applicative
(
	nome_porta VARCHAR2(2000) NOT NULL,
	descrizione VARCHAR2(4000),
	-- Soggetto Virtuale
	id_soggetto_virtuale NUMBER,
	tipo_soggetto_virtuale VARCHAR2(255),
	nome_soggetto_virtuale VARCHAR2(255),
	-- Servizio
	id_servizio NUMBER,
	tipo_servizio VARCHAR2(255) NOT NULL,
	servizio VARCHAR2(255) NOT NULL,
	versione_servizio NUMBER NOT NULL,
	id_accordo NUMBER,
	id_port_type NUMBER,
	-- azione
	azione VARCHAR2(255),
	mode_azione VARCHAR2(255),
	pattern_azione VARCHAR2(255),
	nome_porta_delegante_azione VARCHAR2(255),
	-- abilitato/disabilitato
	force_interface_based_azione VARCHAR2(255),
	-- disable/packaging/unpackaging/verify
	mtom_request_mode VARCHAR2(255),
	-- disable/packaging/unpackaging/verify
	mtom_response_mode VARCHAR2(255),
	-- abilitato/disabilitato (se abilitato le WSSproperties sono presenti nelle tabelle ...._security_request/response)
	security VARCHAR2(255),
	-- abilitato/disabilitato
	security_mtom_req VARCHAR2(255),
	-- abilitato/disabilitato
	security_mtom_res VARCHAR2(255),
	security_request_mode VARCHAR2(255),
	security_response_mode VARCHAR2(255),
	-- abilitato/disabilitato
	ricevuta_asincrona_sim VARCHAR2(255),
	-- abilitato/disabilitato
	ricevuta_asincrona_asim VARCHAR2(255),
	-- abilitato/disabilitato/warningOnly
	validazione_contenuti_stato VARCHAR2(255),
	-- wsdl/interface/xsd
	validazione_contenuti_tipo VARCHAR2(255),
	-- abilitato/disabilitato
	validazione_contenuti_mtom VARCHAR2(255),
	-- lista di tipi separati dalla virgola
	integrazione VARCHAR2(4000),
	-- scadenza correlazione applicativa
	scadenza_correlazione_appl VARCHAR2(255),
	-- abilitato/disabilitato
	allega_body VARCHAR2(255),
	-- abilitato/disabilitato
	scarta_body VARCHAR2(255),
	-- abilitato/disabilitato
	gestione_manifest VARCHAR2(255),
	-- abilitato/disabilitato
	stateless VARCHAR2(255),
	behaviour VARCHAR2(255),
	-- Controllo Accessi
	autenticazione VARCHAR2(255),
	-- abilitato/disabilitato
	autenticazione_opzionale VARCHAR2(255),
	-- Gestione Token
	token_policy VARCHAR2(255),
	token_opzionale VARCHAR2(255),
	token_validazione VARCHAR2(255),
	token_introspection VARCHAR2(255),
	token_user_info VARCHAR2(255),
	token_forward VARCHAR2(255),
	token_options VARCHAR2(4000),
	token_authn_issuer VARCHAR2(255),
	token_authn_client_id VARCHAR2(255),
	token_authn_subject VARCHAR2(255),
	token_authn_username VARCHAR2(255),
	token_authn_email VARCHAR2(255),
	-- Autorizzazione
	autorizzazione VARCHAR2(255),
	autorizzazione_xacml CLOB,
	autorizzazione_contenuto VARCHAR2(255),
	-- all/any
	ruoli_match VARCHAR2(255),
	token_sa_stato VARCHAR2(255),
	token_ruoli_stato VARCHAR2(255),
	-- all/any
	token_ruoli_match VARCHAR2(255),
	token_ruoli_tipologia VARCHAR2(255),
	scope_stato VARCHAR2(255),
	-- all/any
	scope_match VARCHAR2(255),
	-- abilitato/disabilitato
	ricerca_porta_azione_delegata VARCHAR2(255),
	-- Livello Log Messaggi Diagnostici
	msg_diag_severita VARCHAR2(255),
	-- Tracciamento delle transazioni
	tracciamento_stato VARCHAR2(255),
	tracciamento_esiti VARCHAR2(255),
	transazioni_tempi VARCHAR2(255),
	transazioni_token VARCHAR2(255),
	-- Gestione CORS
	cors_stato VARCHAR2(255),
	cors_tipo VARCHAR2(255),
	cors_all_allow_origins VARCHAR2(255),
	cors_all_allow_methods VARCHAR2(255),
	cors_all_allow_headers VARCHAR2(255),
	cors_allow_credentials VARCHAR2(255),
	cors_allow_max_age NUMBER,
	cors_allow_max_age_seconds NUMBER,
	cors_allow_origins CLOB,
	cors_allow_headers CLOB,
	cors_allow_methods CLOB,
	cors_allow_expose_headers CLOB,
	-- Response caching
	response_cache_stato VARCHAR2(255),
	response_cache_seconds NUMBER,
	response_cache_max_msg_size NUMBER,
	response_cache_hash_url VARCHAR2(255),
	response_cache_hash_query VARCHAR2(255),
	response_cache_hash_query_list CLOB,
	response_cache_hash_headers VARCHAR2(255),
	response_cache_hash_hdr_list CLOB,
	response_cache_hash_payload VARCHAR2(255),
	response_cache_control_nocache NUMBER,
	response_cache_control_maxage NUMBER,
	response_cache_control_nostore NUMBER,
	id_sa_default NUMBER,
	-- Stato della porta: abilitato/disabilitato
	stato VARCHAR2(255),
	-- proprietario porta applicativa
	id_soggetto NUMBER NOT NULL,
	ora_registrazione TIMESTAMP,
	options VARCHAR2(4000),
	canale VARCHAR2(255),
	utente_richiedente VARCHAR2(255),
	data_creazione TIMESTAMP,
	utente_ultima_modifica VARCHAR2(255),
	data_ultima_modifica TIMESTAMP,
	-- fk/pk columns
	id NUMBER NOT NULL,
	-- unique constraints
	CONSTRAINT unique_porte_applicative_1 UNIQUE (nome_porta),
	-- fk/pk keys constraints
	CONSTRAINT fk_porte_applicative_1 FOREIGN KEY (id_soggetto) REFERENCES soggetti(id),
	CONSTRAINT pk_porte_applicative PRIMARY KEY (id)
);

-- index
CREATE INDEX index_porte_applicative_1 ON porte_applicative (id_soggetto);
CREATE INDEX index_porte_applicative_2 ON porte_applicative (canale);

ALTER TABLE porte_applicative MODIFY versione_servizio DEFAULT 1;
ALTER TABLE porte_applicative MODIFY ora_registrazione DEFAULT CURRENT_TIMESTAMP;

CREATE TRIGGER trg_porte_applicative
BEFORE
insert on porte_applicative
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_porte_applicative.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/



CREATE SEQUENCE seq_porte_applicative_sa MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE porte_applicative_sa
(
	id_porta NUMBER NOT NULL,
	id_servizio_applicativo NUMBER NOT NULL,
	-- Dati Connettore
	connettore_nome VARCHAR2(255),
	connettore_notifica NUMBER,
	connettore_descrizione VARCHAR2(4000),
	connettore_stato VARCHAR2(255),
	connettore_scheduling VARCHAR2(255),
	connettore_filtri CLOB,
	connettore_coda VARCHAR2(10),
	connettore_priorita VARCHAR2(10),
	connettore_max_priorita NUMBER,
	utente_richiedente VARCHAR2(255),
	data_creazione TIMESTAMP,
	utente_ultima_modifica VARCHAR2(255),
	data_ultima_modifica TIMESTAMP,
	-- fk/pk columns
	id NUMBER NOT NULL,
	-- unique constraints
	CONSTRAINT unique_porte_applicative_sa_1 UNIQUE (id_porta,id_servizio_applicativo),
	-- fk/pk keys constraints
	CONSTRAINT fk_porte_applicative_sa_1 FOREIGN KEY (id_servizio_applicativo) REFERENCES servizi_applicativi(id),
	CONSTRAINT fk_porte_applicative_sa_2 FOREIGN KEY (id_porta) REFERENCES porte_applicative(id),
	CONSTRAINT pk_porte_applicative_sa PRIMARY KEY (id)
);

-- index
CREATE INDEX INDEX_PA_SA ON porte_applicative_sa (id_porta);
CREATE INDEX INDEX_PA_SA_CODA ON porte_applicative_sa (connettore_coda,connettore_priorita,id_servizio_applicativo);
CREATE INDEX INDEX_PA_SA_CODA_MAX ON porte_applicative_sa (connettore_coda,connettore_max_priorita,id_servizio_applicativo);

ALTER TABLE porte_applicative_sa MODIFY connettore_coda DEFAULT 'DEFAULT';
ALTER TABLE porte_applicative_sa MODIFY connettore_priorita DEFAULT 'DEFAULT';
ALTER TABLE porte_applicative_sa MODIFY connettore_max_priorita DEFAULT 0;

CREATE TRIGGER trg_porte_applicative_sa
BEFORE
insert on porte_applicative_sa
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_porte_applicative_sa.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/



CREATE SEQUENCE seq_pa_sa_properties MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE pa_sa_properties
(
	id_porta NUMBER NOT NULL,
	nome VARCHAR2(255) NOT NULL,
	valore VARCHAR2(255) NOT NULL,
	-- fk/pk columns
	id NUMBER NOT NULL,
	-- unique constraints
	CONSTRAINT uniq_pa_sa_props_1 UNIQUE (id_porta,nome),
	-- fk/pk keys constraints
	CONSTRAINT fk_pa_sa_properties_1 FOREIGN KEY (id_porta) REFERENCES porte_applicative_sa(id),
	CONSTRAINT pk_pa_sa_properties PRIMARY KEY (id)
);

-- index
CREATE INDEX INDEX_PA_SA_PROP ON pa_sa_properties (id_porta);
CREATE TRIGGER trg_pa_sa_properties
BEFORE
insert on pa_sa_properties
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_pa_sa_properties.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/



CREATE SEQUENCE seq_pa_behaviour_props MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE pa_behaviour_props
(
	id_porta NUMBER NOT NULL,
	nome VARCHAR2(255) NOT NULL,
	valore VARCHAR2(4000) NOT NULL,
	-- fk/pk columns
	id NUMBER NOT NULL,
	-- unique constraints
	CONSTRAINT uniq_pa_behaviour_props_1 UNIQUE (id_porta,nome),
	-- fk/pk keys constraints
	CONSTRAINT fk_pa_behaviour_props_1 FOREIGN KEY (id_porta) REFERENCES porte_applicative(id),
	CONSTRAINT pk_pa_behaviour_props PRIMARY KEY (id)
);

-- index
CREATE INDEX INDEX_PA_BEHAVIOUR_PROP ON pa_behaviour_props (id_porta);
CREATE TRIGGER trg_pa_behaviour_props
BEFORE
insert on pa_behaviour_props
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_pa_behaviour_props.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/



CREATE SEQUENCE seq_pa_auth_properties MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE pa_auth_properties
(
	id_porta NUMBER NOT NULL,
	nome VARCHAR2(255) NOT NULL,
	valore VARCHAR2(255) NOT NULL,
	enc_value CLOB,
	-- fk/pk columns
	id NUMBER NOT NULL,
	-- unique constraints
	CONSTRAINT uniq_pa_auth_props_1 UNIQUE (id_porta,nome),
	-- fk/pk keys constraints
	CONSTRAINT fk_pa_auth_properties_1 FOREIGN KEY (id_porta) REFERENCES porte_applicative(id),
	CONSTRAINT pk_pa_auth_properties PRIMARY KEY (id)
);

-- index
CREATE INDEX INDEX_PA_AUTH_PROP ON pa_auth_properties (id_porta);
CREATE TRIGGER trg_pa_auth_properties
BEFORE
insert on pa_auth_properties
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_pa_auth_properties.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/



CREATE SEQUENCE seq_pa_authz_properties MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE pa_authz_properties
(
	id_porta NUMBER NOT NULL,
	nome VARCHAR2(255) NOT NULL,
	valore VARCHAR2(255) NOT NULL,
	enc_value CLOB,
	-- fk/pk columns
	id NUMBER NOT NULL,
	-- unique constraints
	CONSTRAINT uniq_pa_authz_props_1 UNIQUE (id_porta,nome),
	-- fk/pk keys constraints
	CONSTRAINT fk_pa_authz_properties_1 FOREIGN KEY (id_porta) REFERENCES porte_applicative(id),
	CONSTRAINT pk_pa_authz_properties PRIMARY KEY (id)
);

-- index
CREATE INDEX INDEX_PA_AUTHZ_PROP ON pa_authz_properties (id_porta);
CREATE TRIGGER trg_pa_authz_properties
BEFORE
insert on pa_authz_properties
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_pa_authz_properties.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/



CREATE SEQUENCE seq_pa_authzc_properties MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE pa_authzc_properties
(
	id_porta NUMBER NOT NULL,
	nome VARCHAR2(255) NOT NULL,
	valore VARCHAR2(255) NOT NULL,
	enc_value CLOB,
	-- fk/pk columns
	id NUMBER NOT NULL,
	-- unique constraints
	CONSTRAINT uniq_pa_authzc_props_1 UNIQUE (id_porta,nome),
	-- fk/pk keys constraints
	CONSTRAINT fk_pa_authzc_properties_1 FOREIGN KEY (id_porta) REFERENCES porte_applicative(id),
	CONSTRAINT pk_pa_authzc_properties PRIMARY KEY (id)
);

-- index
CREATE INDEX INDEX_PA_AUTHZC_PROP ON pa_authzc_properties (id_porta);
CREATE TRIGGER trg_pa_authzc_properties
BEFORE
insert on pa_authzc_properties
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_pa_authzc_properties.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/



CREATE SEQUENCE seq_pa_ct_properties MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE pa_ct_properties
(
	id_porta NUMBER NOT NULL,
	nome VARCHAR2(255) NOT NULL,
	valore VARCHAR2(255) NOT NULL,
	-- fk/pk columns
	id NUMBER NOT NULL,
	-- unique constraints
	CONSTRAINT uniq_pa_ct_props_1 UNIQUE (id_porta,nome),
	-- fk/pk keys constraints
	CONSTRAINT fk_pa_ct_properties_1 FOREIGN KEY (id_porta) REFERENCES porte_applicative(id),
	CONSTRAINT pk_pa_ct_properties PRIMARY KEY (id)
);

-- index
CREATE INDEX idx_pa_ct_props_1 ON pa_ct_properties (valore);
CREATE INDEX INDEX_PA_CT_PROP ON pa_ct_properties (id_porta);
CREATE TRIGGER trg_pa_ct_properties
BEFORE
insert on pa_ct_properties
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_pa_ct_properties.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/



CREATE SEQUENCE seq_pa_properties MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE pa_properties
(
	id_porta NUMBER NOT NULL,
	nome VARCHAR2(255) NOT NULL,
	valore VARCHAR2(255) NOT NULL,
	enc_value CLOB,
	-- fk/pk columns
	id NUMBER NOT NULL,
	-- unique constraints
	CONSTRAINT uniq_pa_properties_1 UNIQUE (id_porta,nome),
	-- fk/pk keys constraints
	CONSTRAINT fk_pa_properties_1 FOREIGN KEY (id_porta) REFERENCES porte_applicative(id),
	CONSTRAINT pk_pa_properties PRIMARY KEY (id)
);

-- index
CREATE INDEX INDEX_PA_PROP ON pa_properties (id_porta);
CREATE TRIGGER trg_pa_properties
BEFORE
insert on pa_properties
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_pa_properties.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/



CREATE SEQUENCE seq_pa_mtom_request MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE pa_mtom_request
(
	id_porta NUMBER NOT NULL,
	nome VARCHAR2(255) NOT NULL,
	pattern CLOB NOT NULL,
	content_type VARCHAR2(255),
	required NUMBER NOT NULL,
	-- fk/pk columns
	id NUMBER NOT NULL,
	-- fk/pk keys constraints
	CONSTRAINT fk_pa_mtom_request_1 FOREIGN KEY (id_porta) REFERENCES porte_applicative(id),
	CONSTRAINT pk_pa_mtom_request PRIMARY KEY (id)
);

-- index
CREATE INDEX INDEX_PA_MTOMTREQ ON pa_mtom_request (id_porta);
CREATE TRIGGER trg_pa_mtom_request
BEFORE
insert on pa_mtom_request
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_pa_mtom_request.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/



CREATE SEQUENCE seq_pa_mtom_response MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE pa_mtom_response
(
	id_porta NUMBER NOT NULL,
	nome VARCHAR2(255) NOT NULL,
	pattern CLOB NOT NULL,
	content_type VARCHAR2(255),
	required NUMBER NOT NULL,
	-- fk/pk columns
	id NUMBER NOT NULL,
	-- fk/pk keys constraints
	CONSTRAINT fk_pa_mtom_response_1 FOREIGN KEY (id_porta) REFERENCES porte_applicative(id),
	CONSTRAINT pk_pa_mtom_response PRIMARY KEY (id)
);

-- index
CREATE INDEX INDEX_PA_MTOMTRES ON pa_mtom_response (id_porta);
CREATE TRIGGER trg_pa_mtom_response
BEFORE
insert on pa_mtom_response
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_pa_mtom_response.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/



CREATE SEQUENCE seq_pa_security_request MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE pa_security_request
(
	id_porta NUMBER NOT NULL,
	nome VARCHAR2(255) NOT NULL,
	valore CLOB NOT NULL,
	enc_value CLOB,
	-- fk/pk columns
	id NUMBER NOT NULL,
	-- fk/pk keys constraints
	CONSTRAINT fk_pa_security_request_1 FOREIGN KEY (id_porta) REFERENCES porte_applicative(id),
	CONSTRAINT pk_pa_security_request PRIMARY KEY (id)
);

-- index
CREATE INDEX INDEX_PA_WSSREQ ON pa_security_request (id_porta);
CREATE TRIGGER trg_pa_security_request
BEFORE
insert on pa_security_request
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_pa_security_request.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/



CREATE SEQUENCE seq_pa_security_response MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE pa_security_response
(
	id_porta NUMBER NOT NULL,
	nome VARCHAR2(255) NOT NULL,
	valore CLOB NOT NULL,
	enc_value CLOB,
	-- fk/pk columns
	id NUMBER NOT NULL,
	-- fk/pk keys constraints
	CONSTRAINT fk_pa_security_response_1 FOREIGN KEY (id_porta) REFERENCES porte_applicative(id),
	CONSTRAINT pk_pa_security_response PRIMARY KEY (id)
);

-- index
CREATE INDEX INDEX_PA_WSSRES ON pa_security_response (id_porta);
CREATE TRIGGER trg_pa_security_response
BEFORE
insert on pa_security_response
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_pa_security_response.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/



CREATE SEQUENCE seq_pa_correlazione MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE pa_correlazione
(
	id_porta NUMBER NOT NULL,
	nome_elemento VARCHAR2(255),
	-- modalita di scelta user input, content-based, url-based, disabilitato
	mode_correlazione VARCHAR2(255),
	-- pattern utilizzato solo per content/url based
	pattern CLOB,
	-- blocca/accetta
	identificazione_fallita VARCHAR2(255),
	-- fk/pk columns
	id NUMBER NOT NULL,
	-- fk/pk keys constraints
	CONSTRAINT fk_pa_correlazione_1 FOREIGN KEY (id_porta) REFERENCES porte_applicative(id),
	CONSTRAINT pk_pa_correlazione PRIMARY KEY (id)
);

-- index
CREATE INDEX INDEX_PA_CORR_REQ ON pa_correlazione (id_porta);
CREATE TRIGGER trg_pa_correlazione
BEFORE
insert on pa_correlazione
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_pa_correlazione.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/



CREATE SEQUENCE seq_pa_correlazione_risposta MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE pa_correlazione_risposta
(
	id_porta NUMBER NOT NULL,
	nome_elemento VARCHAR2(255),
	-- modalita di scelta user input, content-based, url-based, disabilitato
	mode_correlazione VARCHAR2(255),
	-- pattern utilizzato solo per content/url based
	pattern CLOB,
	-- blocca/accetta
	identificazione_fallita VARCHAR2(255),
	-- fk/pk columns
	id NUMBER NOT NULL,
	-- fk/pk keys constraints
	CONSTRAINT fk_pa_correlazione_risposta_1 FOREIGN KEY (id_porta) REFERENCES porte_applicative(id),
	CONSTRAINT pk_pa_correlazione_risposta PRIMARY KEY (id)
);

-- index
CREATE INDEX INDEX_PA_CORR_RES ON pa_correlazione_risposta (id_porta);
CREATE TRIGGER trg_pa_correlazione_risposta
BEFORE
insert on pa_correlazione_risposta
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_pa_correlazione_risposta.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/



CREATE SEQUENCE seq_pa_ruoli MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE pa_ruoli
(
	id_porta NUMBER NOT NULL,
	ruolo VARCHAR2(255) NOT NULL,
	-- fk/pk columns
	id NUMBER NOT NULL,
	-- unique constraints
	CONSTRAINT unique_pa_ruoli_1 UNIQUE (id_porta,ruolo),
	-- fk/pk keys constraints
	CONSTRAINT fk_pa_ruoli_1 FOREIGN KEY (id_porta) REFERENCES porte_applicative(id),
	CONSTRAINT pk_pa_ruoli PRIMARY KEY (id)
);

CREATE TRIGGER trg_pa_ruoli
BEFORE
insert on pa_ruoli
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_pa_ruoli.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/



CREATE SEQUENCE seq_pa_scope MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE pa_scope
(
	id_porta NUMBER NOT NULL,
	scope VARCHAR2(255) NOT NULL,
	-- fk/pk columns
	id NUMBER NOT NULL,
	-- unique constraints
	CONSTRAINT unique_pa_scope_1 UNIQUE (id_porta,scope),
	-- fk/pk keys constraints
	CONSTRAINT fk_pa_scope_1 FOREIGN KEY (id_porta) REFERENCES porte_applicative(id),
	CONSTRAINT pk_pa_scope PRIMARY KEY (id)
);

CREATE TRIGGER trg_pa_scope
BEFORE
insert on pa_scope
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_pa_scope.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/



CREATE SEQUENCE seq_pa_soggetti MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE pa_soggetti
(
	id_porta NUMBER NOT NULL,
	tipo_soggetto VARCHAR2(255) NOT NULL,
	nome_soggetto VARCHAR2(255) NOT NULL,
	-- fk/pk columns
	id NUMBER NOT NULL,
	-- unique constraints
	CONSTRAINT unique_pa_soggetti_1 UNIQUE (id_porta,tipo_soggetto,nome_soggetto),
	-- fk/pk keys constraints
	CONSTRAINT fk_pa_soggetti_1 FOREIGN KEY (id_porta) REFERENCES porte_applicative(id),
	CONSTRAINT pk_pa_soggetti PRIMARY KEY (id)
);

CREATE TRIGGER trg_pa_soggetti
BEFORE
insert on pa_soggetti
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_pa_soggetti.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/



CREATE SEQUENCE seq_porte_applicative_sa_auth MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE porte_applicative_sa_auth
(
	id_porta NUMBER NOT NULL,
	id_servizio_applicativo NUMBER NOT NULL,
	-- fk/pk columns
	id NUMBER NOT NULL,
	-- unique constraints
	CONSTRAINT uniq_pa_sa_auth_1 UNIQUE (id_porta,id_servizio_applicativo),
	-- fk/pk keys constraints
	CONSTRAINT fk_porte_applicative_sa_auth_1 FOREIGN KEY (id_servizio_applicativo) REFERENCES servizi_applicativi(id),
	CONSTRAINT fk_porte_applicative_sa_auth_2 FOREIGN KEY (id_porta) REFERENCES porte_applicative(id),
	CONSTRAINT pk_porte_applicative_sa_auth PRIMARY KEY (id)
);

-- index
CREATE INDEX INDEX_PA_SA_AUTH ON porte_applicative_sa_auth (id_porta);
CREATE TRIGGER trg_porte_applicative_sa_auth
BEFORE
insert on porte_applicative_sa_auth
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_porte_applicative_sa_auth.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/



CREATE SEQUENCE seq_pa_azioni MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE pa_azioni
(
	id_porta NUMBER NOT NULL,
	azione VARCHAR2(255) NOT NULL,
	-- fk/pk columns
	id NUMBER NOT NULL,
	-- unique constraints
	CONSTRAINT unique_pa_azioni_1 UNIQUE (id_porta,azione),
	-- fk/pk keys constraints
	CONSTRAINT fk_pa_azioni_1 FOREIGN KEY (id_porta) REFERENCES porte_applicative(id),
	CONSTRAINT pk_pa_azioni PRIMARY KEY (id)
);

CREATE TRIGGER trg_pa_azioni
BEFORE
insert on pa_azioni
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_pa_azioni.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/



CREATE SEQUENCE seq_pa_cache_regole MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE pa_cache_regole
(
	id_porta NUMBER NOT NULL,
	status_min NUMBER,
	status_max NUMBER,
	fault NUMBER,
	cache_seconds NUMBER,
	-- fk/pk columns
	id NUMBER NOT NULL,
	-- fk/pk keys constraints
	CONSTRAINT fk_pa_cache_regole_1 FOREIGN KEY (id_porta) REFERENCES porte_applicative(id),
	CONSTRAINT pk_pa_cache_regole PRIMARY KEY (id)
);

-- index
CREATE INDEX INDEX_PA_CACHE ON pa_cache_regole (id_porta);

ALTER TABLE pa_cache_regole MODIFY fault DEFAULT 0;

CREATE TRIGGER trg_pa_cache_regole
BEFORE
insert on pa_cache_regole
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_pa_cache_regole.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/



CREATE SEQUENCE seq_pa_transform MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE pa_transform
(
	id_porta NUMBER NOT NULL,
	nome VARCHAR2(255) NOT NULL,
	posizione NUMBER NOT NULL,
	stato VARCHAR2(255),
	applicabilita_azioni CLOB,
	applicabilita_ct CLOB,
	applicabilita_pattern CLOB,
	applicabilita_connettori CLOB,
	req_conversione_enabled NUMBER NOT NULL,
	req_conversione_tipo VARCHAR2(255),
	req_conversione_template BLOB,
	req_content_type VARCHAR2(255),
	rest_transformation NUMBER NOT NULL,
	rest_method VARCHAR2(255),
	rest_path VARCHAR2(4000),
	soap_transformation NUMBER NOT NULL,
	soap_version VARCHAR2(255),
	soap_action VARCHAR2(255),
	soap_envelope NUMBER NOT NULL,
	soap_envelope_as_attach NUMBER NOT NULL,
	soap_envelope_tipo VARCHAR2(255),
	soap_envelope_template BLOB,
	-- fk/pk columns
	id NUMBER NOT NULL,
	-- unique constraints
	CONSTRAINT unique_pa_transform_1 UNIQUE (id_porta,nome),
	CONSTRAINT unique_pa_transform_2 UNIQUE (id_porta,posizione),
	-- fk/pk keys constraints
	CONSTRAINT fk_pa_transform_1 FOREIGN KEY (id_porta) REFERENCES porte_applicative(id),
	CONSTRAINT pk_pa_transform PRIMARY KEY (id)
);


ALTER TABLE pa_transform MODIFY req_conversione_enabled DEFAULT 0;
ALTER TABLE pa_transform MODIFY rest_transformation DEFAULT 0;
ALTER TABLE pa_transform MODIFY soap_transformation DEFAULT 0;
ALTER TABLE pa_transform MODIFY soap_envelope DEFAULT 0;
ALTER TABLE pa_transform MODIFY soap_envelope_as_attach DEFAULT 0;

CREATE TRIGGER trg_pa_transform
BEFORE
insert on pa_transform
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_pa_transform.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/



CREATE SEQUENCE seq_pa_transform_soggetti MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE pa_transform_soggetti
(
	id_trasformazione NUMBER NOT NULL,
	tipo_soggetto VARCHAR2(255) NOT NULL,
	nome_soggetto VARCHAR2(255) NOT NULL,
	-- fk/pk columns
	id NUMBER NOT NULL,
	-- unique constraints
	CONSTRAINT unique_pa_transform_soggetti_1 UNIQUE (id_trasformazione,tipo_soggetto,nome_soggetto),
	-- fk/pk keys constraints
	CONSTRAINT fk_pa_transform_soggetti_1 FOREIGN KEY (id_trasformazione) REFERENCES pa_transform(id),
	CONSTRAINT pk_pa_transform_soggetti PRIMARY KEY (id)
);

CREATE TRIGGER trg_pa_transform_soggetti
BEFORE
insert on pa_transform_soggetti
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_pa_transform_soggetti.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/



CREATE SEQUENCE seq_pa_transform_sa MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE pa_transform_sa
(
	id_trasformazione NUMBER NOT NULL,
	id_servizio_applicativo NUMBER NOT NULL,
	-- fk/pk columns
	id NUMBER NOT NULL,
	-- unique constraints
	CONSTRAINT unique_pa_transform_sa_1 UNIQUE (id_trasformazione,id_servizio_applicativo),
	-- fk/pk keys constraints
	CONSTRAINT fk_pa_transform_sa_1 FOREIGN KEY (id_servizio_applicativo) REFERENCES servizi_applicativi(id),
	CONSTRAINT fk_pa_transform_sa_2 FOREIGN KEY (id_trasformazione) REFERENCES pa_transform(id),
	CONSTRAINT pk_pa_transform_sa PRIMARY KEY (id)
);

CREATE TRIGGER trg_pa_transform_sa
BEFORE
insert on pa_transform_sa
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_pa_transform_sa.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/



CREATE SEQUENCE seq_pa_transform_hdr MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE pa_transform_hdr
(
	id_trasformazione NUMBER NOT NULL,
	tipo VARCHAR2(255) NOT NULL,
	nome VARCHAR2(255) NOT NULL,
	valore CLOB,
	identificazione_fallita VARCHAR2(255),
	-- fk/pk columns
	id NUMBER NOT NULL,
	-- fk/pk keys constraints
	CONSTRAINT fk_pa_transform_hdr_1 FOREIGN KEY (id_trasformazione) REFERENCES pa_transform(id),
	CONSTRAINT pk_pa_transform_hdr PRIMARY KEY (id)
);

-- index
CREATE INDEX idx_pa_trasf_hdr_1 ON pa_transform_hdr (id_trasformazione);
CREATE TRIGGER trg_pa_transform_hdr
BEFORE
insert on pa_transform_hdr
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_pa_transform_hdr.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/



CREATE SEQUENCE seq_pa_transform_url MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE pa_transform_url
(
	id_trasformazione NUMBER NOT NULL,
	tipo VARCHAR2(255) NOT NULL,
	nome VARCHAR2(255) NOT NULL,
	valore CLOB,
	identificazione_fallita VARCHAR2(255),
	-- fk/pk columns
	id NUMBER NOT NULL,
	-- fk/pk keys constraints
	CONSTRAINT fk_pa_transform_url_1 FOREIGN KEY (id_trasformazione) REFERENCES pa_transform(id),
	CONSTRAINT pk_pa_transform_url PRIMARY KEY (id)
);

-- index
CREATE INDEX idx_pa_trasf_url_1 ON pa_transform_url (id_trasformazione);
CREATE TRIGGER trg_pa_transform_url
BEFORE
insert on pa_transform_url
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_pa_transform_url.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/



CREATE SEQUENCE seq_pa_transform_risp MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE pa_transform_risp
(
	id_trasformazione NUMBER NOT NULL,
	nome VARCHAR2(255) NOT NULL,
	posizione NUMBER NOT NULL,
	applicabilita_status_min NUMBER,
	applicabilita_status_max NUMBER,
	applicabilita_ct CLOB,
	applicabilita_pattern CLOB,
	conversione_enabled NUMBER NOT NULL,
	conversione_tipo VARCHAR2(255),
	conversione_template BLOB,
	content_type VARCHAR2(255),
	return_code VARCHAR2(255),
	soap_envelope NUMBER NOT NULL,
	soap_envelope_as_attach NUMBER NOT NULL,
	soap_envelope_tipo VARCHAR2(255),
	soap_envelope_template BLOB,
	-- fk/pk columns
	id NUMBER NOT NULL,
	-- unique constraints
	CONSTRAINT uniq_pa_trasf_resp_1 UNIQUE (id_trasformazione,nome),
	CONSTRAINT uniq_pa_trasf_resp_2 UNIQUE (id_trasformazione,posizione),
	-- fk/pk keys constraints
	CONSTRAINT fk_pa_transform_risp_1 FOREIGN KEY (id_trasformazione) REFERENCES pa_transform(id),
	CONSTRAINT pk_pa_transform_risp PRIMARY KEY (id)
);


ALTER TABLE pa_transform_risp MODIFY conversione_enabled DEFAULT 0;
ALTER TABLE pa_transform_risp MODIFY soap_envelope DEFAULT 0;
ALTER TABLE pa_transform_risp MODIFY soap_envelope_as_attach DEFAULT 0;

CREATE TRIGGER trg_pa_transform_risp
BEFORE
insert on pa_transform_risp
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_pa_transform_risp.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/



CREATE SEQUENCE seq_pa_transform_risp_hdr MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE pa_transform_risp_hdr
(
	id_transform_risp NUMBER NOT NULL,
	tipo VARCHAR2(255) NOT NULL,
	nome VARCHAR2(255) NOT NULL,
	valore CLOB,
	identificazione_fallita VARCHAR2(255),
	-- fk/pk columns
	id NUMBER NOT NULL,
	-- fk/pk keys constraints
	CONSTRAINT fk_pa_transform_risp_hdr_1 FOREIGN KEY (id_transform_risp) REFERENCES pa_transform_risp(id),
	CONSTRAINT pk_pa_transform_risp_hdr PRIMARY KEY (id)
);

-- index
CREATE INDEX idx_pa_trasf_hdr_resp_1 ON pa_transform_risp_hdr (id_transform_risp);
CREATE TRIGGER trg_pa_transform_risp_hdr
BEFORE
insert on pa_transform_risp_hdr
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_pa_transform_risp_hdr.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/



CREATE SEQUENCE seq_pa_handlers MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE pa_handlers
(
	id_porta NUMBER NOT NULL,
	tipologia VARCHAR2(255) NOT NULL,
	tipo VARCHAR2(255) NOT NULL,
	posizione NUMBER NOT NULL,
	stato VARCHAR2(255),
	-- fk/pk columns
	id NUMBER NOT NULL,
	-- unique constraints
	CONSTRAINT unique_pa_handlers_1 UNIQUE (id_porta,tipologia,tipo),
	-- fk/pk keys constraints
	CONSTRAINT fk_pa_handlers_1 FOREIGN KEY (id_porta) REFERENCES porte_applicative(id),
	CONSTRAINT pk_pa_handlers PRIMARY KEY (id)
);

CREATE TRIGGER trg_pa_handlers
BEFORE
insert on pa_handlers
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_pa_handlers.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/



CREATE SEQUENCE seq_pa_aa MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE pa_aa
(
	id_porta NUMBER NOT NULL,
	nome VARCHAR2(255) NOT NULL,
	attributi CLOB,
	-- fk/pk columns
	id NUMBER NOT NULL,
	-- unique constraints
	CONSTRAINT unique_pa_aa_1 UNIQUE (id_porta,nome),
	-- fk/pk keys constraints
	CONSTRAINT fk_pa_aa_1 FOREIGN KEY (id_porta) REFERENCES porte_applicative(id),
	CONSTRAINT pk_pa_aa PRIMARY KEY (id)
);

CREATE TRIGGER trg_pa_aa
BEFORE
insert on pa_aa
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_pa_aa.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/



CREATE SEQUENCE seq_pa_token_sa MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE pa_token_sa
(
	id_porta NUMBER NOT NULL,
	id_servizio_applicativo NUMBER NOT NULL,
	-- fk/pk columns
	id NUMBER NOT NULL,
	-- unique constraints
	CONSTRAINT uniq_pa_token_sa_1 UNIQUE (id_porta,id_servizio_applicativo),
	-- fk/pk keys constraints
	CONSTRAINT fk_pa_token_sa_1 FOREIGN KEY (id_servizio_applicativo) REFERENCES servizi_applicativi(id),
	CONSTRAINT fk_pa_token_sa_2 FOREIGN KEY (id_porta) REFERENCES porte_applicative(id),
	CONSTRAINT pk_pa_token_sa PRIMARY KEY (id)
);

-- index
CREATE INDEX INDEX_PA_TOKEN_SA ON pa_token_sa (id_porta);
CREATE TRIGGER trg_pa_token_sa
BEFORE
insert on pa_token_sa
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_pa_token_sa.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/



CREATE SEQUENCE seq_pa_token_ruoli MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE pa_token_ruoli
(
	id_porta NUMBER NOT NULL,
	ruolo VARCHAR2(255) NOT NULL,
	-- fk/pk columns
	id NUMBER NOT NULL,
	-- unique constraints
	CONSTRAINT unique_pa_token_ruoli_1 UNIQUE (id_porta,ruolo),
	-- fk/pk keys constraints
	CONSTRAINT fk_pa_token_ruoli_1 FOREIGN KEY (id_porta) REFERENCES porte_applicative(id),
	CONSTRAINT pk_pa_token_ruoli PRIMARY KEY (id)
);

CREATE TRIGGER trg_pa_token_ruoli
BEFORE
insert on pa_token_ruoli
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_pa_token_ruoli.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/




-- govway
		  		
-- **** Mapping con Porte ****

CREATE SEQUENCE seq_mapping_fruizione_pd MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE mapping_fruizione_pd
(
	id_fruizione NUMBER NOT NULL,
	id_porta NUMBER NOT NULL,
	is_default NUMBER NOT NULL,
	nome VARCHAR2(255) NOT NULL,
	descrizione VARCHAR2(255) NOT NULL,
	-- fk/pk columns
	id NUMBER NOT NULL,
	-- fk/pk keys constraints
	CONSTRAINT fk_mapping_fruizione_pd_1 FOREIGN KEY (id_fruizione) REFERENCES servizi_fruitori(id),
	CONSTRAINT fk_mapping_fruizione_pd_2 FOREIGN KEY (id_porta) REFERENCES porte_delegate(id),
	CONSTRAINT pk_mapping_fruizione_pd PRIMARY KEY (id)
);

-- index
CREATE INDEX index_mapping_fruizione_pd_1 ON mapping_fruizione_pd (id_fruizione);
CREATE INDEX index_mapping_fruizione_pd_2 ON mapping_fruizione_pd (id_porta);
CREATE TRIGGER trg_mapping_fruizione_pd
BEFORE
insert on mapping_fruizione_pd
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_mapping_fruizione_pd.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/



CREATE SEQUENCE seq_mapping_erogazione_pa MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE mapping_erogazione_pa
(
	id_erogazione NUMBER NOT NULL,
	id_porta NUMBER NOT NULL,
	is_default NUMBER NOT NULL,
	nome VARCHAR2(255) NOT NULL,
	descrizione VARCHAR2(255) NOT NULL,
	-- fk/pk columns
	id NUMBER NOT NULL,
	-- fk/pk keys constraints
	CONSTRAINT fk_mapping_erogazione_pa_1 FOREIGN KEY (id_erogazione) REFERENCES servizi(id),
	CONSTRAINT fk_mapping_erogazione_pa_2 FOREIGN KEY (id_porta) REFERENCES porte_applicative(id),
	CONSTRAINT pk_mapping_erogazione_pa PRIMARY KEY (id)
);

-- index
CREATE INDEX index_mapping_erogazione_pa_1 ON mapping_erogazione_pa (id_erogazione);
CREATE INDEX index_mapping_erogazione_pa_2 ON mapping_erogazione_pa (id_porta);
CREATE TRIGGER trg_mapping_erogazione_pa
BEFORE
insert on mapping_erogazione_pa
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_mapping_erogazione_pa.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/




-- govway
		  		
-- **** Audit Appenders ****

CREATE SEQUENCE seq_audit_operations MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 CYCLE;

CREATE TABLE audit_operations
(
	tipo_operazione VARCHAR2(255) NOT NULL,
	-- non definito in caso di LOGIN/LOGOUT
	tipo VARCHAR2(255),
	-- non definito in caso di LOGIN/LOGOUT
	object_id VARCHAR2(2000),
	object_old_id VARCHAR2(2000),
	utente VARCHAR2(255) NOT NULL,
	stato VARCHAR2(255) NOT NULL,
	-- Dettaglio oggetto in gestione (Rappresentazione tramite JSON o XML format)
	object_details CLOB,
	-- Class, serve eventualmente per riottenere l'oggetto dal dettaglio
	-- non definito in caso di LOGIN/LOGOUT
	object_class VARCHAR2(255),
	-- Eventuale messaggio di errore
	error CLOB,
	time_request TIMESTAMP NOT NULL,
	time_execute TIMESTAMP NOT NULL,
	-- fk/pk columns
	id NUMBER NOT NULL,
	-- check constraints
	CONSTRAINT chk_audit_operations_1 CHECK (tipo_operazione IN ('LOGIN','LOGOUT','ADD','CHANGE','DEL')),
	CONSTRAINT chk_audit_operations_2 CHECK (stato IN ('requesting','error','completed')),
	-- fk/pk keys constraints
	CONSTRAINT pk_audit_operations PRIMARY KEY (id)
);

-- index
CREATE INDEX audit_filter_time ON audit_operations (time_request DESC);
CREATE INDEX audit_object_id ON audit_operations (object_id);
CREATE INDEX audit_object_old_id ON audit_operations (object_old_id);
CREATE INDEX audit_filter ON audit_operations (tipo_operazione,tipo,object_id,utente,stato);
CREATE TRIGGER trg_audit_operations
BEFORE
insert on audit_operations
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_audit_operations.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/



CREATE SEQUENCE seq_audit_binaries MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 CYCLE;

CREATE TABLE audit_binaries
(
	binary_id VARCHAR2(255) NOT NULL,
	checksum NUMBER NOT NULL,
	id_audit_operation NUMBER NOT NULL,
	time_rec TIMESTAMP,
	-- fk/pk columns
	id NUMBER NOT NULL,
	-- unique constraints
	CONSTRAINT unique_audit_binaries_1 UNIQUE (binary_id,id_audit_operation),
	-- fk/pk keys constraints
	CONSTRAINT fk_audit_binaries_1 FOREIGN KEY (id_audit_operation) REFERENCES audit_operations(id),
	CONSTRAINT pk_audit_binaries PRIMARY KEY (id)
);


ALTER TABLE audit_binaries MODIFY time_rec DEFAULT CURRENT_TIMESTAMP;

CREATE TRIGGER trg_audit_binaries
BEFORE
insert on audit_binaries
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_audit_binaries.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/




-- govway
		  		
-- **** Audit Configuration ****

CREATE SEQUENCE seq_audit_conf MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE audit_conf
(
	audit_engine NUMBER NOT NULL,
	enabled NUMBER NOT NULL,
	dump NUMBER NOT NULL,
	dump_format VARCHAR2(255) NOT NULL,
	-- fk/pk columns
	id NUMBER NOT NULL,
	-- fk/pk keys constraints
	CONSTRAINT pk_audit_conf PRIMARY KEY (id)
);


ALTER TABLE audit_conf MODIFY audit_engine DEFAULT 0;
ALTER TABLE audit_conf MODIFY enabled DEFAULT 0;
ALTER TABLE audit_conf MODIFY dump DEFAULT 0;
ALTER TABLE audit_conf MODIFY dump_format DEFAULT 'JSON';

CREATE TRIGGER trg_audit_conf
BEFORE
insert on audit_conf
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_audit_conf.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/



CREATE SEQUENCE seq_audit_filters MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE audit_filters
(
	-- Filter
	username VARCHAR2(255),
	tipo_operazione VARCHAR2(255),
	tipo VARCHAR2(255),
	stato VARCHAR2(255),
	-- Filtri basati su contenuto utilizzabili solo se il dump della configurazione generale e' abilitato
	dump_search VARCHAR2(255),
	-- 1(espressione regolare)/0(semplice stringa da ricercare)
	dump_expr NUMBER NOT NULL,
	-- Action
	enabled NUMBER NOT NULL,
	dump NUMBER NOT NULL,
	-- Tempo di registrazione
	ora_registrazione TIMESTAMP,
	-- fk/pk columns
	id NUMBER NOT NULL,
	-- check constraints
	CONSTRAINT chk_audit_filters_1 CHECK (tipo_operazione IN ('ADD','CHANGE','DEL')),
	CONSTRAINT chk_audit_filters_2 CHECK (stato IN ('requesting','error','completed')),
	-- fk/pk keys constraints
	CONSTRAINT pk_audit_filters PRIMARY KEY (id)
);


ALTER TABLE audit_filters MODIFY dump_expr DEFAULT 0;
ALTER TABLE audit_filters MODIFY enabled DEFAULT 0;
ALTER TABLE audit_filters MODIFY dump DEFAULT 0;
ALTER TABLE audit_filters MODIFY ora_registrazione DEFAULT CURRENT_TIMESTAMP;

CREATE TRIGGER trg_audit_filters
BEFORE
insert on audit_filters
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_audit_filters.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/



CREATE SEQUENCE seq_audit_appender MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE audit_appender
(
	name VARCHAR2(255) NOT NULL,
	class VARCHAR2(255) NOT NULL,
	-- fk/pk columns
	id NUMBER NOT NULL,
	-- unique constraints
	CONSTRAINT unique_audit_appender_1 UNIQUE (name),
	-- fk/pk keys constraints
	CONSTRAINT pk_audit_appender PRIMARY KEY (id)
);

CREATE TRIGGER trg_audit_appender
BEFORE
insert on audit_appender
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_audit_appender.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/



CREATE SEQUENCE seq_audit_appender_prop MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE audit_appender_prop
(
	name VARCHAR2(255) NOT NULL,
	value VARCHAR2(255) NOT NULL,
	id_audit_appender NUMBER NOT NULL,
	-- fk/pk columns
	id NUMBER NOT NULL,
	-- fk/pk keys constraints
	CONSTRAINT fk_audit_appender_prop_1 FOREIGN KEY (id_audit_appender) REFERENCES audit_appender(id),
	CONSTRAINT pk_audit_appender_prop PRIMARY KEY (id)
);

CREATE TRIGGER trg_audit_appender_prop
BEFORE
insert on audit_appender_prop
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_audit_appender_prop.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/




-- govway
		  		
-- **** Operations ****

CREATE SEQUENCE seq_operations MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 CYCLE;

CREATE TABLE operations
(
	operation VARCHAR2(255) NOT NULL,
	tipo VARCHAR2(255) NOT NULL,
	superuser VARCHAR2(255) NOT NULL,
	hostname VARCHAR2(255) NOT NULL,
	status VARCHAR2(255) NOT NULL,
	details CLOB,
	timereq TIMESTAMP NOT NULL,
	timexecute TIMESTAMP NOT NULL,
	deleted NUMBER NOT NULL,
	-- fk/pk columns
	id NUMBER NOT NULL,
	-- fk/pk keys constraints
	CONSTRAINT pk_operations PRIMARY KEY (id)
);

-- index
CREATE INDEX operations_superuser ON operations (superuser);
CREATE INDEX operations_precedenti ON operations (id,deleted,hostname,timereq DESC);
CREATE INDEX operations_gdo ON operations (timereq DESC);

ALTER TABLE operations MODIFY deleted DEFAULT 0;

CREATE TRIGGER trg_operations
BEFORE
insert on operations
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_operations.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/



CREATE SEQUENCE seq_parameters MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 CYCLE;

CREATE TABLE parameters
(
	id_operations NUMBER NOT NULL,
	name VARCHAR2(255) NOT NULL,
	value VARCHAR2(255),
	-- fk/pk columns
	id NUMBER NOT NULL,
	-- fk/pk keys constraints
	CONSTRAINT fk_parameters_1 FOREIGN KEY (id_operations) REFERENCES operations(id),
	CONSTRAINT pk_parameters PRIMARY KEY (id)
);

-- index
CREATE INDEX parameters_index ON parameters (name,value);
CREATE TRIGGER trg_parameters
BEFORE
insert on parameters
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_parameters.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/




-- govway
		  		
-- **** Utenti ****

CREATE SEQUENCE seq_users MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE users
(
	login VARCHAR2(255) NOT NULL,
	password VARCHAR2(255) NOT NULL,
	data_password TIMESTAMP NOT NULL,
	check_data_password NUMBER NOT NULL,
	tipo_interfaccia VARCHAR2(255) NOT NULL,
	interfaccia_completa NUMBER,
	permessi VARCHAR2(255) NOT NULL,
	protocolli CLOB,
	protocollo_pddconsole VARCHAR2(255),
	protocollo_pddmonitor VARCHAR2(255),
	soggetto_pddconsole VARCHAR2(255),
	soggetto_pddmonitor VARCHAR2(255),
	soggetti_all NUMBER,
	servizi_all NUMBER,
	-- fk/pk columns
	id NUMBER NOT NULL,
	-- unique constraints
	CONSTRAINT unique_users_1 UNIQUE (login),
	-- fk/pk keys constraints
	CONSTRAINT pk_users PRIMARY KEY (id)
);


ALTER TABLE users MODIFY data_password DEFAULT CURRENT_TIMESTAMP;
ALTER TABLE users MODIFY check_data_password DEFAULT 1;

CREATE TRIGGER trg_users
BEFORE
insert on users
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_users.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/



CREATE SEQUENCE seq_users_stati MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE users_stati
(
	oggetto VARCHAR2(255) NOT NULL,
	stato CLOB NOT NULL,
	-- fk/pk columns
	id NUMBER NOT NULL,
	id_utente NUMBER NOT NULL,
	-- fk/pk keys constraints
	CONSTRAINT fk_users_stati_1 FOREIGN KEY (id_utente) REFERENCES users(id),
	CONSTRAINT pk_users_stati PRIMARY KEY (id)
);

CREATE TRIGGER trg_users_stati
BEFORE
insert on users_stati
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_users_stati.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/



CREATE SEQUENCE seq_users_password MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE users_password
(
	password VARCHAR2(255) NOT NULL,
	data_password TIMESTAMP NOT NULL,
	-- fk/pk columns
	id NUMBER NOT NULL,
	id_utente NUMBER NOT NULL,
	-- fk/pk keys constraints
	CONSTRAINT fk_users_password_1 FOREIGN KEY (id_utente) REFERENCES users(id),
	CONSTRAINT pk_users_password PRIMARY KEY (id)
);

CREATE TRIGGER trg_users_password
BEFORE
insert on users_password
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_users_password.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/



CREATE SEQUENCE seq_users_soggetti MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE users_soggetti
(
	id_utente NUMBER NOT NULL,
	id_soggetto NUMBER NOT NULL,
	-- fk/pk columns
	id NUMBER NOT NULL,
	-- unique constraints
	CONSTRAINT unique_users_soggetti_1 UNIQUE (id_utente,id_soggetto),
	-- fk/pk keys constraints
	CONSTRAINT fk_users_soggetti_1 FOREIGN KEY (id_soggetto) REFERENCES soggetti(id),
	CONSTRAINT fk_users_soggetti_2 FOREIGN KEY (id_utente) REFERENCES users(id),
	CONSTRAINT pk_users_soggetti PRIMARY KEY (id)
);

CREATE TRIGGER trg_users_soggetti
BEFORE
insert on users_soggetti
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_users_soggetti.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/



CREATE SEQUENCE seq_users_servizi MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE users_servizi
(
	id_utente NUMBER NOT NULL,
	id_servizio NUMBER NOT NULL,
	-- fk/pk columns
	id NUMBER NOT NULL,
	-- unique constraints
	CONSTRAINT unique_users_servizi_1 UNIQUE (id_utente,id_servizio),
	-- fk/pk keys constraints
	CONSTRAINT fk_users_servizi_1 FOREIGN KEY (id_servizio) REFERENCES servizi(id),
	CONSTRAINT fk_users_servizi_2 FOREIGN KEY (id_utente) REFERENCES users(id),
	CONSTRAINT pk_users_servizi PRIMARY KEY (id)
);

CREATE TRIGGER trg_users_servizi
BEFORE
insert on users_servizi
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_users_servizi.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/




-- govway
		  		
CREATE SEQUENCE seq_db_info_console MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE db_info_console
(
	major_version NUMBER NOT NULL,
	minor_version NUMBER NOT NULL,
	notes VARCHAR2(255) NOT NULL,
	-- fk/pk columns
	id NUMBER NOT NULL,
	-- fk/pk keys constraints
	CONSTRAINT pk_db_info_console PRIMARY KEY (id)
);

CREATE TRIGGER trg_db_info_console
BEFORE
insert on db_info_console
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_db_info_console.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/




-- govway
		  		
INSERT INTO OP2_SEMAPHORE (applicative_id) VALUES ('GestoreChaviPDND');
INSERT INTO OP2_SEMAPHORE (applicative_id) VALUES ('ServiziDigestParamsUpdate');



-- govway
			  		

INSERT INTO audit_appender (name,class) VALUES ('dbAppender','org.openspcoop2.web.lib.audit.appender.AuditDBAppender');
INSERT INTO audit_appender_prop (name,value,id_audit_appender) VALUES ('datasource','org.govway.datasource.console',(select id from audit_appender where name='dbAppender'));
INSERT INTO audit_appender_prop (name,value,id_audit_appender) VALUES ('tipoDatabase','oracle',(select id from audit_appender where name='dbAppender'));


-- govway
			  		

INSERT INTO audit_appender (name,class) VALUES ('log4jAppender','org.openspcoop2.web.lib.audit.appender.AuditLog4JAppender');
INSERT INTO audit_appender_prop (name,value,id_audit_appender) VALUES ('fileConfigurazione','console.audit.log4j2.properties',(select id from audit_appender where name='log4jAppender'));
INSERT INTO audit_appender_prop (name,value,id_audit_appender) VALUES ('nomeFileLoaderInstance','console_local.audit.log4j2.properties',(select id from audit_appender where name='log4jAppender'));
INSERT INTO audit_appender_prop (name,value,id_audit_appender) VALUES ('nomeProprietaLoaderInstance','OPENSPCOOP2_CONSOLE_AUDIT_LOG_PROPERTIES',(select id from audit_appender where name='log4jAppender'));
INSERT INTO audit_appender_prop (name,value,id_audit_appender) VALUES ('category','audit',(select id from audit_appender where name='log4jAppender'));
INSERT INTO audit_appender_prop (name,value,id_audit_appender) VALUES ('xml','true',(select id from audit_appender where name='log4jAppender'));


-- govway
			  		

INSERT INTO audit_conf (audit_engine,enabled,dump,dump_format) VALUES (1,1,0,'JSON');


-- govway
			  		
-- users
INSERT INTO users (login, password, tipo_interfaccia, permessi) VALUES ('amministratore',	'$1$.bquKJDS$yZ4EYC3354HqEjmSRL/sR0','STANDARD','SCMAU');


-- govway
			  		


INSERT INTO db_info_console (major_version,minor_version,notes) VALUES (3,3,'[3.3.18] Database della Console di Gestione di GovWay');

-- **** Configurazione ****

CREATE SEQUENCE seq_registri MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE registri
(
	nome VARCHAR2(255) NOT NULL,
	location VARCHAR2(255) NOT NULL,
	tipo VARCHAR2(255) NOT NULL,
	utente VARCHAR2(255),
	password VARCHAR2(255),
	-- fk/pk columns
	id NUMBER NOT NULL,
	-- unique constraints
	CONSTRAINT unique_registri_1 UNIQUE (nome),
	-- fk/pk keys constraints
	CONSTRAINT pk_registri PRIMARY KEY (id)
);

CREATE TRIGGER trg_registri
BEFORE
insert on registri
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_registri.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/



CREATE SEQUENCE seq_routing MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE routing
(
	tipo VARCHAR2(255),
	nome VARCHAR2(255),
	-- registro/gateway
	tiporotta VARCHAR2(255) NOT NULL,
	tiposoggrotta VARCHAR2(255),
	nomesoggrotta VARCHAR2(255),
	-- foreign key per i registri(id)
	registrorotta NUMBER,
	is_default NUMBER NOT NULL,
	-- fk/pk columns
	id NUMBER NOT NULL,
	-- fk/pk keys constraints
	CONSTRAINT pk_routing PRIMARY KEY (id)
);

CREATE TRIGGER trg_routing
BEFORE
insert on routing
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_routing.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/



CREATE SEQUENCE seq_config_cache_regole MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE config_cache_regole
(
	status_min NUMBER,
	status_max NUMBER,
	fault NUMBER,
	cache_seconds NUMBER,
	-- fk/pk columns
	id NUMBER NOT NULL,
	-- fk/pk keys constraints
	CONSTRAINT pk_config_cache_regole PRIMARY KEY (id)
);


ALTER TABLE config_cache_regole MODIFY fault DEFAULT 0;

CREATE TRIGGER trg_config_cache_regole
BEFORE
insert on config_cache_regole
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_config_cache_regole.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/



CREATE SEQUENCE seq_configurazione MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE configurazione
(
	-- Cadenza inoltro Riscontri/BusteAsincrone
	cadenza_inoltro VARCHAR2(255) NOT NULL,
	-- Validazione Buste
	validazione_stato VARCHAR2(255),
	validazione_controllo VARCHAR2(255),
	validazione_profilo VARCHAR2(255),
	validazione_manifest VARCHAR2(255),
	-- Validazione Contenuti Applicativi
	-- abilitato/disabilitato/warningOnly
	validazione_contenuti_stato VARCHAR2(255),
	-- wsdl/interface/xsd
	validazione_contenuti_tipo VARCHAR2(255),
	-- abilitato/disabilitato
	validazione_contenuti_mtom VARCHAR2(255),
	-- Livello Log Messaggi Diagnostici
	msg_diag_severita VARCHAR2(255) NOT NULL,
	msg_diag_severita_log4j VARCHAR2(255) NOT NULL,
	-- Tracciamento Buste
	tracciamento_buste VARCHAR2(255),
	tracciamento_esiti_pd VARCHAR2(255),
	tracciamento_esiti VARCHAR2(255),
	-- Transazione
	transazioni_tempi_pd VARCHAR2(255),
	transazioni_tempi VARCHAR2(255),
	transazioni_token_pd VARCHAR2(255),
	transazioni_token VARCHAR2(255),
	-- Dump
	dump VARCHAR2(255),
	dump_bin_pd VARCHAR2(255),
	dump_bin_pa VARCHAR2(255),
	-- Autenticazione IntegrationManager
	auth_integration_manager VARCHAR2(255),
	-- Cache per l'accesso ai registri
	statocache VARCHAR2(255),
	dimensionecache VARCHAR2(255),
	algoritmocache VARCHAR2(255),
	idlecache VARCHAR2(255),
	lifecache VARCHAR2(255),
	-- Cache per l'accesso alla configurazione
	config_statocache VARCHAR2(255),
	config_dimensionecache VARCHAR2(255),
	config_algoritmocache VARCHAR2(255),
	config_idlecache VARCHAR2(255),
	config_lifecache VARCHAR2(255),
	-- Cache per l'accesso ai dati di autorizzazione
	auth_statocache VARCHAR2(255),
	auth_dimensionecache VARCHAR2(255),
	auth_algoritmocache VARCHAR2(255),
	auth_idlecache VARCHAR2(255),
	auth_lifecache VARCHAR2(255),
	-- Cache per l'accesso ai dati di autenticazione
	authn_statocache VARCHAR2(255),
	authn_dimensionecache VARCHAR2(255),
	authn_algoritmocache VARCHAR2(255),
	authn_idlecache VARCHAR2(255),
	authn_lifecache VARCHAR2(255),
	-- Cache per la gestione dei token
	token_statocache VARCHAR2(255),
	token_dimensionecache VARCHAR2(255),
	token_algoritmocache VARCHAR2(255),
	token_idlecache VARCHAR2(255),
	token_lifecache VARCHAR2(255),
	-- Cache per la gestione dei dati ottenuti da attribute authority
	aa_statocache VARCHAR2(255),
	aa_dimensionecache VARCHAR2(255),
	aa_algoritmocache VARCHAR2(255),
	aa_idlecache VARCHAR2(255),
	aa_lifecache VARCHAR2(255),
	-- Cache per la gestione dei keystore
	keystore_statocache VARCHAR2(255),
	keystore_dimensionecache VARCHAR2(255),
	keystore_algoritmocache VARCHAR2(255),
	keystore_idlecache VARCHAR2(255),
	keystore_lifecache VARCHAR2(255),
	keystore_crl_lifecache VARCHAR2(255),
	-- connessione su cui vengono inviate le risposte
	-- reply: connessione esistente (es. http reply)
	-- new: nuova connessione
	mod_risposta VARCHAR2(255),
	-- Gestione dell'indirizzo
	indirizzo_telematico VARCHAR2(255),
	-- Gestione Manifest
	gestione_manifest VARCHAR2(255),
	-- Routing Table
	routing_enabled VARCHAR2(255) NOT NULL,
	-- Gestione errore di default per la Porta di Dominio
	-- FOREIGN KEY (id_ge_cooperazione) REFERENCES gestione_errore(id) Nota: indica un eventuale tipologia di gestione dell'errore di default durante l'utilizzo di un connettore
	id_ge_cooperazione NUMBER,
	-- FOREIGN KEY (id_ge_integrazione) REFERENCES gestione_errore(id) Nota: indica un eventuale tipologia di gestione dell'errore di default durante l'utilizzo di un connettore
	id_ge_integrazione NUMBER,
	-- Gestione MultiTenant
	multitenant_stato VARCHAR2(255),
	multitenant_fruizioni VARCHAR2(255),
	multitenant_erogazioni VARCHAR2(255),
	-- Gestione CORS
	cors_stato VARCHAR2(255),
	cors_tipo VARCHAR2(255),
	cors_all_allow_origins VARCHAR2(255),
	cors_all_allow_methods VARCHAR2(255),
	cors_all_allow_headers VARCHAR2(255),
	cors_allow_credentials VARCHAR2(255),
	cors_allow_max_age NUMBER,
	cors_allow_max_age_seconds NUMBER,
	cors_allow_origins CLOB,
	cors_allow_headers CLOB,
	cors_allow_methods CLOB,
	cors_allow_expose_headers CLOB,
	-- Response caching
	response_cache_stato VARCHAR2(255),
	response_cache_seconds NUMBER,
	response_cache_max_msg_size NUMBER,
	response_cache_hash_url VARCHAR2(255),
	response_cache_hash_query VARCHAR2(255),
	response_cache_hash_query_list CLOB,
	response_cache_hash_headers VARCHAR2(255),
	response_cache_hash_hdr_list CLOB,
	response_cache_hash_payload VARCHAR2(255),
	response_cache_control_nocache NUMBER,
	response_cache_control_maxage NUMBER,
	response_cache_control_nostore NUMBER,
	-- Cache per il response caching
	response_cache_statocache VARCHAR2(255),
	response_cache_dimensionecache VARCHAR2(255),
	response_cache_algoritmocache VARCHAR2(255),
	response_cache_idlecache VARCHAR2(255),
	response_cache_lifecache VARCHAR2(255),
	-- Cache per la consegna agli applicativi
	consegna_statocache VARCHAR2(255),
	consegna_dimensionecache VARCHAR2(255),
	consegna_algoritmocache VARCHAR2(255),
	consegna_idlecache VARCHAR2(255),
	consegna_lifecache VARCHAR2(255),
	-- Gestione CORS
	canali_stato VARCHAR2(255),
	-- Cache per la gestione delle richieste
	dati_richieste_statocache VARCHAR2(255),
	dati_richieste_dimensionecache VARCHAR2(255),
	dati_richieste_algoritmocache VARCHAR2(255),
	dati_richieste_idlecache VARCHAR2(255),
	dati_richieste_lifecache VARCHAR2(255),
	-- fk/pk columns
	id NUMBER NOT NULL,
	-- fk/pk keys constraints
	CONSTRAINT pk_configurazione PRIMARY KEY (id)
);

CREATE TRIGGER trg_configurazione
BEFORE
insert on configurazione
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_configurazione.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/



-- **** URLInvocazione ****

CREATE SEQUENCE seq_config_url_invocazione MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE config_url_invocazione
(
	base_url VARCHAR2(255) NOT NULL,
	base_url_fruizione VARCHAR2(255),
	-- fk/pk columns
	id NUMBER NOT NULL,
	-- fk/pk keys constraints
	CONSTRAINT pk_config_url_invocazione PRIMARY KEY (id)
);

CREATE TRIGGER trg_config_url_invocazione
BEFORE
insert on config_url_invocazione
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_config_url_invocazione.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/



CREATE SEQUENCE seq_config_url_regole MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE config_url_regole
(
	nome VARCHAR2(255) NOT NULL,
	posizione NUMBER NOT NULL,
	stato VARCHAR2(255),
	descrizione CLOB,
	regexpr NUMBER NOT NULL,
	regola VARCHAR2(255) NOT NULL,
	contesto_esterno VARCHAR2(255) NOT NULL,
	base_url VARCHAR2(255),
	protocollo VARCHAR2(255),
	ruolo VARCHAR2(255),
	service_binding VARCHAR2(255),
	tipo_soggetto VARCHAR2(255),
	nome_soggetto VARCHAR2(255),
	-- fk/pk columns
	id NUMBER NOT NULL,
	-- unique constraints
	CONSTRAINT unique_config_url_regole_1 UNIQUE (nome),
	-- fk/pk keys constraints
	CONSTRAINT pk_config_url_regole PRIMARY KEY (id)
);

CREATE TRIGGER trg_config_url_regole
BEFORE
insert on config_url_regole
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_config_url_regole.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/



-- **** Messaggi diagnostici ****

CREATE SEQUENCE seq_msgdiag_appender MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE msgdiag_appender
(
	tipo VARCHAR2(255) NOT NULL,
	-- fk/pk columns
	id NUMBER NOT NULL,
	-- fk/pk keys constraints
	CONSTRAINT pk_msgdiag_appender PRIMARY KEY (id)
);

CREATE TRIGGER trg_msgdiag_appender
BEFORE
insert on msgdiag_appender
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_msgdiag_appender.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/



CREATE SEQUENCE seq_msgdiag_appender_prop MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE msgdiag_appender_prop
(
	id_appender NUMBER NOT NULL,
	nome VARCHAR2(255) NOT NULL,
	valore VARCHAR2(255) NOT NULL,
	-- fk/pk columns
	id NUMBER NOT NULL,
	-- unique constraints
	CONSTRAINT uniq_msgdiag_app_prop_1 UNIQUE (id_appender,nome,valore),
	-- fk/pk keys constraints
	CONSTRAINT fk_msgdiag_appender_prop_1 FOREIGN KEY (id_appender) REFERENCES msgdiag_appender(id),
	CONSTRAINT pk_msgdiag_appender_prop PRIMARY KEY (id)
);

CREATE TRIGGER trg_msgdiag_appender_prop
BEFORE
insert on msgdiag_appender_prop
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_msgdiag_appender_prop.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/



-- **** Tracciamento ****

CREATE SEQUENCE seq_tracce_appender MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE tracce_appender
(
	tipo VARCHAR2(255) NOT NULL,
	-- fk/pk columns
	id NUMBER NOT NULL,
	-- fk/pk keys constraints
	CONSTRAINT pk_tracce_appender PRIMARY KEY (id)
);

CREATE TRIGGER trg_tracce_appender
BEFORE
insert on tracce_appender
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_tracce_appender.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/



CREATE SEQUENCE seq_tracce_appender_prop MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE tracce_appender_prop
(
	id_appender NUMBER NOT NULL,
	nome VARCHAR2(255) NOT NULL,
	valore VARCHAR2(255) NOT NULL,
	-- fk/pk columns
	id NUMBER NOT NULL,
	-- unique constraints
	CONSTRAINT uniq_tracce_app_prop_1 UNIQUE (id_appender,nome,valore),
	-- fk/pk keys constraints
	CONSTRAINT fk_tracce_appender_prop_1 FOREIGN KEY (id_appender) REFERENCES tracce_appender(id),
	CONSTRAINT pk_tracce_appender_prop PRIMARY KEY (id)
);

CREATE TRIGGER trg_tracce_appender_prop
BEFORE
insert on tracce_appender_prop
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_tracce_appender_prop.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/



CREATE SEQUENCE seq_tracce_config MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE tracce_config
(
	proprietario VARCHAR2(255) NOT NULL,
	tipo VARCHAR2(255) NOT NULL,
	id_proprietario NUMBER NOT NULL,
	stato VARCHAR2(255),
	filtro_esiti VARCHAR2(255),
	request_in VARCHAR2(255),
	request_out VARCHAR2(255),
	response_out VARCHAR2(255),
	response_out_complete VARCHAR2(255),
	-- fk/pk columns
	id NUMBER NOT NULL,
	-- fk/pk keys constraints
	CONSTRAINT pk_tracce_config PRIMARY KEY (id)
);

-- index
CREATE INDEX index_tracce_config_1 ON tracce_config (proprietario,tipo);
CREATE TRIGGER trg_tracce_config
BEFORE
insert on tracce_config
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_tracce_config.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/



CREATE SEQUENCE seq_filetrace_config MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE filetrace_config
(
	proprietario VARCHAR2(255) NOT NULL,
	id_proprietario NUMBER NOT NULL,
	config VARCHAR2(255),
	dump_in_stato VARCHAR2(255),
	dump_in_stato_hdr VARCHAR2(255),
	dump_in_stato_body VARCHAR2(255),
	dump_out_stato VARCHAR2(255),
	dump_out_stato_hdr VARCHAR2(255),
	dump_out_stato_body VARCHAR2(255),
	-- fk/pk columns
	id NUMBER NOT NULL,
	-- fk/pk keys constraints
	CONSTRAINT pk_filetrace_config PRIMARY KEY (id)
);

-- index
CREATE INDEX index_filetrace_config_1 ON filetrace_config (proprietario);
CREATE TRIGGER trg_filetrace_config
BEFORE
insert on filetrace_config
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_filetrace_config.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/



-- **** Dump ****

CREATE SEQUENCE seq_dump_config MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE dump_config
(
	proprietario VARCHAR2(255) NOT NULL,
	id_proprietario NUMBER NOT NULL,
	dump_realtime VARCHAR2(255),
	id_richiesta_ingresso NUMBER NOT NULL,
	id_richiesta_uscita NUMBER NOT NULL,
	id_risposta_ingresso NUMBER NOT NULL,
	id_risposta_uscita NUMBER NOT NULL,
	-- fk/pk columns
	id NUMBER NOT NULL,
	-- fk/pk keys constraints
	CONSTRAINT pk_dump_config PRIMARY KEY (id)
);

-- index
CREATE INDEX index_dump_config_1 ON dump_config (proprietario);
CREATE TRIGGER trg_dump_config
BEFORE
insert on dump_config
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_dump_config.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/



CREATE SEQUENCE seq_dump_config_regola MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE dump_config_regola
(
	payload VARCHAR2(255) NOT NULL,
	payload_parsing VARCHAR2(255) NOT NULL,
	body VARCHAR2(255) NOT NULL,
	attachments VARCHAR2(255) NOT NULL,
	headers VARCHAR2(255) NOT NULL,
	-- fk/pk columns
	id NUMBER NOT NULL,
	-- fk/pk keys constraints
	CONSTRAINT pk_dump_config_regola PRIMARY KEY (id)
);

CREATE TRIGGER trg_dump_config_regola
BEFORE
insert on dump_config_regola
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_dump_config_regola.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/



CREATE SEQUENCE seq_dump_appender MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE dump_appender
(
	tipo VARCHAR2(255) NOT NULL,
	-- fk/pk columns
	id NUMBER NOT NULL,
	-- fk/pk keys constraints
	CONSTRAINT pk_dump_appender PRIMARY KEY (id)
);

CREATE TRIGGER trg_dump_appender
BEFORE
insert on dump_appender
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_dump_appender.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/



CREATE SEQUENCE seq_dump_appender_prop MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE dump_appender_prop
(
	id_appender NUMBER NOT NULL,
	nome VARCHAR2(255) NOT NULL,
	valore VARCHAR2(255) NOT NULL,
	-- fk/pk columns
	id NUMBER NOT NULL,
	-- unique constraints
	CONSTRAINT uniq_dump_app_prop_1 UNIQUE (id_appender,nome,valore),
	-- fk/pk keys constraints
	CONSTRAINT fk_dump_appender_prop_1 FOREIGN KEY (id_appender) REFERENCES dump_appender(id),
	CONSTRAINT pk_dump_appender_prop PRIMARY KEY (id)
);

CREATE TRIGGER trg_dump_appender_prop
BEFORE
insert on dump_appender_prop
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_dump_appender_prop.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/



-- **** Datasources dove reperire i messaggi diagnostici salvati ****

CREATE SEQUENCE seq_msgdiag_ds MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE msgdiag_ds
(
	nome VARCHAR2(255) NOT NULL,
	nome_jndi VARCHAR2(255) NOT NULL,
	tipo_database VARCHAR2(255) NOT NULL,
	-- fk/pk columns
	id NUMBER NOT NULL,
	-- unique constraints
	CONSTRAINT unique_msgdiag_ds_1 UNIQUE (nome),
	-- fk/pk keys constraints
	CONSTRAINT pk_msgdiag_ds PRIMARY KEY (id)
);

CREATE TRIGGER trg_msgdiag_ds
BEFORE
insert on msgdiag_ds
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_msgdiag_ds.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/



CREATE SEQUENCE seq_msgdiag_ds_prop MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE msgdiag_ds_prop
(
	id_prop NUMBER NOT NULL,
	nome VARCHAR2(255) NOT NULL,
	valore VARCHAR2(255) NOT NULL,
	-- fk/pk columns
	id NUMBER NOT NULL,
	-- unique constraints
	CONSTRAINT unique_msgdiag_ds_prop_1 UNIQUE (id_prop,nome,valore),
	-- fk/pk keys constraints
	CONSTRAINT fk_msgdiag_ds_prop_1 FOREIGN KEY (id_prop) REFERENCES msgdiag_ds(id),
	CONSTRAINT pk_msgdiag_ds_prop PRIMARY KEY (id)
);

CREATE TRIGGER trg_msgdiag_ds_prop
BEFORE
insert on msgdiag_ds_prop
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_msgdiag_ds_prop.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/



-- **** Datasources dove reperire le tracce salvate ****

CREATE SEQUENCE seq_tracce_ds MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE tracce_ds
(
	nome VARCHAR2(255) NOT NULL,
	nome_jndi VARCHAR2(255) NOT NULL,
	tipo_database VARCHAR2(255) NOT NULL,
	-- fk/pk columns
	id NUMBER NOT NULL,
	-- unique constraints
	CONSTRAINT unique_tracce_ds_1 UNIQUE (nome),
	-- fk/pk keys constraints
	CONSTRAINT pk_tracce_ds PRIMARY KEY (id)
);

CREATE TRIGGER trg_tracce_ds
BEFORE
insert on tracce_ds
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_tracce_ds.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/



CREATE SEQUENCE seq_tracce_ds_prop MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE tracce_ds_prop
(
	id_prop NUMBER NOT NULL,
	nome VARCHAR2(255) NOT NULL,
	valore VARCHAR2(255) NOT NULL,
	-- fk/pk columns
	id NUMBER NOT NULL,
	-- unique constraints
	CONSTRAINT unique_tracce_ds_prop_1 UNIQUE (id_prop,nome,valore),
	-- fk/pk keys constraints
	CONSTRAINT fk_tracce_ds_prop_1 FOREIGN KEY (id_prop) REFERENCES tracce_ds(id),
	CONSTRAINT pk_tracce_ds_prop PRIMARY KEY (id)
);

CREATE TRIGGER trg_tracce_ds_prop
BEFORE
insert on tracce_ds_prop
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_tracce_ds_prop.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/



-- **** Stato dei servizi attivi sulla Porta di Dominio ****

CREATE SEQUENCE seq_servizi_pdd MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE servizi_pdd
(
	componente VARCHAR2(255) NOT NULL,
	-- Stato dei servizi attivi sulla Porta di Dominio
	stato NUMBER,
	-- fk/pk columns
	id NUMBER NOT NULL,
	-- unique constraints
	CONSTRAINT unique_servizi_pdd_1 UNIQUE (componente),
	-- fk/pk keys constraints
	CONSTRAINT pk_servizi_pdd PRIMARY KEY (id)
);


ALTER TABLE servizi_pdd MODIFY stato DEFAULT 1;

CREATE TRIGGER trg_servizi_pdd
BEFORE
insert on servizi_pdd
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_servizi_pdd.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/



CREATE SEQUENCE seq_servizi_pdd_filtri MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE servizi_pdd_filtri
(
	id_servizio_pdd NUMBER NOT NULL,
	tipo_filtro VARCHAR2(255) NOT NULL,
	tipo_soggetto_fruitore VARCHAR2(255),
	soggetto_fruitore VARCHAR2(255),
	identificativo_porta_fruitore VARCHAR2(255),
	tipo_soggetto_erogatore VARCHAR2(255),
	soggetto_erogatore VARCHAR2(255),
	identificativo_porta_erogatore VARCHAR2(255),
	tipo_servizio VARCHAR2(255),
	servizio VARCHAR2(255),
	versione_servizio NUMBER,
	azione VARCHAR2(255),
	-- fk/pk columns
	id NUMBER NOT NULL,
	-- check constraints
	CONSTRAINT chk_servizi_pdd_filtri_1 CHECK (tipo_filtro IN ('abilitazione','disabilitazione')),
	-- fk/pk keys constraints
	CONSTRAINT fk_servizi_pdd_filtri_1 FOREIGN KEY (id_servizio_pdd) REFERENCES servizi_pdd(id),
	CONSTRAINT pk_servizi_pdd_filtri PRIMARY KEY (id)
);

CREATE TRIGGER trg_servizi_pdd_filtri
BEFORE
insert on servizi_pdd_filtri
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_servizi_pdd_filtri.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/



-- **** PddSystemProperties ****

CREATE SEQUENCE seq_pdd_sys_props MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE pdd_sys_props
(
	nome VARCHAR2(255) NOT NULL,
	valore VARCHAR2(255) NOT NULL,
	enc_value CLOB,
	-- fk/pk columns
	id NUMBER NOT NULL,
	-- unique constraints
	CONSTRAINT unique_pdd_sys_props_1 UNIQUE (nome,valore),
	-- fk/pk keys constraints
	CONSTRAINT pk_pdd_sys_props PRIMARY KEY (id)
);

CREATE TRIGGER trg_pdd_sys_props
BEFORE
insert on pdd_sys_props
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_pdd_sys_props.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/



-- **** Proprieta Generiche ****

CREATE SEQUENCE seq_generic_properties MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE generic_properties
(
	nome VARCHAR2(255) NOT NULL,
	descrizione VARCHAR2(4000),
	tipologia VARCHAR2(255) NOT NULL,
	tipo VARCHAR2(255) NOT NULL,
	utente_richiedente VARCHAR2(255),
	data_creazione TIMESTAMP,
	utente_ultima_modifica VARCHAR2(255),
	data_ultima_modifica TIMESTAMP,
	-- fk/pk columns
	id NUMBER NOT NULL,
	-- unique constraints
	CONSTRAINT unique_generic_properties_1 UNIQUE (tipologia,nome),
	-- fk/pk keys constraints
	CONSTRAINT pk_generic_properties PRIMARY KEY (id)
);

CREATE TRIGGER trg_generic_properties
BEFORE
insert on generic_properties
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_generic_properties.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/



CREATE SEQUENCE seq_generic_property MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE generic_property
(
	id_props NUMBER NOT NULL,
	nome VARCHAR2(255) NOT NULL,
	valore VARCHAR2(4000) NOT NULL,
	enc_value CLOB,
	-- fk/pk columns
	id NUMBER NOT NULL,
	-- unique constraints
	CONSTRAINT unique_generic_property_1 UNIQUE (id_props,nome),
	-- fk/pk keys constraints
	CONSTRAINT fk_generic_property_1 FOREIGN KEY (id_props) REFERENCES generic_properties(id),
	CONSTRAINT pk_generic_property PRIMARY KEY (id)
);

-- index
CREATE INDEX index_generic_property_1 ON generic_property (id_props);
CREATE TRIGGER trg_generic_property
BEFORE
insert on generic_property
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_generic_property.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/



-- **** Canali ****

CREATE SEQUENCE seq_canali_configurazione MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE canali_configurazione
(
	nome VARCHAR2(255) NOT NULL,
	descrizione VARCHAR2(255),
	canale_default NUMBER NOT NULL,
	-- fk/pk columns
	id NUMBER NOT NULL,
	-- unique constraints
	CONSTRAINT unique_canali_configurazione_1 UNIQUE (nome),
	-- fk/pk keys constraints
	CONSTRAINT pk_canali_configurazione PRIMARY KEY (id)
);

CREATE TRIGGER trg_canali_configurazione
BEFORE
insert on canali_configurazione
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_canali_configurazione.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/



CREATE SEQUENCE seq_canali_nodi MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE canali_nodi
(
	nome VARCHAR2(255) NOT NULL,
	descrizione VARCHAR2(255),
	canali VARCHAR2(4000) NOT NULL,
	-- fk/pk columns
	id NUMBER NOT NULL,
	-- unique constraints
	CONSTRAINT unique_canali_nodi_1 UNIQUE (nome),
	-- fk/pk keys constraints
	CONSTRAINT pk_canali_nodi PRIMARY KEY (id)
);

CREATE TRIGGER trg_canali_nodi
BEFORE
insert on canali_nodi
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_canali_nodi.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/



-- **** Regitro Plugins ****

CREATE SEQUENCE seq_registro_plugins MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE registro_plugins
(
	nome VARCHAR2(255) NOT NULL,
	posizione NUMBER NOT NULL,
	stato VARCHAR2(255),
	descrizione VARCHAR2(255),
	data TIMESTAMP NOT NULL,
	compatibilita CLOB,
	-- fk/pk columns
	id NUMBER NOT NULL,
	-- unique constraints
	CONSTRAINT unique_registro_plugins_1 UNIQUE (nome),
	-- fk/pk keys constraints
	CONSTRAINT pk_registro_plugins PRIMARY KEY (id)
);


ALTER TABLE registro_plugins MODIFY data DEFAULT CURRENT_TIMESTAMP;

CREATE TRIGGER trg_registro_plugins
BEFORE
insert on registro_plugins
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_registro_plugins.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/



CREATE SEQUENCE seq_registro_plug_jar MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE registro_plug_jar
(
	id_plugin NUMBER NOT NULL,
	nome VARCHAR2(255) NOT NULL,
	sorgente VARCHAR2(255) NOT NULL,
	contenuto BLOB,
	url VARCHAR2(4000),
	dir VARCHAR2(4000),
	data TIMESTAMP NOT NULL,
	-- fk/pk columns
	id NUMBER NOT NULL,
	-- unique constraints
	CONSTRAINT unique_registro_plug_jar_1 UNIQUE (id_plugin,nome),
	-- fk/pk keys constraints
	CONSTRAINT fk_registro_plug_jar_1 FOREIGN KEY (id_plugin) REFERENCES registro_plugins(id),
	CONSTRAINT pk_registro_plug_jar PRIMARY KEY (id)
);

-- index
CREATE INDEX index_registro_plug_jar_1 ON registro_plug_jar (id_plugin);

ALTER TABLE registro_plug_jar MODIFY data DEFAULT CURRENT_TIMESTAMP;

CREATE TRIGGER trg_registro_plug_jar
BEFORE
insert on registro_plug_jar
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_registro_plug_jar.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/



-- **** Handlers ****

CREATE SEQUENCE seq_config_handlers MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE config_handlers
(
	tipologia VARCHAR2(255) NOT NULL,
	tipo VARCHAR2(255) NOT NULL,
	posizione NUMBER NOT NULL,
	stato VARCHAR2(255),
	-- fk/pk columns
	id NUMBER NOT NULL,
	-- unique constraints
	CONSTRAINT unique_config_handlers_1 UNIQUE (tipologia,tipo),
	-- fk/pk keys constraints
	CONSTRAINT pk_config_handlers PRIMARY KEY (id)
);

CREATE TRIGGER trg_config_handlers
BEFORE
insert on config_handlers
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_config_handlers.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/



-- **** Nodi Runtime ****

CREATE SEQUENCE seq_nodi_runtime MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE nodi_runtime
(
	hostname VARCHAR2(3000) NOT NULL,
	gruppo VARCHAR2(3000) NOT NULL,
	data_registrazione TIMESTAMP NOT NULL,
	data_refresh TIMESTAMP NOT NULL,
	id_numerico NUMBER NOT NULL,
	-- fk/pk columns
	id NUMBER NOT NULL,
	-- unique constraints
	CONSTRAINT unique_nodi_runtime_1 UNIQUE (hostname),
	CONSTRAINT unique_nodi_runtime_2 UNIQUE (gruppo,id_numerico),
	-- fk/pk keys constraints
	CONSTRAINT pk_nodi_runtime PRIMARY KEY (id)
);

CREATE TRIGGER trg_nodi_runtime
BEFORE
insert on nodi_runtime
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_nodi_runtime.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/



CREATE SEQUENCE seq_nodi_runtime_operations MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE nodi_runtime_operations
(
	descrizione VARCHAR2(4000) NOT NULL,
	operazione VARCHAR2(4000) NOT NULL,
	data_registrazione TIMESTAMP NOT NULL,
	-- fk/pk columns
	id NUMBER NOT NULL,
	-- fk/pk keys constraints
	CONSTRAINT pk_nodi_runtime_operations PRIMARY KEY (id)
);

-- index
CREATE INDEX INDEX_NODI_OP ON nodi_runtime_operations (data_registrazione);
CREATE TRIGGER trg_nodi_runtime_operations
BEFORE
insert on nodi_runtime_operations
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_nodi_runtime_operations.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/



-- **** Keystore ****

CREATE SEQUENCE seq_remote_store MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE remote_store
(
	nome VARCHAR2(255) NOT NULL,
	data_aggiornamento TIMESTAMP NOT NULL,
	last_event VARCHAR2(4000),
	-- fk/pk columns
	id NUMBER NOT NULL,
	-- unique constraints
	CONSTRAINT unique_remote_store_1 UNIQUE (nome),
	-- fk/pk keys constraints
	CONSTRAINT pk_remote_store PRIMARY KEY (id)
);

CREATE TRIGGER trg_remote_store
BEFORE
insert on remote_store
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_remote_store.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/



CREATE SEQUENCE seq_remote_store_key MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE remote_store_key
(
	id_remote_store NUMBER NOT NULL,
	kid VARCHAR2(255) NOT NULL,
	content_key BLOB NOT NULL,
	data_registrazione TIMESTAMP NOT NULL,
	data_aggiornamento TIMESTAMP NOT NULL,
	client_id VARCHAR2(255),
	client_details CLOB,
	organization_details CLOB,
	client_data_aggiornamento TIMESTAMP,
	-- fk/pk columns
	id NUMBER NOT NULL,
	-- unique constraints
	CONSTRAINT unique_remote_store_key_1 UNIQUE (id_remote_store,kid),
	-- fk/pk keys constraints
	CONSTRAINT fk_remote_store_key_1 FOREIGN KEY (id_remote_store) REFERENCES remote_store(id),
	CONSTRAINT pk_remote_store_key PRIMARY KEY (id)
);

-- index
CREATE INDEX REMOTE_STORE_UPDATE ON remote_store_key (data_aggiornamento);
CREATE INDEX REMOTE_STORE_CREATE ON remote_store_key (data_registrazione);
CREATE TRIGGER trg_remote_store_key
BEFORE
insert on remote_store_key
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_remote_store_key.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/



-- **** Digests Servizi Params ****

CREATE SEQUENCE seq_servizi_digest_params MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE servizi_digest_params
(
	id_servizio NUMBER NOT NULL,
	serial_number NUMBER,
	data_registrazione TIMESTAMP NOT NULL,
	seed VARCHAR2(4000) NOT NULL,
	algorithm VARCHAR2(32) NOT NULL,
	lifetime NUMBER NOT NULL,
	-- fk/pk columns
	id NUMBER NOT NULL,
	-- fk/pk keys constraints
	CONSTRAINT pk_servizi_digest_params PRIMARY KEY (id)
);

-- index
CREATE INDEX INDEX_REF_SERVIZI ON servizi_digest_params (id_servizio);
CREATE TRIGGER trg_servizi_digest_params
BEFORE
insert on servizi_digest_params
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_servizi_digest_params.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/




-- govway
		  		
-- CONTROLLO TRAFFICO

CREATE SEQUENCE seq_ct_config MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE ct_config
(
	-- Numero Massimo Richieste Simultanee
	max_threads_enabled NUMBER NOT NULL,
	max_threads_warning_only NUMBER NOT NULL,
	max_threads NUMBER NOT NULL,
	max_threads_tipo_errore VARCHAR2(255) NOT NULL,
	max_threads_includi_errore NUMBER NOT NULL,
	-- Controllo della Congestione
	cc_enabled NUMBER NOT NULL,
	cc_threshold NUMBER,
	-- Tempi di Risposta Fruizione
	pd_connection_timeout NUMBER,
	pd_read_timeout NUMBER,
	pd_avg_time NUMBER,
	-- Tempi di Risposta Erogazione
	pa_connection_timeout NUMBER,
	pa_read_timeout NUMBER,
	pa_avg_time NUMBER,
	-- Rate Limiting
	rt_tipo_errore VARCHAR2(255) NOT NULL,
	rt_includi_errore NUMBER NOT NULL,
	-- Cache
	cache NUMBER NOT NULL,
	cache_size NUMBER,
	cache_algorithm VARCHAR2(255),
	cache_idle_time NUMBER,
	cache_life_time NUMBER,
	-- fk/pk columns
	id NUMBER NOT NULL,
	-- check constraints
	CONSTRAINT chk_ct_config_1 CHECK (cache_algorithm IN ('LRU','MRU')),
	-- fk/pk keys constraints
	CONSTRAINT pk_ct_config PRIMARY KEY (id)
);


ALTER TABLE ct_config MODIFY max_threads_enabled DEFAULT 1;
ALTER TABLE ct_config MODIFY max_threads_warning_only DEFAULT 0;
ALTER TABLE ct_config MODIFY max_threads_tipo_errore DEFAULT 'fault';
ALTER TABLE ct_config MODIFY max_threads_includi_errore DEFAULT 1;
ALTER TABLE ct_config MODIFY cc_enabled DEFAULT 0;
ALTER TABLE ct_config MODIFY rt_tipo_errore DEFAULT 'fault';
ALTER TABLE ct_config MODIFY rt_includi_errore DEFAULT 1;
ALTER TABLE ct_config MODIFY cache DEFAULT 1;

CREATE TRIGGER trg_ct_config
BEFORE
insert on ct_config
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_ct_config.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/



CREATE SEQUENCE seq_ct_rt_props MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE ct_rt_props
(
	rt_prop_name VARCHAR2(255) NOT NULL,
	rt_prop_value VARCHAR2(255) NOT NULL,
	-- fk/pk columns
	id NUMBER NOT NULL,
	-- unique constraints
	CONSTRAINT uniq_rt_prop_policy_1 UNIQUE (rt_prop_name),
	-- fk/pk keys constraints
	CONSTRAINT pk_ct_rt_props PRIMARY KEY (id)
);

-- index
CREATE INDEX idx_rt_prop_policy_1 ON ct_rt_props (rt_prop_value);
CREATE TRIGGER trg_ct_rt_props
BEFORE
insert on ct_rt_props
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_ct_rt_props.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/



CREATE SEQUENCE seq_ct_config_policy MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE ct_config_policy
(
	-- Dati Generali
	policy_id VARCHAR2(255) NOT NULL,
	policy_built_in NUMBER NOT NULL,
	rt_descrizione CLOB NOT NULL,
	rt_risorsa VARCHAR2(255) NOT NULL,
	-- Valori di Soglia
	rt_simultanee NUMBER NOT NULL,
	rt_valore NUMBER,
	rt_valore2 NUMBER,
	rt_bytes_type VARCHAR2(255),
	rt_latency_type VARCHAR2(255),
	rt_modalita_controllo VARCHAR2(255),
	rt_interval_type_real VARCHAR2(255),
	rt_interval_type_stat VARCHAR2(255),
	rt_interval NUMBER,
	rt_finestra VARCHAR2(255),
	-- Applicabilità
	rt_applicabilita VARCHAR2(255) NOT NULL,
	-- Applicabilità con Congestione in Corso
	rt_applicabilita_con_cc NUMBER NOT NULL,
	-- Applicabilità con Degrado Prestazionale
	rt_applicabilita_degrado NUMBER NOT NULL,
	degrato_modalita_controllo VARCHAR2(255),
	degrado_avg_interval_type_real VARCHAR2(255),
	degrado_avg_interval_type_stat VARCHAR2(255),
	degrado_avg_interval NUMBER,
	degrado_avg_finestra VARCHAR2(255),
	degrado_avg_latency_type VARCHAR2(255),
	-- Applicabilità con Stato Allarme
	rt_applicabilita_allarme NUMBER NOT NULL,
	allarme_nome VARCHAR2(255),
	allarme_stato NUMBER,
	allarme_not_stato NUMBER NOT NULL,
	-- fk/pk columns
	id NUMBER NOT NULL,
	-- check constraints
	CONSTRAINT chk_cong_gen_policy_1 CHECK (rt_bytes_type IN ('complessiva','interna','esterna')),
	CONSTRAINT chk_cong_gen_policy_2 CHECK (rt_latency_type IN ('servizio','porta','totale')),
	CONSTRAINT chk_cong_gen_policy_3 CHECK (rt_modalita_controllo IN ('realtime','statistic')),
	CONSTRAINT chk_cong_gen_policy_4 CHECK (rt_interval_type_real IN ('secondi','minuti','orario','giornaliero')),
	CONSTRAINT chk_cong_gen_policy_5 CHECK (rt_interval_type_stat IN ('mensile','settimanale','giornaliero','orario')),
	CONSTRAINT chk_cong_gen_policy_6 CHECK (rt_finestra IN ('corrente','precedente','scorrevole')),
	CONSTRAINT chk_cong_gen_policy_7 CHECK (rt_applicabilita IN ('sempre','condizionale')),
	CONSTRAINT chk_cong_gen_policy_8 CHECK (degrato_modalita_controllo IN ('realtime','statistic')),
	CONSTRAINT chk_cong_gen_policy_9 CHECK (degrado_avg_interval_type_real IN ('secondi','minuti','orario','giornaliero')),
	CONSTRAINT chk_cong_gen_policy_10 CHECK (degrado_avg_interval_type_stat IN ('mensile','settimanale','giornaliero','orario')),
	CONSTRAINT chk_cong_gen_policy_11 CHECK (degrado_avg_finestra IN ('corrente','precedente','scorrevole')),
	CONSTRAINT chk_cong_gen_policy_12 CHECK (degrado_avg_latency_type IN ('servizio','porta','totale')),
	-- unique constraints
	CONSTRAINT uniq_cong_gen_policy_1 UNIQUE (policy_id),
	-- fk/pk keys constraints
	CONSTRAINT pk_ct_config_policy PRIMARY KEY (id)
);


ALTER TABLE ct_config_policy MODIFY policy_built_in DEFAULT 0;
ALTER TABLE ct_config_policy MODIFY rt_simultanee DEFAULT 0;
ALTER TABLE ct_config_policy MODIFY rt_applicabilita DEFAULT 'sempre';
ALTER TABLE ct_config_policy MODIFY rt_applicabilita_con_cc DEFAULT 0;
ALTER TABLE ct_config_policy MODIFY rt_applicabilita_degrado DEFAULT 0;
ALTER TABLE ct_config_policy MODIFY rt_applicabilita_allarme DEFAULT 0;
ALTER TABLE ct_config_policy MODIFY allarme_not_stato DEFAULT 0;

CREATE TRIGGER trg_ct_config_policy
BEFORE
insert on ct_config_policy
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_ct_config_policy.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/



CREATE SEQUENCE seq_ct_active_policy MINVALUE 1 MAXVALUE 9223372036854775807 START WITH 1 INCREMENT BY 1 CACHE 2 NOCYCLE;

CREATE TABLE ct_active_policy
(
	-- Dati Generali
	active_policy_id VARCHAR2(275) NOT NULL,
	policy_alias VARCHAR2(255),
	policy_update_time TIMESTAMP NOT NULL,
	policy_posizione NUMBER NOT NULL,
	policy_continue NUMBER NOT NULL,
	policy_id VARCHAR2(255) NOT NULL,
	policy_enabled NUMBER NOT NULL,
	policy_warning NUMBER NOT NULL,
	-- Valori di Soglia
	policy_redefined NUMBER NOT NULL,
	policy_valore NUMBER,
	policy_valore2 NUMBER,
	-- Filtro
	filtro_enabled NUMBER NOT NULL,
	filtro_protocollo VARCHAR2(255),
	filtro_ruolo VARCHAR2(255),
	filtro_porta VARCHAR2(2000),
	filtro_tipo_fruitore VARCHAR2(255),
	filtro_nome_fruitore VARCHAR2(255),
	filtro_ruolo_fruitore VARCHAR2(255),
	filtro_sa_fruitore VARCHAR2(255),
	filtro_tipo_erogatore VARCHAR2(255),
	filtro_nome_erogatore VARCHAR2(255),
	filtro_ruolo_erogatore VARCHAR2(255),
	filtro_sa_erogatore VARCHAR2(255),
	filtro_tag VARCHAR2(255),
	filtro_tipo_servizio VARCHAR2(255),
	filtro_nome_servizio VARCHAR2(255),
	filtro_versione_servizio NUMBER,
	filtro_azione CLOB,
	filtro_token_claims CLOB,
	-- Filtro per Chiave Applicativa
	filtro_key_enabled NUMBER NOT NULL,
	filtro_key_type VARCHAR2(255),
	filtro_key_name CLOB,
	filtro_key_value CLOB,
	-- Raggruppamento
	group_enabled NUMBER NOT NULL,
	group_ruolo NUMBER NOT NULL,
	group_protocollo NUMBER NOT NULL,
	group_fruitore NUMBER NOT NULL,
	group_sa_fruitore NUMBER NOT NULL,
	group_id_autenticato NUMBER NOT NULL,
	group_token CLOB,
	group_erogatore NUMBER NOT NULL,
	group_sa_erogatore NUMBER NOT NULL,
	group_servizio NUMBER NOT NULL,
	group_azione NUMBER NOT NULL,
	-- Raggruppamento per Chiave Applicativa
	group_key_enabled NUMBER NOT NULL,
	group_key_type VARCHAR2(255),
	group_key_name CLOB,
	-- fk/pk columns
	id NUMBER NOT NULL,
	-- check constraints
	CONSTRAINT chk_cong_att_policy_1 CHECK (filtro_ruolo IN ('delegata','applicativa','entrambi')),
	-- unique constraints
	CONSTRAINT uniq_cong_att_policy_1 UNIQUE (active_policy_id),
	-- fk/pk keys constraints
	CONSTRAINT pk_ct_active_policy PRIMARY KEY (id)
);

-- index
CREATE INDEX idx_cong_att_policy_1 ON ct_active_policy (filtro_ruolo,filtro_porta);

ALTER TABLE ct_active_policy MODIFY policy_continue DEFAULT 0;
ALTER TABLE ct_active_policy MODIFY policy_warning DEFAULT 0;
ALTER TABLE ct_active_policy MODIFY filtro_enabled DEFAULT 0;
ALTER TABLE ct_active_policy MODIFY filtro_key_enabled DEFAULT 0;
ALTER TABLE ct_active_policy MODIFY group_enabled DEFAULT 0;
ALTER TABLE ct_active_policy MODIFY group_ruolo DEFAULT 0;
ALTER TABLE ct_active_policy MODIFY group_protocollo DEFAULT 0;
ALTER TABLE ct_active_policy MODIFY group_fruitore DEFAULT 0;
ALTER TABLE ct_active_policy MODIFY group_sa_fruitore DEFAULT 0;
ALTER TABLE ct_active_policy MODIFY group_id_autenticato DEFAULT 0;
ALTER TABLE ct_active_policy MODIFY group_erogatore DEFAULT 0;
ALTER TABLE ct_active_policy MODIFY group_sa_erogatore DEFAULT 0;
ALTER TABLE ct_active_policy MODIFY group_servizio DEFAULT 0;
ALTER TABLE ct_active_policy MODIFY group_azione DEFAULT 0;
ALTER TABLE ct_active_policy MODIFY group_key_enabled DEFAULT 0;

CREATE TRIGGER trg_ct_active_policy
BEFORE
insert on ct_active_policy
for each row
begin
   IF (:new.id IS NULL) THEN
      SELECT seq_ct_active_policy.nextval INTO :new.id
                FROM DUAL;
   END IF;
end;
/




-- govway
		  		
-- Configurazione
INSERT INTO configurazione (cadenza_inoltro, validazione_stato, validazione_controllo, msg_diag_severita, msg_diag_severita_log4j, auth_integration_manager,validazione_profilo, mod_risposta, indirizzo_telematico, routing_enabled, validazione_manifest, gestione_manifest, tracciamento_buste, dump, dump_bin_pd, dump_bin_pa, statocache,dimensionecache,algoritmocache,lifecache, config_statocache, config_dimensionecache, config_algoritmocache, config_lifecache, auth_statocache, auth_dimensionecache, auth_algoritmocache, auth_lifecache, authn_statocache, authn_dimensionecache, authn_algoritmocache, authn_lifecache, token_statocache, token_dimensionecache, token_algoritmocache, token_lifecache, aa_statocache, aa_dimensionecache, aa_algoritmocache, aa_lifecache, cors_stato, cors_tipo, cors_all_allow_origins, cors_allow_headers, cors_allow_methods, response_cache_stato, response_cache_statocache, response_cache_dimensionecache, response_cache_algoritmocache, response_cache_lifecache) VALUES( '60',       'abilitato',    'rigido', 'infoIntegration', 'infoIntegration', 'principal,ssl,basic', 'disabilitato','reply','disabilitato','disabilitato', 'abilitato', 'disabilitato', 'abilitato', 'disabilitato', 'disabilitato', 'disabilitato', 'abilitato','10000','lru','7200','abilitato','10000','lru','7200','abilitato','5000','lru','7200','abilitato','5000','lru','7200', 'abilitato','5000','lru','600','abilitato','5000','lru','7200', 'abilitato','gateway', 'abilitato', 'Authorization,Content-Type,SOAPAction','GET,PUT,POST,DELETE,PATCH','disabilitato','abilitato','10000','lru','1800');

-- Configurazione Controllo Traffico
INSERT INTO ct_config (max_threads,pd_connection_timeout,pd_read_timeout,pd_avg_time,pa_connection_timeout,pa_read_timeout,pa_avg_time,cache_size,cache_algorithm,cache_life_time) VALUES (200,10000,150000,10000,10000,120000,10000,10000,'LRU',300);

-- Init Configurazione URL Invocazione
INSERT INTO config_url_invocazione (base_url) VALUES ('http://localhost:8080/govway/');

-- Rotta di default per routing
insert INTO routing (tiporotta,registrorotta,is_default) VALUES ('registro',0,1);

-- Registro locale
insert INTO registri (nome,location,tipo) VALUES ('RegistroDB','org.govway.datasource.console','db');

-- Porta di Dominio locale
INSERT INTO pdd (nome,tipo,superuser) VALUES ('GovWay','operativo','amministratore');


-- govway
			  		
-- users
INSERT INTO users (login, password, tipo_interfaccia, permessi, soggetti_all, servizi_all) VALUES ('operatore',	'$1$.bquKJDS$yZ4EYC3354HqEjmSRL/sR0','STANDARD','DR', 1, 1);


-- govway
			  		




