/*
 * ====================================================================
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 *
 */
package org.apache.http.benchmark;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.PrintWriter;
import java.net.InetSocketAddress;
import java.net.Socket;
import java.nio.charset.Charset;
import java.util.Random;

import javax.net.SocketFactory;

import org.apache.http.ConnectionReuseStrategy;
import org.apache.http.Header;
import org.apache.http.HeaderIterator;
import org.apache.http.HttpEntity;
import org.apache.http.HttpEntityEnclosingRequest;
import org.apache.http.HttpException;
import org.apache.http.HttpHost;
import org.apache.http.HttpRequest;
import org.apache.http.HttpResponse;
import org.apache.http.entity.ContentType;
import org.apache.http.impl.DefaultConnectionReuseStrategy;
import org.apache.http.protocol.HTTP;
import org.apache.http.protocol.HttpCoreContext;
import org.apache.http.protocol.HttpProcessor;
import org.apache.http.protocol.HttpRequestExecutor;
import org.apache.http.protocol.ImmutableHttpProcessor;
import org.apache.http.protocol.RequestConnControl;
import org.apache.http.protocol.RequestContent;
import org.apache.http.protocol.RequestExpectContinue;
import org.apache.http.protocol.RequestTargetHost;
import org.apache.http.protocol.RequestUserAgent;
import org.apache.http.util.EntityUtils;

/**
 * Worker thread for the {@link HttpBenchmark HttpBenchmark}.
 *
 *
 * @since 4.0
 */
class BenchmarkWorker implements Runnable {

    private final byte[] buffer = new byte[4096];
    private final HttpCoreContext context;
    private final HttpProcessor httpProcessor;
    private final HttpRequestExecutor httpexecutor;
    private final ConnectionReuseStrategy connstrategy;
    private final HttpRequest request;
    private final HttpHost targetHost;
    private final Config config;
    private final SocketFactory socketFactory;
    private final Stats stats = new Stats();
    static int IdPrefix;
    private boolean timeoutReach;

    private StringBuffer bfDetails = new StringBuffer();
    public String getDetails() {
        if(bfDetails.length()>0){
            return bfDetails.toString();
        }else{
            return null;
        }
    }


    public BenchmarkWorker(
            final HttpRequest request,
            final HttpHost targetHost,
            final SocketFactory socketFactory,
            final Config config) {
        super();
        this.context = new HttpCoreContext();
        this.request = request;
        this.targetHost = targetHost;
        this.config = config;
        this.httpProcessor = new ImmutableHttpProcessor(
                new RequestContent(),
                new RequestTargetHost(),
                new RequestConnControl(),
                new RequestUserAgent("HttpCore-AB/1.1"),
                new RequestExpectContinue(this.config.isUseExpectContinue()));
        this.httpexecutor = new HttpRequestExecutor();

        this.connstrategy = DefaultConnectionReuseStrategy.INSTANCE;
        this.socketFactory = socketFactory;

        this.timeoutReach = false;
    }

    public void setTimedOut() {
        this.timeoutReach = true;
    }
    public boolean isTimedOut() {
        return this.timeoutReach;
    }

    @Override
    public void run() {

        if(this.config.isSleepBeforeEveryMessage()==false){
            this.sleep(-1);
        }

        HttpResponse response = null;
        final BenchmarkConnection conn = new BenchmarkConnection(8 * 1024, stats);

        final String scheme = targetHost.getSchemeName();
        final String hostname = targetHost.getHostName();
        int port = targetHost.getPort();
        if (port == -1) {
            if (scheme.equalsIgnoreCase("https")) {
                port = 443;
            } else {
                port = 80;
            }
        }

        // Populate the execution context
        this.context.setTargetHost(this.targetHost);

        stats.start();
        final int count = config.getRequests();
        //for (int i = 0; i < count; i++) {
        int i;
        if (count < 0)
            i = -2;
        else
            i = 0;

        int requestNumber = 0;

        int numMessaggioDaInviare = 0;

        while (i < count) {

            numMessaggioDaInviare++;

            if(this.config.isSleepBeforeEveryMessage()){
                this.sleep(numMessaggioDaInviare);
            }

            //System.out.println("COUNT["+count+"] i["+i+"] timeoutReach["+timeoutReach+"]");

            if (this.timeoutReach) {
                i = count;
                break;
            }

            if (this.config.getVerbosity() >= 2) {
                this.bfDetails.append("*** Message n."+(++requestNumber)+" ***\n\n");
            }

            //System.out.println("Invoke ??");

            try {
                resetHeader(request);
                if (!conn.isOpen()) {

                    final Socket socket;
                    if (socketFactory != null) {
                        socket = socketFactory.createSocket();
                    } else {
                        socket = new Socket();
                    }

                    final int timeout = config.getSocketTimeout();
                    socket.setSoTimeout(timeout);
                    socket.connect(new InetSocketAddress(hostname, port), timeout);

                    conn.bind(socket);
                }

                final long startInvocation, stopInvocation;

                try {
                    // Prepare request
                    if (this.config.getVerbosity() >= 4) {
                        if(this.request instanceof HttpEntityEnclosingRequest){
                                final String str = EntityUtils.toString(((HttpEntityEnclosingRequest)this.request).getEntity());
                                this.bfDetails.append(str+"\n");
                        }
                        else{
                                final org.apache.http.message.BasicHttpRequest basicReq = (org.apache.http.message.BasicHttpRequest) this.request;
                                final String str = basicReq.getRequestLine().getMethod() +" "+basicReq.getRequestLine().getUri();
                                this.bfDetails.append(str+"\n");
                        }
                    }
                    this.httpexecutor.preProcess(this.request, this.httpProcessor, this.context);

                    // Execute request and get a response
                    if (this.config.getVerbosity() >= 4) {
                        this.bfDetails.append(">> "+ this.request.getRequestLine().toString()+"\n");
                        final Header[] headers = this.request.getAllHeaders();
                        for (int h = 0; h < headers.length; h++) {
                            this.bfDetails.append(">> " + headers[h].toString()+"\n");
                        }
                        this.bfDetails.append("\n");
                    }
                    startInvocation = System.nanoTime();
                    response = this.httpexecutor.execute(this.request, conn, this.context);

                    // Finalize response
                    this.httpexecutor.postProcess(response, this.httpProcessor, this.context);

                } catch (final HttpException e) {
                    stats.incWriteErrors();
                    if (this.config.getVerbosity() >= 2) {
                        this.bfDetails.append("ERROR Failed HTTP request : " + e.getMessage()+"\n");
                    }
                    conn.shutdown();
                    continue;
                }

                stopInvocation = System.nanoTime();
                stats.addInvocationTime(stopInvocation - startInvocation);

                verboseOutput(response);

                //if ( (response.getStatusLine().getStatusCode() == HttpStatus.SC_OK)  || (response.getStatusLine().getStatusCode() == HttpStatus.SC_ACCEPTED) ) {
                boolean statusCodeAccepted = false;
                for (int j = 0; j < this.config.getAcceptedReturnCode().size(); j++) {
                    final int accepted = this.config.getAcceptedReturnCode().get(j);
                    if(accepted == response.getStatusLine().getStatusCode()){
                        statusCodeAccepted = true;
                    }
                }
                if(statusCodeAccepted){
                    stats.incSuccessCount();
                } else {
                    //i++;
                    stats.incFailureCount();
                }

                final HttpEntity entity = response.getEntity();
                if (entity != null) {
                    final ContentType ct = ContentType.getOrDefault(entity);
                    Charset charset = ct.getCharset();
                    if (charset == null) {
                        charset = HTTP.DEF_CONTENT_CHARSET;
                    }
                    long contentLen = 0;
                    final InputStream inStream = entity.getContent();
                    int l;
                    while ((l = inStream.read(this.buffer)) != -1) {
                        contentLen += l;
                        if (config.getVerbosity() >= 4) {
                            final String s = new String(this.buffer, 0, l, charset);
                            this.bfDetails.append(s+"\n");
                        }
                    }
                    inStream.close();
                    stats.setContentLength(contentLen);
                }

                if (config.getVerbosity() >= 4) {
                    this.bfDetails.append("\n");
                    this.bfDetails.append("\n");
                }

                if (!config.isKeepAlive() || !this.connstrategy.keepAlive(response, this.context)) {
                    conn.close();
                } else {
                    stats.incKeepAliveCount();
                }

            } catch (final IOException ex) {
                stats.incFailureCount();
                if (config.getVerbosity() >= 2) {
                    this.bfDetails.append("I/O error: " + ex.getMessage()+"\n");
                    try{
                        final ByteArrayOutputStream bout = new ByteArrayOutputStream();
                        final PrintWriter pw = new PrintWriter(bout);
                        ex.printStackTrace(pw);
                        pw.flush();
                        bout.flush();
                        pw.close();
                        bout.close();
                        this.bfDetails.append(bout.toString());
                        this.bfDetails.append("\n");
                    }catch(final Exception eWrite){}
                }
            } catch (final Exception ex) {
                stats.incFailureCount();
                if (config.getVerbosity() >= 2) {
                    this.bfDetails.append("Generic error: " + ex.getMessage()+"\n");
                    try{
                        final ByteArrayOutputStream bout = new ByteArrayOutputStream();
                        final PrintWriter pw = new PrintWriter(bout);
                        ex.printStackTrace(pw);
                        pw.flush();
                        bout.flush();
                        pw.close();
                        bout.close();
                        this.bfDetails.append(bout.toString());
                        this.bfDetails.append("\n");
                    }catch(final Exception eWrite){}
                }
                }

            //System.out.println("Invoke Finished count["+count+"]");

            if (count < 0){
                i = -2;
            }else{
                i++;
            }

        }
        stats.finish();

        if (response != null) {
            final Header header = response.getFirstHeader("Server");
            if (header != null) {
                stats.setServerName(header.getValue());
            }
        }

        try {
            conn.close();
        } catch (final IOException ex) {
            stats.incFailureCount();
            if (config.getVerbosity() >= 2) {
                this.bfDetails.append("I/O error: " + ex.getMessage()+"\n");
                try{
                    final ByteArrayOutputStream bout = new ByteArrayOutputStream();
                    final PrintWriter pw = new PrintWriter(bout);
                    ex.printStackTrace(pw);
                    pw.flush();
                    bout.flush();
                    pw.close();
                    bout.close();
                    this.bfDetails.append(bout.toString());
                    this.bfDetails.append("\n");
                }catch(final Exception eWrite){}
            }
        }
    }

    private void verboseOutput(final HttpResponse response) {
        if (config.getVerbosity() >= 3) {
            this.bfDetails.append(">> " + request.getRequestLine().toString()+"\n");
            final Header[] headers = request.getAllHeaders();
            for (final Header header : headers) {
                this.bfDetails.append(">> " + header.toString()+"\n");
            }
            this.bfDetails.append("\n");
        }
        if (config.getVerbosity() >= 2) {
            this.bfDetails.append(response.getStatusLine().getStatusCode()+"\n");
        }
        if (config.getVerbosity() >= 3) {
            this.bfDetails.append("<< " + response.getStatusLine().toString()+"\n");
            final Header[] headers = response.getAllHeaders();
            for (final Header header : headers) {
                this.bfDetails.append("<< " + header.toString()+"\n");
            }
            this.bfDetails.append("\n");
        }
    }

    private static void resetHeader(final HttpRequest request) {
        for (final HeaderIterator it = request.headerIterator(); it.hasNext();) {
            final Header header = it.nextHeader();
            if (!(header instanceof DefaultHeader)) {
                it.remove();
            }
        }
    }

    public Stats getStats() {
        return stats;
    }

    private void sleep(final int message){

        boolean sleep = false;
        boolean sleepConstant = false;
        if(this.config.getSleepMaxBeforeRun()!=null && this.config.getSleepMinBeforeRun()!=null){
            sleep = true;
            sleepConstant = this.config.getSleepMaxBeforeRun().intValue() == this.config.getSleepMinBeforeRun().intValue();
        }

        if(sleep == false){
            return;
        }

        final boolean print = false;

        String prefix = "";
        if(message>-1){
            prefix = "[msg-"+message+"] ";
        }

        if(sleepConstant == false){
            final int maxSleep = this.config.getSleepMaxBeforeRun().intValue();
            final int minSleep = this.config.getSleepMinBeforeRun().intValue();
            final Random r = new Random();
            final int sleepMs = minSleep + r.nextInt(maxSleep-minSleep);
            if(sleepMs>1000){
                final int count = sleepMs/1000;
                final int resto = sleepMs%1000;
                if(print)
                    System.out.println(prefix+"sleep "+sleepMs+"ms ...");
                for (int i = 0; i < count; i++) {
                    try{
                        Thread.sleep(1000);
                    }catch(final Exception e){}
                }
                try{
                    Thread.sleep(resto);
                }catch(final Exception e){}
                if(print)
                    System.out.println(prefix+"sleep "+sleepMs+"ms terminated");
            }else{
                if(print)
                    System.out.println(prefix+"sleep "+sleepMs+"ms ...");
                try{
                    Thread.sleep(sleepMs);
                }catch(final Exception e){}
                if(print)
                    System.out.println(prefix+"sleep "+sleepMs+"ms terminated");
            }
        }
        else{
            final int millisecond = this.config.getSleepMaxBeforeRun().intValue();
            if(millisecond>1000){
                final int count = millisecond/1000;
                final int resto = millisecond%1000;
                if(print)
                    System.out.println(prefix+"sleep "+millisecond+"ms ...");
                for (int i = 0; i < count; i++) {
                    try{
                        Thread.sleep(1000);
                    }catch(final Exception e){}
                }
                try{
                    Thread.sleep(resto);
                }catch(final Exception e){}
                if(print)
                    System.out.println(prefix+"sleep "+millisecond+"ms terminated");
            }else{
                if(print)
                    System.out.println(prefix+"sleep "+millisecond+"ms ...");
                try{
                    Thread.sleep(millisecond);
                }catch(final Exception e){}
                if(print)
                    System.out.println(prefix+"sleep "+millisecond+"ms terminated");
            }
        }
    }
}
