/*
 * ====================================================================
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 *
 */
package org.apache.http.benchmark;

import java.io.File;
import java.net.URL;
import java.util.ArrayList;
import java.util.List;

import org.apache.http.HttpStatus;

public class Config {

    private URL url;
    private int requests;
    private int threads;
    private boolean keepAlive;
    private int verbosity;
    private boolean headInsteadOfGet;
    private boolean useHttp1_0;
    private String contentType;
    private String[] headers;
    private int socketTimeout;
    private String method = "GET";
    private boolean useChunking;
    private boolean useExpectContinue;
    private boolean useAcceptGZip;
    private File payloadFile = null;
    private boolean payloadFileTemplate = false;
    private String payloadText = null;
    private String soapAction = null;

    private boolean disableSSLVerification = true;
    private String trustStorePath = null;
    private String identityStorePath = null;
    private String trustStorePassword = null;
    private String identityStorePassword = null;

    private File attachmentsDir = null;

    private File busta;
    private String reportFile = null;
    //private int clientId;

    private int durationSec;

    private List<Integer> acceptedReturnCode = new ArrayList<Integer>();
    {
        acceptedReturnCode.add(HttpStatus.SC_OK);
        acceptedReturnCode.add(HttpStatus.SC_ACCEPTED);
    }

    private Integer sleepMinBeforeRun;
    private Integer sleepMaxBeforeRun;
    private boolean sleepBeforeEveryMessage = false;


    public Config() {
        super();
        this.url = null;
        this.requests = 1;
        this.threads = 1;
        this.keepAlive = false;
        this.verbosity = 0;
        this.headInsteadOfGet = false;
        this.useHttp1_0 = false;
        this.payloadFile = null;
        this.payloadText = null;
        this.contentType = null;
        this.headers = null;
        this.socketTimeout = 60000;
        this.attachmentsDir = null;
        this.reportFile = null;
    }

    public URL getUrl() {
        return url;
    }

    public void setUrl(final URL url) {
        this.url = url;
    }

    public int getRequests() {
        return requests;
    }

    public void setRequests(final int requests) {
        this.requests = requests;
    }

    public int getThreads() {
        return threads;
    }

    public void setThreads(final int threads) {
        this.threads = threads;
    }

    public boolean isKeepAlive() {
        return keepAlive;
    }

    public void setKeepAlive(final boolean keepAlive) {
        this.keepAlive = keepAlive;
    }

    public int getVerbosity() {
        return verbosity;
    }

    public void setVerbosity(final int verbosity) {
        this.verbosity = verbosity;
    }

    public boolean isHeadInsteadOfGet() {
        return headInsteadOfGet;
    }

    public void setHeadInsteadOfGet(final boolean headInsteadOfGet) {
        this.headInsteadOfGet = headInsteadOfGet;
        this.method = "HEAD";
    }

    public boolean isUseHttp1_0() {
        return useHttp1_0;
    }

    public void setUseHttp1_0(final boolean useHttp1_0) {
        this.useHttp1_0 = useHttp1_0;
    }

    public File getPayloadFile() {
        return payloadFile;
    }

    public void setPayloadFile(final File payloadFile) throws Exception {
        if(payloadFile.exists()==false){
            throw new Exception("File not found: " + payloadFile.getAbsolutePath());
        }
        if (!payloadFile.isFile()) {
            throw new Exception("Expected file, found other type: " + payloadFile.getAbsolutePath());
        }
        if (!payloadFile.canRead()) {
            throw new Exception("Cannot read file: " + payloadFile.getAbsolutePath());
        }
        this.payloadFile = payloadFile;
    }

    public boolean isPayloadFileTemplate() {
        return this.payloadFileTemplate;
    }

    public String getContentType() {
        return contentType;
    }

    public void setContentType(final String contentType) {
        this.contentType = contentType;
    }

    public String[] getHeaders() {
        return headers;
    }

    public void setHeaders(final String[] headers) {
        this.headers = headers;
    }

    public int getSocketTimeout() {
        return socketTimeout;
    }

    public void setSocketTimeout(final int socketTimeout) {
        this.socketTimeout = socketTimeout;
    }

    public void setMethod(final String method) {
        this.method = method;
    }

    public void setUseChunking(final boolean useChunking) {
        this.useChunking = useChunking;
    }

    public void setUseExpectContinue(final boolean useExpectContinue) {
        this.useExpectContinue = useExpectContinue;
    }

    public void setUseAcceptGZip(final boolean useAcceptGZip) {
        this.useAcceptGZip = useAcceptGZip;
    }

    public String getMethod() {
        return method;
    }

    public boolean isUseChunking() {
        return useChunking;
    }

    public boolean isUseExpectContinue() {
        return useExpectContinue;
    }

    public boolean isUseAcceptGZip() {
        return useAcceptGZip;
    }

    public String getPayloadText() {
        return payloadText;
    }

    public String getSoapAction() {
        return soapAction;
    }

    public boolean isDisableSSLVerification() {
        return disableSSLVerification;
    }

    public String getTrustStorePath() {
        return trustStorePath;
    }

    public String getIdentityStorePath() {
        return identityStorePath;
    }

    public String getTrustStorePassword() {
        return trustStorePassword;
    }

    public String getIdentityStorePassword() {
        return identityStorePassword;
    }

    public void setPayloadText(final String payloadText) {
        this.payloadText = payloadText;
    }

    public void setSoapAction(final String soapAction) {
        this.soapAction = soapAction;
    }

    public void setDisableSSLVerification(final boolean disableSSLVerification) {
        this.disableSSLVerification = disableSSLVerification;
    }

    public void setTrustStorePath(final String trustStorePath) {
        this.trustStorePath = trustStorePath;
    }

    public void setIdentityStorePath(final String identityStorePath) {
        this.identityStorePath = identityStorePath;
    }

    public void setTrustStorePassword(final String trustStorePassword) {
        this.trustStorePassword = trustStorePassword;
    }

    public void setIdentityStorePassword(final String identityStorePassword) {
        this.identityStorePassword = identityStorePassword;
    }

    public Integer getSleepMinBeforeRun() {
        return sleepMinBeforeRun;
    }

    public void setSleepMinBeforeRun(final Integer sleepMinBeforeRun) {
        this.sleepMinBeforeRun = sleepMinBeforeRun;
    }

    public Integer getSleepMaxBeforeRun() {
        return sleepMaxBeforeRun;
    }

    public void setSleepMaxBeforeRun(final Integer sleepMaxBeforeRun) {
        this.sleepMaxBeforeRun = sleepMaxBeforeRun;
    }

    public boolean isSleepBeforeEveryMessage() {
        return sleepBeforeEveryMessage;
    }

    public void setSleepBeforeEveryMessage(final boolean sleepBeforeEveryMessage) {
        this.sleepBeforeEveryMessage = sleepBeforeEveryMessage;
    }

    public List<Integer> getAcceptedReturnCode() {
        return acceptedReturnCode;
    }

    public void setAcceptedReturnCode(final List<Integer> acceptedReturnCode) {
        this.acceptedReturnCode = acceptedReturnCode;
    }

    public File getAttachmentsDir() {
        return this.attachmentsDir;
    }

    public void setAttachmentsDir(final File attachmentsDir) throws Exception {
        if(attachmentsDir.exists()==false){
            throw new Exception("File not found: " + attachmentsDir.getAbsolutePath());
        }
        if (!attachmentsDir.isDirectory()) {
            throw new Exception("Expected dir, found file: " + attachmentsDir.getAbsolutePath());
        }
        if (!attachmentsDir.canRead()) {
            throw new Exception("Cannot read directory: " + attachmentsDir.getAbsolutePath());
        }
        this.attachmentsDir = attachmentsDir;
    }

    public File getBustaFileHeader() {
        return this.busta;
    }

    public void setBustaFileHeader(final File busta) throws Exception {
        if(this.payloadFileTemplate){
            throw new Exception("Cannot set busta with payloadFileTemplate enabled");
        }
        if(busta.exists()==false){
            throw new Exception("File not found: " + busta.getAbsolutePath());
        }
        if (!busta.isFile()) {
            throw new Exception("Expected file, found other type: " + busta.getAbsolutePath());
        }
        if (!busta.canRead()) {
            throw new Exception("Cannot read file: " + busta.getAbsolutePath());
        }
        this.busta = busta;
    }

    public void setPayloadFileTemplate(final boolean payloadFileTemplate) throws Exception {
        this.payloadFileTemplate = payloadFileTemplate;
        if(this.payloadFileTemplate && this.busta!=null){
            throw new Exception("Cannot set payloadFileTemplate with busta enabled (found busta: "+this.busta.getAbsolutePath()+")");
        }
    }

//    public int getClientId() {
//        return this.clientId;
//    }
//
//    public void setClientId(final int clientId) {
//        this.clientId = clientId;
//    }

    public int getDurationSec() {
        return this.durationSec;
    }

    public void setDurationSec(final int durationSec) {
        this.durationSec = durationSec;
        this.requests = -1;
    }

    public void setReportFile(final String filename) {
        this.reportFile = filename;
    }

    public String getReportFile() {
        return this.reportFile;
    }


    public Config copy() {
        final Config copy = new Config();
        copy.url = this.url;
        copy.requests = this.requests;
        copy.threads = this.threads;
        copy.keepAlive = this.keepAlive;
        copy.verbosity = this.verbosity;
        copy.headInsteadOfGet = this.headInsteadOfGet;
        copy.useHttp1_0 = this.useHttp1_0;
        copy.contentType = this.contentType;
        copy.headers = this.headers;
        copy.socketTimeout = this.socketTimeout;
        copy.method = this.method;
        copy.useChunking = this.useChunking;
        copy.useExpectContinue = this.useExpectContinue;
        copy.useAcceptGZip = this.useAcceptGZip;
        copy.payloadFile = this.payloadFile;
        copy.payloadFileTemplate = this.payloadFileTemplate;
        copy.payloadText = this.payloadText;
        copy.soapAction = this.soapAction;

        copy.disableSSLVerification = this.disableSSLVerification;
        copy.trustStorePath = this.trustStorePath;
        copy.identityStorePath = this.identityStorePath;
        copy.trustStorePassword = this.trustStorePassword;
        copy.identityStorePassword = this.identityStorePassword;

        copy.attachmentsDir = this.attachmentsDir;
        copy.busta = this.busta;
        copy.reportFile = this.reportFile;
        //copy.clientId = this.clientId;
        copy.durationSec = this.durationSec;
        copy.acceptedReturnCode = this.acceptedReturnCode;
        copy.sleepMinBeforeRun = this.sleepMinBeforeRun;
        copy.sleepMaxBeforeRun = this.sleepMaxBeforeRun;
        copy.sleepBeforeEveryMessage = this.sleepBeforeEveryMessage;

        return copy;
    }

}
