/*
 * ====================================================================
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 *
 */
package org.apache.http.benchmark;

import java.text.NumberFormat;
import java.util.ArrayList;
import java.util.List;

import org.apache.http.HttpHost;

public class ResultProcessor {

    static NumberFormat nf0 = NumberFormat.getInstance();
    static NumberFormat nf2 = NumberFormat.getInstance();
    static NumberFormat nf3 = NumberFormat.getInstance();
    static NumberFormat nf6 = NumberFormat.getInstance();

    static {
        nf0.setMaximumFractionDigits(0);
        nf0.setMinimumFractionDigits(0);
        nf2.setMaximumFractionDigits(2);
        nf2.setMinimumFractionDigits(2);
        nf3.setMaximumFractionDigits(3);
        nf3.setMinimumFractionDigits(3);
        nf6.setMaximumFractionDigits(6);
        nf6.setMinimumFractionDigits(6);
    }

    static Results collectResults(final BenchmarkWorker[] workers, final HttpHost host, final String uri) {
        long totalTimeNano = 0;
        long successCount    = 0;
        long failureCount    = 0;
        long writeErrors     = 0;
        long keepAliveCount  = 0;
        long totalBytesRcvd  = 0;
        long totalBytesSent  = 0;
        long fifty = 0;
        long sixty = 0;
        long seventy = 0;
        long eighty = 0;
        long ninety = 0;
        double rpsSUM = 0;

        final Stats stats = workers[0].getStats();

        for (final BenchmarkWorker worker : workers) {
            final Stats s = worker.getStats();
            totalTimeNano  += s.getDuration();
            successCount   += s.getSuccessCount();
            failureCount   += s.getFailureCount();
            writeErrors    += s.getWriteErrors();
            keepAliveCount += s.getKeepAliveCount();
            totalBytesRcvd += s.getTotalBytesRecv();
            totalBytesSent += s.getTotalBytesSent();
            fifty = Math.max(fifty, s.getPercentile(50));
            sixty = Math.max(sixty, s.getPercentile(60));
            seventy = Math.max(seventy, s.getPercentile(70));
            eighty = Math.max(eighty, s.getPercentile(80));
            ninety = Math.max(ninety, s.getNinetyPercentile());
            rpsSUM += s.getRPSfromTimes();
        }

        final Results results = new Results();
        results.serverName = stats.getServerName();
        results.hostName = host.getHostName();
        results.hostPort = host.getPort() > 0 ? host.getPort() :
            host.getSchemeName().equalsIgnoreCase("https") ? 443 : 80;
        results.documentPath = uri;
        results.contentLength = stats.getContentLength();
        results.concurrencyLevel = workers.length;
        results.totalTimeNano = totalTimeNano;
        results.successCount = successCount;
        results.failureCount = failureCount;
        results.writeErrors = writeErrors;
        results.keepAliveCount = keepAliveCount;
        results.totalBytesRcvd = totalBytesRcvd;
        results.totalBytesSent = totalBytesSent;
        results.totalBytes = totalBytesRcvd + (totalBytesSent > 0 ? totalBytesSent : 0);

        results.fifty = fifty;
        results.sixty = sixty;
        results.seventy = seventy;
        results.eighty = eighty;
        results.ninety = ninety;
        results.rpsSUM = rpsSUM;

        results.workers = workers;

        return results;
    }

    static String printResults(final Results results) {
        final int threads = results.getConcurrencyLevel();
        final double totalTimeMs  = (results.getTotalTimeNano() / threads) / 1000000; // convert nano secs to milli secs
        final double timePerReqMs = totalTimeMs / results.getSuccessCount();
        final double totalTimeSec = totalTimeMs / 1000;
        final double reqsPerSec   = results.getSuccessCount() / totalTimeSec;
        @SuppressWarnings("unused")
        final double RPSTimes = results.rpsSUM;// workers.length;

        final StringBuilder sb = new StringBuilder(1024);

        printAndAppend(sb, "\nServer Software:\t\t" + results.getServerName());
        printAndAppend(sb,  "Server Hostname:\t\t" + results.getHostName());
        printAndAppend(sb,  "Server Port:\t\t\t" + Integer.valueOf(results.getHostPort()));
        printAndAppend(sb,  "Document Path:\t\t\t" + results.getDocumentPath());
        printAndAppend(sb,  "Document Length:\t\t" + results.getContentLength() + " bytes\n");
        printAndAppend(sb,  "Concurrency Level:\t\t" + results.getConcurrencyLevel());
        printAndAppend(sb,  "Time taken for tests:\t\t" + nf6.format(totalTimeSec) + " seconds");
        printAndAppend(sb,  "Complete requests:\t\t" + results.getSuccessCount());
        printAndAppend(sb,  "Failed requests:\t\t" + results.getFailureCount());
        printAndAppend(sb,  "Write errors:\t\t\t" + results.getWriteErrors());
        printAndAppend(sb,  "Kept alive:\t\t\t" + results.getKeepAliveCount());
        printAndAppend(sb,  "Total transferred:\t\t" + results.getTotalBytes() + " bytes");
        printAndAppend(sb,  "Requests per second:\t\t" + nf2.format(reqsPerSec) + " [#/sec] (mean)");
        // NOTA: verificare se va bene, a volte non sembra
        //printAndAppend(sb, "RPS from times:\t\t\t" + nf2.format(RPSTimes)+ " [#/sec] (mean)");
        printAndAppend(sb,  "Time per request:\t\t" + nf3.format(timePerReqMs
                * results.getConcurrencyLevel()) + " [ms] (mean)");
        printAndAppend(sb,  "Time per request:\t\t" + nf3.format(timePerReqMs) +
            " [ms] (mean, across all concurrent requests)");
        printAndAppend(sb,  "Transfer rate:\t\t\t" +
            nf2.format(results.getTotalBytesRcvd() / 1000 / totalTimeSec) + " [Kbytes/sec] received");
        printAndAppend(sb,  "\t\t\t\t" +
            (results.getTotalBytesSent() > 0 ? nf2.format(results.getTotalBytesSent()
                    / 1000 / totalTimeSec) : Integer.valueOf(-1)) + " kb/s sent");
        printAndAppend(sb,  "\t\t\t\t" +
            nf2.format(results.getTotalBytes() / 1000 / totalTimeSec) + " kb/s total");
        printAndAppend(sb, "Percentile of the requests served within a certain time (ms)");
        printAndAppend(sb, " 50%:\t\t" + nf2.format((double) (results.fifty / 1000000)));
        printAndAppend(sb, " 60%:\t\t" + nf2.format((double) (results.sixty / 1000000)));
        printAndAppend(sb, " 70%:\t\t"+ nf2.format((double) (results.seventy / 1000000)));
        printAndAppend(sb, " 80%:\t\t"+ nf2.format((double) (results.eighty / 1000000)));
        printAndAppend(sb, " 90%:\t\t"+ nf2.format((double) (results.ninety / 1000000)));

        printFrequencies(sb, results.workers);

        printDebug(sb, results.workers);

        return sb.toString();
    }

    private static void printAndAppend(final StringBuilder sb, final String s) {
        printAndAppend(sb, s, true);
    }
    private static void printAndAppend(final StringBuilder sb, final String s, final boolean returnLine) {
        //System.out.println(s);
        sb.append(s);
        if(returnLine){
            sb.append("\r\n");
        }
    }

    public static void printFrequencies(final StringBuilder sb, final BenchmarkWorker[] workers) {
        long mintime = Long.MAX_VALUE;
        long maxtime = 0;
        long minfreq = 0;
        @SuppressWarnings("unused")
        long maxfreq = 0;

        // Determinazione dell'intervallo
        for (int i = 0; i < workers.length; i++) {
            final Stats s = workers[i].getStats();
            final long worker_min = s.getMinInvocationTime();
            if (worker_min < mintime) {
                mintime = worker_min;
                minfreq = 1;
            } else if (worker_min == mintime)
                minfreq++;

            final long worker_max = s.getMaxInvocationTime();
            if (worker_max > maxtime) {
                maxtime = worker_max;
                maxfreq = 1;
            } else if (worker_max == maxtime)
                maxfreq++;

        }

        // calcolo delle frequenze
        // intervallo = max - min
        // suddivisione = intervallo / quantita_suddivisioni
        // freq[quantita_suddivisioni]
        // indice=min
        // while indice < max
        // foreach worker:
        // f = worker.statistica.getFreq(indice,indice+suddivisione)
        // freq[i] += f
        // indice += suddivisione

        printAndAppend(sb, String.format("Invocation time frequency distribution on interval (ms) ["
                +0+"," + maxtime / 1000000 + "]:"));
                        //+ mintime / 1000000 + "," + maxtime / 1000000 + "]:"));
        final int quantita_suddivisioni = 10; // TODO renderlo dinamico sulla
                                        // lunghezza dell'intervallo
        final long interval = maxtime - mintime;
        final long ampiezza = interval / quantita_suddivisioni;

        final List<Long> numero = new ArrayList<Long>();

        //printAndAppend(sb, String.format("Freqs: %-6d", minfreq),false);
        numero.add(minfreq);
        // System.err.println("[0,"+mintime+"] --> "+minfreq);
        long indice = mintime;
        while (indice < maxtime) {
            long sum = 0;
            // long limite_superiore = ((indice+ampiezza) >= maxtime) ? maxtime
            // : (indice + ampiezza);
            long limite_superiore = indice + ampiezza;

            if (maxtime - limite_superiore < ampiezza) {
                limite_superiore = maxtime;
                // System.err.print("*");
            }

            for (int i = 0; i < workers.length; i++) {
                final Stats s = workers[i].getStats();
                sum += s.getFrequencyIn(indice, limite_superiore);
            }
            // System.err.println("["+indice+","+limite_superiore+"] --> "+sum);
            indice = limite_superiore;

    //            if (indice >= maxtime)
    //                printAndAppend(sb, String.format("%-6d%n", sum),false);
    //            else
    //                printAndAppend(sb, String.format("%-6d", sum),false);
            numero.add(sum);
        }

        //printAndAppend(sb, String.format("Interval/Freq: "),false);
        int indexNumber = 0;
        long precIndex = 0;
        indice = mintime;
        while (indice < maxtime) {
            final long newIndice = (indice / 1000000);
            printAndAppend(sb, String.format(" [%d-%d]   \t%d ", precIndex, newIndice, numero.get(indexNumber++)));
            precIndex = newIndice;
            indice += ampiezza;
            if (maxtime - indice < ampiezza)
                indice = maxtime;
        }
        final long newIndice = (indice / 1000000);
        try{
            printAndAppend(sb, String.format(" [%d-%d]   \t%d %n", precIndex, newIndice, numero.get(indexNumber++)));
        }catch(final Exception e){
            printAndAppend(sb, String.format(" [%d-%d]   \tERR%d %n", precIndex, newIndice, -1));
        }

    }


    public static void printDebug(final StringBuilder sb, final BenchmarkWorker[] workers) {

        // Determinazione dell'intervallo
        for (int i = 0; i < workers.length; i++) {

            if(workers[i].getDetails()!=null && workers[i].getDetails().length()>0){

                sb.append("==================================\n");
                sb.append("Worker-"+i+": \n\n");
                sb.append(workers[i].getDetails());
                sb.append("\n\n");

            }

        }

    }

}
