

-- GovWay Config

-- **** Porte di Dominio ****

CREATE SEQUENCE seq_pdd AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE pdd
(
	nome VARCHAR(255) NOT NULL,
	descrizione VARCHAR(255),
	-- Tipo della Porta
	tipo VARCHAR(255),
	implementazione VARCHAR(255),
	subject VARCHAR(255),
	-- client auth: disabilitato/abilitato
	client_auth VARCHAR(255),
	ora_registrazione TIMESTAMP,
	superuser VARCHAR(255),
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	-- check constraints
	CONSTRAINT chk_pdd_1 CHECK (tipo IN ('operativo','nonoperativo','esterno')),
	-- unique constraints
	CONSTRAINT unique_pdd_1 UNIQUE (nome),
	-- fk/pk keys constraints
	CONSTRAINT pk_pdd PRIMARY KEY (id)
);

-- index
CREATE UNIQUE INDEX index_pdd_1 ON pdd (nome);

ALTER TABLE pdd ALTER COLUMN implementazione SET DEFAULT 'standard';
ALTER TABLE pdd ALTER COLUMN ora_registrazione SET DEFAULT CURRENT_TIMESTAMP;

CREATE TABLE pdd_init_seq (id BIGINT);
INSERT INTO pdd_init_seq VALUES (NEXT VALUE FOR seq_pdd);




-- govway
		  		
-- **** Connettori ****

CREATE SEQUENCE seq_connettori AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE connettori
(
	-- (disabilitato,http,jms)
	endpointtype VARCHAR(255) NOT NULL,
	nome_connettore VARCHAR(2000) NOT NULL,
	-- url nel caso http
	url VARCHAR(4000),
	-- nel caso di http indicazione se usare chunking
	transfer_mode VARCHAR(255),
	transfer_mode_chunk_size INT,
	-- nel caso di http indicazione se seguire il redirect o meno
	redirect_mode VARCHAR(255),
	redirect_max_hop INT,
	-- nome coda jms
	nome VARCHAR(255),
	-- tipo coda jms (queue,topic)
	tipo VARCHAR(255),
	-- utente di una connessione jms
	utente VARCHAR(255),
	-- password per una connessione jms
	password VARCHAR(255),
	enc_password VARCHAR(65535),
	-- context property: initial_content
	initcont VARCHAR(255),
	-- context property: url_pkg
	urlpkg VARCHAR(255),
	-- context property: provider_url
	provurl VARCHAR(255),
	-- ConnectionFactory JMS
	connection_factory VARCHAR(255),
	-- Messaggio JMS inviato come text/byte message
	send_as VARCHAR(255),
	-- 1/0 (true/false) abilita il debug tramite il connettore
	debug INT,
	-- 1/0 (true/false) abilita l'utilizzo di un proxy tramite il connettore
	proxy INT,
	proxy_type VARCHAR(255),
	proxy_hostname VARCHAR(255),
	proxy_port VARCHAR(255),
	proxy_username VARCHAR(255),
	proxy_password VARCHAR(255),
	enc_proxy_password VARCHAR(65535),
	-- Indicazione sull'intervallo massimo di tempo necessario per instaurare una connessione (intervallo in ms)
	connection_timeout INT,
	-- Indicazione sull'intervallo massimo di tempo che può occorrere prima di ricevere una risposta (intervallo in ms)
	read_timeout INT,
	-- Indicazione sull'intervallo massimo di tempo medio atteso prima di ricevere una risposta (intervallo in ms)
	avg_response_time INT,
	-- 1/0 (true/false) indica se il connettore e' gestito tramite le proprieta' custom
	custom INT,
	-- Gestione Token
	token_policy VARCHAR(255),
	api_key VARCHAR(65535),
	api_key_header VARCHAR(255),
	app_id VARCHAR(65535),
	app_id_header VARCHAR(255),
	http_impl VARCHAR(255),
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	-- unique constraints
	CONSTRAINT unique_connettori_1 UNIQUE (nome_connettore),
	-- fk/pk keys constraints
	CONSTRAINT pk_connettori PRIMARY KEY (id)
);

-- index
CREATE UNIQUE INDEX index_connettori_1 ON connettori (nome_connettore);

ALTER TABLE connettori ALTER COLUMN debug SET DEFAULT 0;
ALTER TABLE connettori ALTER COLUMN proxy SET DEFAULT 0;
ALTER TABLE connettori ALTER COLUMN custom SET DEFAULT 0;

CREATE TABLE connettori_init_seq (id BIGINT);
INSERT INTO connettori_init_seq VALUES (NEXT VALUE FOR seq_connettori);



CREATE SEQUENCE seq_connettori_custom AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE connettori_custom
(
	name VARCHAR(255) NOT NULL,
	value VARCHAR(4000) NOT NULL,
	enc_value VARCHAR(65535),
	id_connettore BIGINT NOT NULL,
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	-- unique constraints
	CONSTRAINT unique_connettori_custom_1 UNIQUE (id_connettore,name),
	-- fk/pk keys constraints
	CONSTRAINT fk_connettori_custom_1 FOREIGN KEY (id_connettore) REFERENCES connettori(id),
	CONSTRAINT pk_connettori_custom PRIMARY KEY (id)
);

-- index
CREATE INDEX idx_conn_custom_1 ON connettori_custom (id_connettore);
CREATE TABLE connettori_custom_init_seq (id BIGINT);
INSERT INTO connettori_custom_init_seq VALUES (NEXT VALUE FOR seq_connettori_custom);



CREATE SEQUENCE seq_connettori_properties AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE connettori_properties
(
	-- nome connettore personalizzato attraverso file properties
	nome_connettore VARCHAR(255) NOT NULL,
	-- location del file properties
	path VARCHAR(255) NOT NULL,
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	-- unique constraints
	CONSTRAINT unique_connettori_properties_1 UNIQUE (nome_connettore),
	-- fk/pk keys constraints
	CONSTRAINT pk_connettori_properties PRIMARY KEY (id)
);

-- index
CREATE UNIQUE INDEX index_connettori_properties_1 ON connettori_properties (nome_connettore);
CREATE TABLE connettori_properties_init_seq (id BIGINT);
INSERT INTO connettori_properties_init_seq VALUES (NEXT VALUE FOR seq_connettori_properties);




-- govway
		  		
-- **** Connettori Gestione Errore ****

CREATE SEQUENCE seq_gestione_errore AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE gestione_errore
(
	-- accetta/rispedisci
	comportamento_default VARCHAR(255),
	cadenza_rispedizione VARCHAR(255),
	nome VARCHAR(255) NOT NULL,
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	-- unique constraints
	CONSTRAINT unique_gestione_errore_1 UNIQUE (nome),
	-- fk/pk keys constraints
	CONSTRAINT pk_gestione_errore PRIMARY KEY (id)
);

CREATE TABLE gestione_errore_init_seq (id BIGINT);
INSERT INTO gestione_errore_init_seq VALUES (NEXT VALUE FOR seq_gestione_errore);



CREATE SEQUENCE seq_gestione_errore_trasporto AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE gestione_errore_trasporto
(
	id_gestione_errore BIGINT NOT NULL,
	valore_massimo INT,
	valore_minimo INT,
	-- accetta/rispedisci
	comportamento VARCHAR(255),
	cadenza_rispedizione VARCHAR(255),
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	-- fk/pk keys constraints
	CONSTRAINT fk_gestione_errore_trasporto_1 FOREIGN KEY (id_gestione_errore) REFERENCES gestione_errore(id),
	CONSTRAINT pk_gestione_errore_trasporto PRIMARY KEY (id)
);

-- index
CREATE INDEX idx_gest_err_trasporto_1 ON gestione_errore_trasporto (id_gestione_errore);
CREATE TABLE gestione_errore_trasporto_init_seq (id BIGINT);
INSERT INTO gestione_errore_trasporto_init_seq VALUES (NEXT VALUE FOR seq_gestione_errore_trasporto);



CREATE SEQUENCE seq_gestione_errore_soap AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE gestione_errore_soap
(
	id_gestione_errore BIGINT NOT NULL,
	fault_actor VARCHAR(255),
	fault_code VARCHAR(255),
	fault_string VARCHAR(255),
	-- accetta/rispedisci
	comportamento VARCHAR(255),
	cadenza_rispedizione VARCHAR(255),
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	-- fk/pk keys constraints
	CONSTRAINT fk_gestione_errore_soap_1 FOREIGN KEY (id_gestione_errore) REFERENCES gestione_errore(id),
	CONSTRAINT pk_gestione_errore_soap PRIMARY KEY (id)
);

-- index
CREATE INDEX idx_gest_err_soap_1 ON gestione_errore_soap (id_gestione_errore);
CREATE TABLE gestione_errore_soap_init_seq (id BIGINT);
INSERT INTO gestione_errore_soap_init_seq VALUES (NEXT VALUE FOR seq_gestione_errore_soap);




-- govway
		  		
-- **** Gruppi ****

CREATE SEQUENCE seq_gruppi AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE gruppi
(
	nome VARCHAR(255) NOT NULL,
	descrizione VARCHAR(4000),
	service_binding VARCHAR(255),
	superuser VARCHAR(255),
	ora_registrazione TIMESTAMP,
	utente_richiedente VARCHAR(255),
	data_creazione TIMESTAMP,
	utente_ultima_modifica VARCHAR(255),
	data_ultima_modifica TIMESTAMP,
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	-- check constraints
	CONSTRAINT chk_gruppi_1 CHECK (service_binding IN ('soap','rest')),
	-- unique constraints
	CONSTRAINT unique_gruppi_1 UNIQUE (nome),
	-- fk/pk keys constraints
	CONSTRAINT pk_gruppi PRIMARY KEY (id)
);

-- index
CREATE UNIQUE INDEX index_gruppi_1 ON gruppi (nome);

ALTER TABLE gruppi ALTER COLUMN ora_registrazione SET DEFAULT CURRENT_TIMESTAMP;

CREATE TABLE gruppi_init_seq (id BIGINT);
INSERT INTO gruppi_init_seq VALUES (NEXT VALUE FOR seq_gruppi);




-- govway
		  		
-- **** Ruoli ****

CREATE SEQUENCE seq_ruoli AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE ruoli
(
	nome VARCHAR(255) NOT NULL,
	descrizione VARCHAR(4000),
	tipologia VARCHAR(255) NOT NULL,
	nome_esterno VARCHAR(255),
	contesto_utilizzo VARCHAR(255) NOT NULL,
	superuser VARCHAR(255),
	ora_registrazione TIMESTAMP,
	utente_richiedente VARCHAR(255),
	data_creazione TIMESTAMP,
	utente_ultima_modifica VARCHAR(255),
	data_ultima_modifica TIMESTAMP,
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	-- check constraints
	CONSTRAINT chk_ruoli_1 CHECK (tipologia IN ('interno','esterno','qualsiasi')),
	CONSTRAINT chk_ruoli_2 CHECK (contesto_utilizzo IN ('portaDelegata','portaApplicativa','qualsiasi')),
	-- unique constraints
	CONSTRAINT unique_ruoli_1 UNIQUE (nome),
	-- fk/pk keys constraints
	CONSTRAINT pk_ruoli PRIMARY KEY (id)
);

-- index
CREATE UNIQUE INDEX index_ruoli_1 ON ruoli (nome);

ALTER TABLE ruoli ALTER COLUMN tipologia SET DEFAULT 'qualsiasi';
ALTER TABLE ruoli ALTER COLUMN contesto_utilizzo SET DEFAULT 'qualsiasi';
ALTER TABLE ruoli ALTER COLUMN ora_registrazione SET DEFAULT CURRENT_TIMESTAMP;

CREATE TABLE ruoli_init_seq (id BIGINT);
INSERT INTO ruoli_init_seq VALUES (NEXT VALUE FOR seq_ruoli);




-- govway
		  		
-- **** Scope ****

CREATE SEQUENCE seq_scope AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE scope
(
	nome VARCHAR(255) NOT NULL,
	descrizione VARCHAR(4000),
	tipologia VARCHAR(255),
	nome_esterno VARCHAR(255),
	contesto_utilizzo VARCHAR(255) NOT NULL,
	superuser VARCHAR(255),
	ora_registrazione TIMESTAMP,
	utente_richiedente VARCHAR(255),
	data_creazione TIMESTAMP,
	utente_ultima_modifica VARCHAR(255),
	data_ultima_modifica TIMESTAMP,
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	-- check constraints
	CONSTRAINT chk_scope_1 CHECK (contesto_utilizzo IN ('portaDelegata','portaApplicativa','qualsiasi')),
	-- unique constraints
	CONSTRAINT unique_scope_1 UNIQUE (nome),
	-- fk/pk keys constraints
	CONSTRAINT pk_scope PRIMARY KEY (id)
);

-- index
CREATE UNIQUE INDEX index_scope_1 ON scope (nome);

ALTER TABLE scope ALTER COLUMN contesto_utilizzo SET DEFAULT 'qualsiasi';
ALTER TABLE scope ALTER COLUMN ora_registrazione SET DEFAULT CURRENT_TIMESTAMP;

CREATE TABLE scope_init_seq (id BIGINT);
INSERT INTO scope_init_seq VALUES (NEXT VALUE FOR seq_scope);




-- govway
		  		
-- **** Protocol Properties ****

CREATE SEQUENCE seq_protocol_properties AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE protocol_properties
(
	-- tipoProprietario
	tipo_proprietario VARCHAR(255) NOT NULL,
	-- idOggettoProprietarioDocumento
	id_proprietario BIGINT NOT NULL,
	-- nome property
	name VARCHAR(255) NOT NULL,
	-- valore come stringa
	value_string VARCHAR(4000),
	-- valore come stringa cifrato
	enc_value_string VARCHAR(65535),
	-- valore come numero
	value_number BIGINT,
	-- valore true o false
	value_boolean INT,
	-- valore binario
	value_binary VARBINARY(16777215),
	file_name VARCHAR(4000),
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	-- unique constraints
	CONSTRAINT unique_protocol_properties_1 UNIQUE (tipo_proprietario,id_proprietario,name),
	-- fk/pk keys constraints
	CONSTRAINT pk_protocol_properties PRIMARY KEY (id)
);

-- index
CREATE UNIQUE INDEX index_protocol_properties_1 ON protocol_properties (tipo_proprietario,id_proprietario,name);
CREATE TABLE protocol_properties_init_seq (id BIGINT);
INSERT INTO protocol_properties_init_seq VALUES (NEXT VALUE FOR seq_protocol_properties);




-- govway
		  		
-- **** Soggetti ****

CREATE SEQUENCE seq_soggetti AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE soggetti
(
	nome_soggetto VARCHAR(255) NOT NULL,
	tipo_soggetto VARCHAR(255) NOT NULL,
	descrizione VARCHAR(4000),
	identificativo_porta VARCHAR(255),
	-- 1/0 (true/false) Indicazione se il soggetto svolge è quello di default per il protocollo
	is_default INT,
	-- 1/0 (true/false) svolge attivita di router
	is_router INT,
	id_connettore BIGINT NOT NULL,
	superuser VARCHAR(255),
	server VARCHAR(255),
	-- 1/0 (true/false) indica se il soggetto e' privato/pubblico
	privato INT,
	ora_registrazione TIMESTAMP,
	profilo VARCHAR(255),
	codice_ipa VARCHAR(255) NOT NULL,
	tipoauth VARCHAR(255),
	utente VARCHAR(2800),
	password VARCHAR(255),
	subject VARCHAR(2800),
	cn_subject VARCHAR(255),
	issuer VARCHAR(2800),
	cn_issuer VARCHAR(255),
	certificate VARBINARY(16777215),
	cert_strict_verification INT,
	utente_richiedente VARCHAR(255),
	data_creazione TIMESTAMP,
	utente_ultima_modifica VARCHAR(255),
	data_ultima_modifica TIMESTAMP,
	pd_url_prefix_rewriter VARCHAR(255),
	pa_url_prefix_rewriter VARCHAR(255),
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	-- unique constraints
	CONSTRAINT unique_soggetti_1 UNIQUE (nome_soggetto,tipo_soggetto),
	CONSTRAINT unique_soggetti_2 UNIQUE (codice_ipa),
	-- fk/pk keys constraints
	CONSTRAINT fk_soggetti_1 FOREIGN KEY (id_connettore) REFERENCES connettori(id),
	CONSTRAINT pk_soggetti PRIMARY KEY (id)
);

-- index
CREATE UNIQUE INDEX index_soggetti_1 ON soggetti (nome_soggetto,tipo_soggetto);
CREATE UNIQUE INDEX index_soggetti_2 ON soggetti (codice_ipa);

ALTER TABLE soggetti ALTER COLUMN is_default SET DEFAULT 0;
ALTER TABLE soggetti ALTER COLUMN is_router SET DEFAULT 0;
ALTER TABLE soggetti ALTER COLUMN privato SET DEFAULT 0;
ALTER TABLE soggetti ALTER COLUMN ora_registrazione SET DEFAULT CURRENT_TIMESTAMP;

CREATE TABLE soggetti_init_seq (id BIGINT);
INSERT INTO soggetti_init_seq VALUES (NEXT VALUE FOR seq_soggetti);



CREATE SEQUENCE seq_soggetti_ruoli AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE soggetti_ruoli
(
	id_soggetto BIGINT NOT NULL,
	id_ruolo BIGINT NOT NULL,
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	-- unique constraints
	CONSTRAINT unique_soggetti_ruoli_1 UNIQUE (id_soggetto,id_ruolo),
	-- fk/pk keys constraints
	CONSTRAINT fk_soggetti_ruoli_1 FOREIGN KEY (id_soggetto) REFERENCES soggetti(id),
	CONSTRAINT fk_soggetti_ruoli_2 FOREIGN KEY (id_ruolo) REFERENCES ruoli(id),
	CONSTRAINT pk_soggetti_ruoli PRIMARY KEY (id)
);

-- index
CREATE UNIQUE INDEX index_soggetti_ruoli_1 ON soggetti_ruoli (id_soggetto,id_ruolo);
CREATE INDEX INDEX_SOGGETTI_RUOLI ON soggetti_ruoli (id_soggetto);
CREATE TABLE soggetti_ruoli_init_seq (id BIGINT);
INSERT INTO soggetti_ruoli_init_seq VALUES (NEXT VALUE FOR seq_soggetti_ruoli);



CREATE SEQUENCE seq_soggetti_credenziali AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE soggetti_credenziali
(
	id_soggetto BIGINT NOT NULL,
	subject VARCHAR(2800),
	cn_subject VARCHAR(255),
	issuer VARCHAR(2800),
	cn_issuer VARCHAR(255),
	certificate VARBINARY(16777215),
	cert_strict_verification INT,
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	-- fk/pk keys constraints
	CONSTRAINT fk_soggetti_credenziali_1 FOREIGN KEY (id_soggetto) REFERENCES soggetti(id),
	CONSTRAINT pk_soggetti_credenziali PRIMARY KEY (id)
);

-- index
CREATE INDEX INDEX_SOGGETTI_CRED ON soggetti_credenziali (id_soggetto);
CREATE TABLE soggetti_credenziali_init_seq (id BIGINT);
INSERT INTO soggetti_credenziali_init_seq VALUES (NEXT VALUE FOR seq_soggetti_credenziali);



CREATE SEQUENCE seq_soggetti_properties AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE soggetti_properties
(
	id_soggetto BIGINT NOT NULL,
	nome VARCHAR(255) NOT NULL,
	valore VARCHAR(4000) NOT NULL,
	enc_value VARCHAR(65535),
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	-- unique constraints
	CONSTRAINT uniq_soggetti_props_1 UNIQUE (id_soggetto,nome),
	-- fk/pk keys constraints
	CONSTRAINT fk_soggetti_properties_1 FOREIGN KEY (id_soggetto) REFERENCES soggetti(id),
	CONSTRAINT pk_soggetti_properties PRIMARY KEY (id)
);

-- index
CREATE INDEX INDEX_SOGGETTI_PROP ON soggetti_properties (id_soggetto);
CREATE TABLE soggetti_properties_init_seq (id BIGINT);
INSERT INTO soggetti_properties_init_seq VALUES (NEXT VALUE FOR seq_soggetti_properties);




-- govway
		  		
-- **** Documenti ****

CREATE SEQUENCE seq_documenti AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE documenti
(
	ruolo VARCHAR(255) NOT NULL,
	-- tipo (es. xsd,xml...)
	tipo VARCHAR(255) NOT NULL,
	-- nome documento
	nome VARCHAR(255) NOT NULL,
	-- contenuto documento
	contenuto VARBINARY(16777215) NOT NULL,
	-- idOggettoProprietarioDocumento
	id_proprietario BIGINT NOT NULL,
	-- tipoProprietario
	tipo_proprietario VARCHAR(255) NOT NULL,
	ora_registrazione TIMESTAMP NOT NULL,
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	-- check constraints
	CONSTRAINT chk_documenti_1 CHECK (ruolo IN ('allegato','specificaSemiformale','specificaLivelloServizio','specificaSicurezza','specificaCoordinamento')),
	CONSTRAINT chk_documenti_2 CHECK (tipo_proprietario IN ('accordoServizio','accordoCooperazione','servizio')),
	-- unique constraints
	CONSTRAINT unique_documenti_1 UNIQUE (ruolo,tipo,nome,id_proprietario,tipo_proprietario),
	-- fk/pk keys constraints
	CONSTRAINT pk_documenti PRIMARY KEY (id)
);

-- index
CREATE UNIQUE INDEX index_documenti_1 ON documenti (ruolo,tipo,nome,id_proprietario,tipo_proprietario);
CREATE INDEX INDEX_DOC_SEARCH ON documenti (id_proprietario);

ALTER TABLE documenti ALTER COLUMN ora_registrazione SET DEFAULT CURRENT_TIMESTAMP;

CREATE TABLE documenti_init_seq (id BIGINT);
INSERT INTO documenti_init_seq VALUES (NEXT VALUE FOR seq_documenti);



-- **** Accordi di Servizio Parte Comune ****

CREATE SEQUENCE seq_accordi AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE accordi
(
	nome VARCHAR(255) NOT NULL,
	descrizione VARCHAR(4000),
	service_binding VARCHAR(255) NOT NULL,
	message_type VARCHAR(255),
	profilo_collaborazione VARCHAR(255),
	formato_specifica VARCHAR(255),
	wsdl_definitorio LONGVARCHAR,
	wsdl_concettuale LONGVARCHAR,
	wsdl_logico_erogatore LONGVARCHAR,
	wsdl_logico_fruitore LONGVARCHAR,
	spec_conv_concettuale LONGVARCHAR,
	spec_conv_erogatore LONGVARCHAR,
	spec_conv_fruitore LONGVARCHAR,
	filtro_duplicati VARCHAR(255),
	conferma_ricezione VARCHAR(255),
	identificativo_collaborazione VARCHAR(255),
	id_riferimento_richiesta VARCHAR(255),
	consegna_in_ordine VARCHAR(255),
	scadenza VARCHAR(255),
	superuser VARCHAR(255),
	-- id del soggetto referente
	id_referente BIGINT,
	versione INT,
	-- 1/0 (vero/falso) indica se questo accordo di servizio e' utilizzabile in mancanza di azioni associate
	utilizzo_senza_azione INT,
	-- 1/0 (true/false) indica se il soggetto e' privato/pubblico
	privato INT,
	ora_registrazione TIMESTAMP,
	stato VARCHAR(255) NOT NULL,
	canale VARCHAR(255),
	utente_richiedente VARCHAR(255),
	data_creazione TIMESTAMP,
	utente_ultima_modifica VARCHAR(255),
	data_ultima_modifica TIMESTAMP,
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	-- check constraints
	CONSTRAINT chk_accordi_1 CHECK (service_binding IN ('soap','rest')),
	CONSTRAINT chk_accordi_2 CHECK (stato IN ('finale','bozza','operativo')),
	-- unique constraints
	CONSTRAINT unique_accordi_1 UNIQUE (nome,id_referente,versione),
	-- fk/pk keys constraints
	CONSTRAINT pk_accordi PRIMARY KEY (id)
);

-- index
CREATE UNIQUE INDEX index_accordi_1 ON accordi (nome,id_referente,versione);
CREATE INDEX index_accordi_2 ON accordi (canale);

ALTER TABLE accordi ALTER COLUMN id_referente SET DEFAULT 0;
ALTER TABLE accordi ALTER COLUMN versione SET DEFAULT 1;
ALTER TABLE accordi ALTER COLUMN utilizzo_senza_azione SET DEFAULT 1;
ALTER TABLE accordi ALTER COLUMN privato SET DEFAULT 0;
ALTER TABLE accordi ALTER COLUMN ora_registrazione SET DEFAULT CURRENT_TIMESTAMP;
ALTER TABLE accordi ALTER COLUMN stato SET DEFAULT 'finale';

CREATE TABLE accordi_init_seq (id BIGINT);
INSERT INTO accordi_init_seq VALUES (NEXT VALUE FOR seq_accordi);



CREATE SEQUENCE seq_accordi_azioni AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE accordi_azioni
(
	id_accordo BIGINT NOT NULL,
	nome VARCHAR(255) NOT NULL,
	profilo_collaborazione VARCHAR(255),
	filtro_duplicati VARCHAR(255),
	conferma_ricezione VARCHAR(255),
	identificativo_collaborazione VARCHAR(255),
	id_riferimento_richiesta VARCHAR(255),
	consegna_in_ordine VARCHAR(255),
	scadenza VARCHAR(255),
	correlata VARCHAR(255),
	-- ridefinito/default
	profilo_azione VARCHAR(255),
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	-- unique constraints
	CONSTRAINT unique_accordi_azioni_1 UNIQUE (id_accordo,nome),
	-- fk/pk keys constraints
	CONSTRAINT fk_accordi_azioni_1 FOREIGN KEY (id_accordo) REFERENCES accordi(id),
	CONSTRAINT pk_accordi_azioni PRIMARY KEY (id)
);

-- index
CREATE UNIQUE INDEX index_accordi_azioni_1 ON accordi_azioni (id_accordo,nome);
CREATE TABLE accordi_azioni_init_seq (id BIGINT);
INSERT INTO accordi_azioni_init_seq VALUES (NEXT VALUE FOR seq_accordi_azioni);



CREATE SEQUENCE seq_port_type AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE port_type
(
	id_accordo BIGINT NOT NULL,
	nome VARCHAR(255) NOT NULL,
	descrizione VARCHAR(255),
	profilo_collaborazione VARCHAR(255),
	filtro_duplicati VARCHAR(255),
	conferma_ricezione VARCHAR(255),
	identificativo_collaborazione VARCHAR(255),
	id_riferimento_richiesta VARCHAR(255),
	consegna_in_ordine VARCHAR(255),
	scadenza VARCHAR(255),
	-- ridefinito/default
	profilo_pt VARCHAR(255),
	-- document/RPC
	soap_style VARCHAR(255),
	message_type VARCHAR(255),
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	-- unique constraints
	CONSTRAINT unique_port_type_1 UNIQUE (id_accordo,nome),
	-- fk/pk keys constraints
	CONSTRAINT fk_port_type_1 FOREIGN KEY (id_accordo) REFERENCES accordi(id),
	CONSTRAINT pk_port_type PRIMARY KEY (id)
);

-- index
CREATE UNIQUE INDEX index_port_type_1 ON port_type (id_accordo,nome);
CREATE TABLE port_type_init_seq (id BIGINT);
INSERT INTO port_type_init_seq VALUES (NEXT VALUE FOR seq_port_type);



CREATE SEQUENCE seq_port_type_azioni AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE port_type_azioni
(
	id_port_type BIGINT NOT NULL,
	nome VARCHAR(255) NOT NULL,
	profilo_collaborazione VARCHAR(255),
	filtro_duplicati VARCHAR(255),
	conferma_ricezione VARCHAR(255),
	identificativo_collaborazione VARCHAR(255),
	id_riferimento_richiesta VARCHAR(255),
	consegna_in_ordine VARCHAR(255),
	scadenza VARCHAR(255),
	correlata_servizio VARCHAR(255),
	correlata VARCHAR(255),
	-- ridefinito/default
	profilo_pt_azione VARCHAR(255),
	-- document/rpc
	soap_style VARCHAR(255),
	soap_action VARCHAR(255),
	-- literal/encoded
	soap_use_msg_input VARCHAR(255),
	-- namespace utilizzato per RPC
	soap_namespace_msg_input VARCHAR(255),
	-- literal/encoded
	soap_use_msg_output VARCHAR(255),
	-- namespace utilizzato per RPC
	soap_namespace_msg_output VARCHAR(255),
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	-- unique constraints
	CONSTRAINT unique_port_type_azioni_1 UNIQUE (id_port_type,nome),
	-- fk/pk keys constraints
	CONSTRAINT fk_port_type_azioni_1 FOREIGN KEY (id_port_type) REFERENCES port_type(id),
	CONSTRAINT pk_port_type_azioni PRIMARY KEY (id)
);

-- index
CREATE UNIQUE INDEX index_port_type_azioni_1 ON port_type_azioni (id_port_type,nome);
CREATE TABLE port_type_azioni_init_seq (id BIGINT);
INSERT INTO port_type_azioni_init_seq VALUES (NEXT VALUE FOR seq_port_type_azioni);



CREATE SEQUENCE seq_operation_messages AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE operation_messages
(
	id_port_type_azione BIGINT NOT NULL,
	-- true(1)/false(0), true indica un input-message, false un output-message
	input_message INT,
	name VARCHAR(255) NOT NULL,
	element_name VARCHAR(255),
	element_namespace VARCHAR(255),
	type_name VARCHAR(255),
	type_namespace VARCHAR(255),
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	-- fk/pk keys constraints
	CONSTRAINT fk_operation_messages_1 FOREIGN KEY (id_port_type_azione) REFERENCES port_type_azioni(id),
	CONSTRAINT pk_operation_messages PRIMARY KEY (id)
);

-- index
CREATE INDEX INDEX_OP_MESSAGES ON operation_messages (id_port_type_azione,input_message);

ALTER TABLE operation_messages ALTER COLUMN input_message SET DEFAULT 1;

CREATE TABLE operation_messages_init_seq (id BIGINT);
INSERT INTO operation_messages_init_seq VALUES (NEXT VALUE FOR seq_operation_messages);



CREATE SEQUENCE seq_api_resources AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE api_resources
(
	id_accordo BIGINT NOT NULL,
	nome VARCHAR(255) NOT NULL,
	descrizione VARCHAR(255),
	http_method VARCHAR(255) NOT NULL,
	path VARCHAR(255) NOT NULL,
	message_type VARCHAR(255),
	message_type_request VARCHAR(255),
	message_type_response VARCHAR(255),
	filtro_duplicati VARCHAR(255),
	conferma_ricezione VARCHAR(255),
	identificativo_collaborazione VARCHAR(255),
	id_riferimento_richiesta VARCHAR(255),
	consegna_in_ordine VARCHAR(255),
	scadenza VARCHAR(255),
	-- ridefinito/default
	profilo_azione VARCHAR(255),
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	-- unique constraints
	CONSTRAINT unique_api_resources_1 UNIQUE (id_accordo,nome),
	CONSTRAINT unique_api_resources_2 UNIQUE (id_accordo,http_method,path),
	-- fk/pk keys constraints
	CONSTRAINT fk_api_resources_1 FOREIGN KEY (id_accordo) REFERENCES accordi(id),
	CONSTRAINT pk_api_resources PRIMARY KEY (id)
);

-- index
CREATE UNIQUE INDEX index_api_resources_1 ON api_resources (id_accordo,nome);
CREATE UNIQUE INDEX index_api_resources_2 ON api_resources (id_accordo,http_method,path);
CREATE TABLE api_resources_init_seq (id BIGINT);
INSERT INTO api_resources_init_seq VALUES (NEXT VALUE FOR seq_api_resources);



CREATE SEQUENCE seq_api_resources_response AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE api_resources_response
(
	id_resource BIGINT NOT NULL,
	descrizione VARCHAR(255),
	status INT NOT NULL,
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	-- unique constraints
	CONSTRAINT uniq_api_resp_1 UNIQUE (id_resource,status),
	-- fk/pk keys constraints
	CONSTRAINT fk_api_resources_response_1 FOREIGN KEY (id_resource) REFERENCES api_resources(id),
	CONSTRAINT pk_api_resources_response PRIMARY KEY (id)
);

-- index
CREATE UNIQUE INDEX index_api_resp_1 ON api_resources_response (id_resource,status);
CREATE TABLE api_resources_response_init_seq (id BIGINT);
INSERT INTO api_resources_response_init_seq VALUES (NEXT VALUE FOR seq_api_resources_response);



CREATE SEQUENCE seq_api_resources_media AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE api_resources_media
(
	id_resource_media BIGINT,
	id_resource_response_media BIGINT,
	media_type VARCHAR(255) NOT NULL,
	message_type VARCHAR(255),
	nome VARCHAR(255),
	descrizione VARCHAR(255),
	tipo VARCHAR(255),
	xml_tipo VARCHAR(255),
	xml_name VARCHAR(255),
	xml_namespace VARCHAR(255),
	json_type VARCHAR(255),
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	-- fk/pk keys constraints
	CONSTRAINT fk_api_resources_media_1 FOREIGN KEY (id_resource_response_media) REFERENCES api_resources_response(id),
	CONSTRAINT fk_api_resources_media_2 FOREIGN KEY (id_resource_media) REFERENCES api_resources(id),
	CONSTRAINT pk_api_resources_media PRIMARY KEY (id)
);

CREATE TABLE api_resources_media_init_seq (id BIGINT);
INSERT INTO api_resources_media_init_seq VALUES (NEXT VALUE FOR seq_api_resources_media);



CREATE SEQUENCE seq_api_resources_parameter AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE api_resources_parameter
(
	id_resource_parameter BIGINT,
	id_resource_response_par BIGINT,
	nome VARCHAR(255) NOT NULL,
	descrizione VARCHAR(255),
	tipo_parametro VARCHAR(255) NOT NULL,
	required BOOLEAN NOT NULL,
	tipo VARCHAR(255) NOT NULL,
	restrizioni VARCHAR(65535),
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	-- fk/pk keys constraints
	CONSTRAINT fk_api_resources_parameter_1 FOREIGN KEY (id_resource_response_par) REFERENCES api_resources_response(id),
	CONSTRAINT fk_api_resources_parameter_2 FOREIGN KEY (id_resource_parameter) REFERENCES api_resources(id),
	CONSTRAINT pk_api_resources_parameter PRIMARY KEY (id)
);


ALTER TABLE api_resources_parameter ALTER COLUMN required SET DEFAULT false;

CREATE TABLE api_resources_parameter_init_seq (id BIGINT);
INSERT INTO api_resources_parameter_init_seq VALUES (NEXT VALUE FOR seq_api_resources_parameter);



CREATE SEQUENCE seq_accordi_gruppi AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE accordi_gruppi
(
	id_accordo BIGINT NOT NULL,
	id_gruppo BIGINT NOT NULL,
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	-- unique constraints
	CONSTRAINT unique_acc_gruppi_1 UNIQUE (id_accordo,id_gruppo),
	-- fk/pk keys constraints
	CONSTRAINT fk_accordi_gruppi_1 FOREIGN KEY (id_gruppo) REFERENCES gruppi(id),
	CONSTRAINT fk_accordi_gruppi_2 FOREIGN KEY (id_accordo) REFERENCES accordi(id),
	CONSTRAINT pk_accordi_gruppi PRIMARY KEY (id)
);

-- index
CREATE UNIQUE INDEX idx_acc_gruppi_1 ON accordi_gruppi (id_accordo,id_gruppo);
CREATE TABLE accordi_gruppi_init_seq (id BIGINT);
INSERT INTO accordi_gruppi_init_seq VALUES (NEXT VALUE FOR seq_accordi_gruppi);



-- **** Accordi di Cooperazione ****

CREATE SEQUENCE seq_accordi_cooperazione AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE accordi_cooperazione
(
	nome VARCHAR(255) NOT NULL,
	descrizione VARCHAR(255),
	-- id del soggetto referente
	id_referente BIGINT,
	versione INT,
	stato VARCHAR(255) NOT NULL,
	-- 1/0 (true/false) indica se il soggetto e' privato/pubblico
	privato INT,
	superuser VARCHAR(255),
	ora_registrazione TIMESTAMP,
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	-- check constraints
	CONSTRAINT chk_accordi_cooperazione_1 CHECK (stato IN ('finale','bozza','operativo')),
	-- unique constraints
	CONSTRAINT unique_accordi_cooperazione_1 UNIQUE (nome,id_referente,versione),
	-- fk/pk keys constraints
	CONSTRAINT pk_accordi_cooperazione PRIMARY KEY (id)
);

-- index
CREATE UNIQUE INDEX index_accordi_cooperazione_1 ON accordi_cooperazione (nome,id_referente,versione);

ALTER TABLE accordi_cooperazione ALTER COLUMN id_referente SET DEFAULT 0;
ALTER TABLE accordi_cooperazione ALTER COLUMN versione SET DEFAULT 1;
ALTER TABLE accordi_cooperazione ALTER COLUMN stato SET DEFAULT 'finale';
ALTER TABLE accordi_cooperazione ALTER COLUMN privato SET DEFAULT 0;
ALTER TABLE accordi_cooperazione ALTER COLUMN ora_registrazione SET DEFAULT CURRENT_TIMESTAMP;

CREATE TABLE accordi_cooperazione_init_seq (id BIGINT);
INSERT INTO accordi_cooperazione_init_seq VALUES (NEXT VALUE FOR seq_accordi_cooperazione);



CREATE SEQUENCE seq_accordi_coop_partecipanti AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE accordi_coop_partecipanti
(
	id_accordo_cooperazione BIGINT NOT NULL,
	id_soggetto BIGINT NOT NULL,
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	-- unique constraints
	CONSTRAINT unique_acc_coop_part_1 UNIQUE (id_accordo_cooperazione,id_soggetto),
	-- fk/pk keys constraints
	CONSTRAINT fk_accordi_coop_partecipanti_1 FOREIGN KEY (id_soggetto) REFERENCES soggetti(id),
	CONSTRAINT fk_accordi_coop_partecipanti_2 FOREIGN KEY (id_accordo_cooperazione) REFERENCES accordi_cooperazione(id),
	CONSTRAINT pk_accordi_coop_partecipanti PRIMARY KEY (id)
);

-- index
CREATE INDEX INDEX_AC_COOP_PAR ON accordi_coop_partecipanti (id_accordo_cooperazione);
CREATE INDEX INDEX_AC_SOG ON accordi_coop_partecipanti (id_soggetto);
CREATE TABLE accordi_coop_partecipanti_init_seq (id BIGINT);
INSERT INTO accordi_coop_partecipanti_init_seq VALUES (NEXT VALUE FOR seq_accordi_coop_partecipanti);



-- **** Accordi di Servizio Parte Specifica ****

CREATE SEQUENCE seq_servizi AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE servizi
(
	nome_servizio VARCHAR(255) NOT NULL,
	tipo_servizio VARCHAR(255) NOT NULL,
	versione_servizio INT,
	id_soggetto BIGINT NOT NULL,
	id_accordo BIGINT NOT NULL,
	servizio_correlato VARCHAR(255),
	id_connettore BIGINT NOT NULL,
	wsdl_implementativo_erogatore LONGVARCHAR,
	wsdl_implementativo_fruitore LONGVARCHAR,
	superuser VARCHAR(255),
	-- 1/0 (true/false) indica se il soggetto e' privato/pubblico
	privato INT,
	ora_registrazione TIMESTAMP,
	port_type VARCHAR(255),
	profilo VARCHAR(255),
	descrizione VARCHAR(4000),
	stato VARCHAR(255) NOT NULL,
	message_type VARCHAR(255),
	utente_richiedente VARCHAR(255),
	data_creazione TIMESTAMP,
	utente_ultima_modifica VARCHAR(255),
	data_ultima_modifica TIMESTAMP,
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	-- check constraints
	CONSTRAINT chk_servizi_1 CHECK (stato IN ('finale','bozza','operativo')),
	-- unique constraints
	CONSTRAINT unique_servizi_1 UNIQUE (id_soggetto,tipo_servizio,nome_servizio,versione_servizio),
	-- fk/pk keys constraints
	CONSTRAINT fk_servizi_1 FOREIGN KEY (id_connettore) REFERENCES connettori(id),
	CONSTRAINT fk_servizi_2 FOREIGN KEY (id_soggetto) REFERENCES soggetti(id),
	CONSTRAINT fk_servizi_3 FOREIGN KEY (id_accordo) REFERENCES accordi(id),
	CONSTRAINT pk_servizi PRIMARY KEY (id)
);

-- index
CREATE UNIQUE INDEX index_servizi_1 ON servizi (id_soggetto,tipo_servizio,nome_servizio,versione_servizio);
CREATE INDEX INDEX_SERV_ACC ON servizi (id_accordo);
CREATE INDEX INDEX_SERV_SOG ON servizi (id_soggetto);

ALTER TABLE servizi ALTER COLUMN versione_servizio SET DEFAULT 1;
ALTER TABLE servizi ALTER COLUMN privato SET DEFAULT 0;
ALTER TABLE servizi ALTER COLUMN ora_registrazione SET DEFAULT CURRENT_TIMESTAMP;
ALTER TABLE servizi ALTER COLUMN stato SET DEFAULT 'finale';

CREATE TABLE servizi_init_seq (id BIGINT);
INSERT INTO servizi_init_seq VALUES (NEXT VALUE FOR seq_servizi);



CREATE SEQUENCE seq_servizi_azioni AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE servizi_azioni
(
	id_servizio BIGINT NOT NULL,
	id_connettore BIGINT NOT NULL,
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	-- fk/pk keys constraints
	CONSTRAINT fk_servizi_azioni_1 FOREIGN KEY (id_connettore) REFERENCES connettori(id),
	CONSTRAINT fk_servizi_azioni_2 FOREIGN KEY (id_servizio) REFERENCES servizi(id),
	CONSTRAINT pk_servizi_azioni PRIMARY KEY (id)
);

-- index
CREATE INDEX index_servizi_azioni_1 ON servizi_azioni (id_servizio);
CREATE TABLE servizi_azioni_init_seq (id BIGINT);
INSERT INTO servizi_azioni_init_seq VALUES (NEXT VALUE FOR seq_servizi_azioni);



CREATE SEQUENCE seq_servizi_azione AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE servizi_azione
(
	nome_azione VARCHAR(255) NOT NULL,
	id_servizio_azioni BIGINT NOT NULL,
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	-- unique constraints
	CONSTRAINT unique_servizi_azione_1 UNIQUE (nome_azione,id_servizio_azioni),
	-- fk/pk keys constraints
	CONSTRAINT fk_servizi_azione_1 FOREIGN KEY (id_servizio_azioni) REFERENCES servizi_azioni(id),
	CONSTRAINT pk_servizi_azione PRIMARY KEY (id)
);

-- index
CREATE UNIQUE INDEX index_servizi_azione_1 ON servizi_azione (nome_azione,id_servizio_azioni);
CREATE TABLE servizi_azione_init_seq (id BIGINT);
INSERT INTO servizi_azione_init_seq VALUES (NEXT VALUE FOR seq_servizi_azione);



CREATE SEQUENCE seq_servizi_fruitori AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE servizi_fruitori
(
	id_servizio BIGINT NOT NULL,
	id_soggetto BIGINT NOT NULL,
	id_connettore BIGINT NOT NULL,
	wsdl_implementativo_erogatore LONGVARCHAR,
	wsdl_implementativo_fruitore LONGVARCHAR,
	ora_registrazione TIMESTAMP,
	stato VARCHAR(255) NOT NULL,
	descrizione VARCHAR(4000),
	utente_richiedente VARCHAR(255),
	data_creazione TIMESTAMP,
	utente_ultima_modifica VARCHAR(255),
	data_ultima_modifica TIMESTAMP,
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	-- check constraints
	CONSTRAINT chk_servizi_fruitori_1 CHECK (stato IN ('finale','bozza','operativo')),
	-- unique constraints
	CONSTRAINT unique_servizi_fruitori_1 UNIQUE (id_servizio,id_soggetto),
	-- fk/pk keys constraints
	CONSTRAINT fk_servizi_fruitori_1 FOREIGN KEY (id_connettore) REFERENCES connettori(id),
	CONSTRAINT fk_servizi_fruitori_2 FOREIGN KEY (id_soggetto) REFERENCES soggetti(id),
	CONSTRAINT fk_servizi_fruitori_3 FOREIGN KEY (id_servizio) REFERENCES servizi(id),
	CONSTRAINT pk_servizi_fruitori PRIMARY KEY (id)
);

-- index
CREATE UNIQUE INDEX index_servizi_fruitori_1 ON servizi_fruitori (id_servizio,id_soggetto);
CREATE INDEX INDEX_SERV_FRU_SOG ON servizi_fruitori (id_soggetto);

ALTER TABLE servizi_fruitori ALTER COLUMN ora_registrazione SET DEFAULT CURRENT_TIMESTAMP;
ALTER TABLE servizi_fruitori ALTER COLUMN stato SET DEFAULT 'finale';

CREATE TABLE servizi_fruitori_init_seq (id BIGINT);
INSERT INTO servizi_fruitori_init_seq VALUES (NEXT VALUE FOR seq_servizi_fruitori);



CREATE SEQUENCE seq_servizi_fruitori_azioni AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE servizi_fruitori_azioni
(
	id_fruizione BIGINT NOT NULL,
	id_connettore BIGINT NOT NULL,
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	-- fk/pk keys constraints
	CONSTRAINT fk_servizi_fruitori_azioni_1 FOREIGN KEY (id_connettore) REFERENCES connettori(id),
	CONSTRAINT fk_servizi_fruitori_azioni_2 FOREIGN KEY (id_fruizione) REFERENCES servizi_fruitori(id),
	CONSTRAINT pk_servizi_fruitori_azioni PRIMARY KEY (id)
);

-- index
CREATE INDEX index_serv_fru_azioni_1 ON servizi_fruitori_azioni (id_fruizione);
CREATE TABLE servizi_fruitori_azioni_init_seq (id BIGINT);
INSERT INTO servizi_fruitori_azioni_init_seq VALUES (NEXT VALUE FOR seq_servizi_fruitori_azioni);



CREATE SEQUENCE seq_servizi_fruitori_azione AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE servizi_fruitori_azione
(
	nome_azione VARCHAR(255) NOT NULL,
	id_fruizione_azioni BIGINT NOT NULL,
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	-- unique constraints
	CONSTRAINT uniq_serv_fru_azione_1 UNIQUE (nome_azione,id_fruizione_azioni),
	-- fk/pk keys constraints
	CONSTRAINT fk_servizi_fruitori_azione_1 FOREIGN KEY (id_fruizione_azioni) REFERENCES servizi_fruitori_azioni(id),
	CONSTRAINT pk_servizi_fruitori_azione PRIMARY KEY (id)
);

-- index
CREATE UNIQUE INDEX index_serv_fru_azione_1 ON servizi_fruitori_azione (nome_azione,id_fruizione_azioni);
CREATE TABLE servizi_fruitori_azione_init_seq (id BIGINT);
INSERT INTO servizi_fruitori_azione_init_seq VALUES (NEXT VALUE FOR seq_servizi_fruitori_azione);



-- **** Accordi di Servizio Composti ****

CREATE SEQUENCE seq_acc_serv_composti AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE acc_serv_composti
(
	id_accordo BIGINT NOT NULL,
	id_accordo_cooperazione BIGINT NOT NULL,
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	-- unique constraints
	CONSTRAINT unique_acc_serv_composti_1 UNIQUE (id_accordo),
	-- fk/pk keys constraints
	CONSTRAINT fk_acc_serv_composti_1 FOREIGN KEY (id_accordo_cooperazione) REFERENCES accordi_cooperazione(id),
	CONSTRAINT fk_acc_serv_composti_2 FOREIGN KEY (id_accordo) REFERENCES accordi(id),
	CONSTRAINT pk_acc_serv_composti PRIMARY KEY (id)
);

-- index
CREATE UNIQUE INDEX index_acc_serv_composti_1 ON acc_serv_composti (id_accordo);
CREATE INDEX INDEX_AC_SC ON acc_serv_composti (id_accordo_cooperazione);
CREATE TABLE acc_serv_composti_init_seq (id BIGINT);
INSERT INTO acc_serv_composti_init_seq VALUES (NEXT VALUE FOR seq_acc_serv_composti);



CREATE SEQUENCE seq_acc_serv_componenti AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE acc_serv_componenti
(
	id_servizio_composto BIGINT NOT NULL,
	id_servizio_componente BIGINT NOT NULL,
	azione VARCHAR(255),
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	-- fk/pk keys constraints
	CONSTRAINT fk_acc_serv_componenti_1 FOREIGN KEY (id_servizio_composto) REFERENCES acc_serv_composti(id),
	CONSTRAINT fk_acc_serv_componenti_2 FOREIGN KEY (id_servizio_componente) REFERENCES servizi(id),
	CONSTRAINT pk_acc_serv_componenti PRIMARY KEY (id)
);

-- index
CREATE INDEX INDEX_AC_SC_SC ON acc_serv_componenti (id_servizio_composto);
CREATE INDEX INDEX_AC_SC_SERV ON acc_serv_componenti (id_servizio_componente);
CREATE TABLE acc_serv_componenti_init_seq (id BIGINT);
INSERT INTO acc_serv_componenti_init_seq VALUES (NEXT VALUE FOR seq_acc_serv_componenti);




-- govway
		  		
-- PLUGINS

CREATE SEQUENCE seq_plugins AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE plugins
(
	tipo_plugin VARCHAR(255) NOT NULL,
	class_name VARCHAR(255) NOT NULL,
	tipo VARCHAR(255) NOT NULL,
	descrizione VARCHAR(255),
	label VARCHAR(255) NOT NULL,
	stato BOOLEAN,
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	-- unique constraints
	CONSTRAINT unique_plugins_1 UNIQUE (tipo_plugin,class_name),
	CONSTRAINT unique_plugins_2 UNIQUE (tipo_plugin,tipo),
	CONSTRAINT unique_plugins_3 UNIQUE (tipo_plugin,label),
	-- fk/pk keys constraints
	CONSTRAINT pk_plugins PRIMARY KEY (id)
);

-- index
CREATE UNIQUE INDEX index_plugins_1 ON plugins (tipo_plugin,class_name);
CREATE UNIQUE INDEX index_plugins_2 ON plugins (tipo_plugin,tipo);
CREATE UNIQUE INDEX index_plugins_3 ON plugins (tipo_plugin,label);

ALTER TABLE plugins ALTER COLUMN stato SET DEFAULT true;

CREATE TABLE plugins_init_seq (id BIGINT);
INSERT INTO plugins_init_seq VALUES (NEXT VALUE FOR seq_plugins);



CREATE SEQUENCE seq_plugins_servizi_comp AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE plugins_servizi_comp
(
	-- Accordo di Servizio
	uri_accordo VARCHAR(255),
	-- Nome del port-type
	servizio VARCHAR(255),
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	id_plugin BIGINT NOT NULL,
	-- fk/pk keys constraints
	CONSTRAINT fk_plugins_servizi_comp_1 FOREIGN KEY (id_plugin) REFERENCES plugins(id) ON DELETE CASCADE,
	CONSTRAINT pk_plugins_servizi_comp PRIMARY KEY (id)
);

-- index
CREATE INDEX idx_plug_ser_com_1 ON plugins_servizi_comp (id_plugin);
CREATE TABLE plugins_servizi_comp_init_seq (id BIGINT);
INSERT INTO plugins_servizi_comp_init_seq VALUES (NEXT VALUE FOR seq_plugins_servizi_comp);



CREATE SEQUENCE seq_plugins_azioni_comp AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE plugins_azioni_comp
(
	azione VARCHAR(255) NOT NULL,
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	id_plugin_servizio_comp BIGINT NOT NULL,
	-- unique constraints
	CONSTRAINT unique_plugins_azioni_comp_1 UNIQUE (id_plugin_servizio_comp,azione),
	-- fk/pk keys constraints
	CONSTRAINT fk_plugins_azioni_comp_1 FOREIGN KEY (id_plugin_servizio_comp) REFERENCES plugins_servizi_comp(id) ON DELETE CASCADE,
	CONSTRAINT pk_plugins_azioni_comp PRIMARY KEY (id)
);

-- index
CREATE UNIQUE INDEX index_plugins_azioni_comp_1 ON plugins_azioni_comp (id_plugin_servizio_comp,azione);
CREATE TABLE plugins_azioni_comp_init_seq (id BIGINT);
INSERT INTO plugins_azioni_comp_init_seq VALUES (NEXT VALUE FOR seq_plugins_azioni_comp);



CREATE SEQUENCE seq_plugins_props_comp AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE plugins_props_comp
(
	nome VARCHAR(255) NOT NULL,
	valore VARCHAR(255) NOT NULL,
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	id_plugin BIGINT NOT NULL,
	-- fk/pk keys constraints
	CONSTRAINT fk_plugins_props_comp_1 FOREIGN KEY (id_plugin) REFERENCES plugins(id) ON DELETE CASCADE,
	CONSTRAINT pk_plugins_props_comp PRIMARY KEY (id)
);

-- index
CREATE INDEX idx_plug_prop_com_1 ON plugins_props_comp (id_plugin);
CREATE TABLE plugins_props_comp_init_seq (id BIGINT);
INSERT INTO plugins_props_comp_init_seq VALUES (NEXT VALUE FOR seq_plugins_props_comp);



-- CONFIGURAZIONE DEI SERVIZI PER PERSONALIZZAZIONI

CREATE SEQUENCE seq_plugins_conf_servizi AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE plugins_conf_servizi
(
	-- Accordo di Servizio
	accordo VARCHAR(255) NOT NULL,
	tipo_soggetto_referente VARCHAR(255) NOT NULL,
	nome_soggetto_referente VARCHAR(255) NOT NULL,
	versione INT NOT NULL,
	-- Nome del port-type
	servizio VARCHAR(255),
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	-- fk/pk keys constraints
	CONSTRAINT pk_plugins_conf_servizi PRIMARY KEY (id)
);

-- index
CREATE INDEX idx_conf_servizi_1 ON plugins_conf_servizi (accordo,tipo_soggetto_referente,nome_soggetto_referente,versione,servizio);

ALTER TABLE plugins_conf_servizi ALTER COLUMN versione SET DEFAULT 1;

CREATE TABLE plugins_conf_servizi_init_seq (id BIGINT);
INSERT INTO plugins_conf_servizi_init_seq VALUES (NEXT VALUE FOR seq_plugins_conf_servizi);



CREATE SEQUENCE seq_plugins_conf_azioni AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE plugins_conf_azioni
(
	azione VARCHAR(255) NOT NULL,
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	id_config_servizio BIGINT NOT NULL,
	-- unique constraints
	CONSTRAINT unique_plugins_conf_azioni_1 UNIQUE (id_config_servizio,azione),
	-- fk/pk keys constraints
	CONSTRAINT fk_plugins_conf_azioni_1 FOREIGN KEY (id_config_servizio) REFERENCES plugins_conf_servizi(id) ON DELETE CASCADE,
	CONSTRAINT pk_plugins_conf_azioni PRIMARY KEY (id)
);

-- index
CREATE UNIQUE INDEX index_plugins_conf_azioni_1 ON plugins_conf_azioni (id_config_servizio,azione);
CREATE TABLE plugins_conf_azioni_init_seq (id BIGINT);
INSERT INTO plugins_conf_azioni_init_seq VALUES (NEXT VALUE FOR seq_plugins_conf_azioni);



-- FILTRO PER PERSONALIZZAZIONI

CREATE SEQUENCE seq_plugins_conf_filtri AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE plugins_conf_filtri
(
	-- Dati Identificativi Regola
	nome VARCHAR(255) NOT NULL,
	descrizione VARCHAR(255),
	-- Regole di Filtraggio
	tipo_mittente VARCHAR(255),
	nome_mittente VARCHAR(255),
	idporta_mittente VARCHAR(255),
	tipo_destinatario VARCHAR(255),
	nome_destinatario VARCHAR(255),
	idporta_destinatario VARCHAR(255),
	tipo_servizio VARCHAR(255),
	nome_servizio VARCHAR(255),
	versione_servizio INT,
	azione VARCHAR(255),
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	-- unique constraints
	CONSTRAINT uniq_conf_filtri_1 UNIQUE (nome),
	-- fk/pk keys constraints
	CONSTRAINT pk_plugins_conf_filtri PRIMARY KEY (id)
);

-- index
CREATE UNIQUE INDEX idx_conf_filtri_1 ON plugins_conf_filtri (nome);

ALTER TABLE plugins_conf_filtri ALTER COLUMN versione_servizio SET DEFAULT 1;

CREATE TABLE plugins_conf_filtri_init_seq (id BIGINT);
INSERT INTO plugins_conf_filtri_init_seq VALUES (NEXT VALUE FOR seq_plugins_conf_filtri);




-- govway
		  		
-- PERSONALIZZAZIONI/PLUGINS DELLE TRANSAZIONI

CREATE SEQUENCE seq_config_transazioni AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE config_transazioni
(
	enabled BOOLEAN NOT NULL,
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	id_conf_servizio_azione BIGINT NOT NULL,
	-- unique constraints
	CONSTRAINT uniq_trans_pers_1 UNIQUE (id_conf_servizio_azione),
	-- fk/pk keys constraints
	CONSTRAINT fk_config_transazioni_1 FOREIGN KEY (id_conf_servizio_azione) REFERENCES plugins_conf_azioni(id),
	CONSTRAINT pk_config_transazioni PRIMARY KEY (id)
);

-- index
CREATE UNIQUE INDEX idx_trans_pers_1 ON config_transazioni (id_conf_servizio_azione);
CREATE TABLE config_transazioni_init_seq (id BIGINT);
INSERT INTO config_transazioni_init_seq VALUES (NEXT VALUE FOR seq_config_transazioni);



CREATE SEQUENCE seq_config_tran_plugins AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE config_tran_plugins
(
	id_conf_trans_plugin VARCHAR(255) NOT NULL,
	enabled BOOLEAN NOT NULL,
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	id_configurazione_transazione BIGINT NOT NULL,
	id_plugin BIGINT NOT NULL,
	-- unique constraints
	CONSTRAINT uniq_conf_trans_plug_1 UNIQUE (id_configurazione_transazione,id_conf_trans_plugin),
	CONSTRAINT uniq_conf_trans_plug_2 UNIQUE (id_configurazione_transazione,id_plugin),
	-- fk/pk keys constraints
	CONSTRAINT fk_config_tran_plugins_1 FOREIGN KEY (id_configurazione_transazione) REFERENCES config_transazioni(id),
	CONSTRAINT fk_config_tran_plugins_2 FOREIGN KEY (id_plugin) REFERENCES plugins(id),
	CONSTRAINT pk_config_tran_plugins PRIMARY KEY (id)
);

-- index
CREATE UNIQUE INDEX idx_conf_trans_plug_1 ON config_tran_plugins (id_configurazione_transazione,id_conf_trans_plugin);
CREATE UNIQUE INDEX idx_conf_trans_plug_2 ON config_tran_plugins (id_configurazione_transazione,id_plugin);
CREATE TABLE config_tran_plugins_init_seq (id BIGINT);
INSERT INTO config_tran_plugins_init_seq VALUES (NEXT VALUE FOR seq_config_tran_plugins);



CREATE SEQUENCE seq_configurazione_stati AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE configurazione_stati
(
	enabled BOOLEAN NOT NULL,
	nome VARCHAR(255) NOT NULL,
	tipo_controllo VARCHAR(255) NOT NULL,
	tipo_messaggio VARCHAR(255) NOT NULL,
	valore VARCHAR(255),
	xpath VARCHAR(255) NOT NULL,
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	id_configurazione_transazione BIGINT NOT NULL,
	-- check constraints
	CONSTRAINT chk_configurazione_stati_1 CHECK (tipo_controllo IN ('Match','Equals','Exist')),
	CONSTRAINT chk_configurazione_stati_2 CHECK (tipo_messaggio IN ('Richiesta','Risposta')),
	-- unique constraints
	CONSTRAINT uniq_trans_pers_stati_1 UNIQUE (id_configurazione_transazione,nome),
	-- fk/pk keys constraints
	CONSTRAINT fk_configurazione_stati_1 FOREIGN KEY (id_configurazione_transazione) REFERENCES config_transazioni(id),
	CONSTRAINT pk_configurazione_stati PRIMARY KEY (id)
);

-- index
CREATE UNIQUE INDEX idx_trans_pers_stati_1 ON configurazione_stati (id_configurazione_transazione,nome);
CREATE TABLE configurazione_stati_init_seq (id BIGINT);
INSERT INTO configurazione_stati_init_seq VALUES (NEXT VALUE FOR seq_configurazione_stati);



CREATE SEQUENCE seq_conf_risorse_contenuti AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE conf_risorse_contenuti
(
	abilita_anonimizzazione INT NOT NULL,
	abilita_compressione INT NOT NULL,
	tipo_compressione VARCHAR(255),
	carattere_maschera CHAR(1),
	num_char_maschera INT,
	posizionamento_maschera VARCHAR(255),
	tipo_mascheramento VARCHAR(255),
	enabled BOOLEAN NOT NULL,
	nome VARCHAR(255) NOT NULL,
	tipo_messaggio VARCHAR(255) NOT NULL,
	xpath VARCHAR(255) NOT NULL,
	stat_enabled BOOLEAN NOT NULL,
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	id_conf_transazione BIGINT NOT NULL,
	id_configurazione_stato BIGINT,
	-- check constraints
	CONSTRAINT chk_conf_risorse_contenuti_1 CHECK (tipo_compressione IN ('Deflater','Gzip','Zip')),
	CONSTRAINT chk_conf_risorse_contenuti_2 CHECK (posizionamento_maschera IN ('Primi','Ultimi')),
	CONSTRAINT chk_conf_risorse_contenuti_3 CHECK (tipo_mascheramento IN ('Fisico','Logico')),
	CONSTRAINT chk_conf_risorse_contenuti_4 CHECK (tipo_messaggio IN ('Richiesta','Risposta')),
	-- unique constraints
	CONSTRAINT uniq_trans_pers_risorse_1 UNIQUE (id_conf_transazione,nome),
	-- fk/pk keys constraints
	CONSTRAINT fk_conf_risorse_contenuti_1 FOREIGN KEY (id_conf_transazione) REFERENCES config_transazioni(id),
	CONSTRAINT fk_conf_risorse_contenuti_2 FOREIGN KEY (id_configurazione_stato) REFERENCES configurazione_stati(id),
	CONSTRAINT pk_conf_risorse_contenuti PRIMARY KEY (id)
);

-- index
CREATE UNIQUE INDEX idx_trans_pers_risorse_1 ON conf_risorse_contenuti (id_conf_transazione,nome);

ALTER TABLE conf_risorse_contenuti ALTER COLUMN abilita_anonimizzazione SET DEFAULT 0;
ALTER TABLE conf_risorse_contenuti ALTER COLUMN abilita_compressione SET DEFAULT 0;
ALTER TABLE conf_risorse_contenuti ALTER COLUMN stat_enabled SET DEFAULT false;

CREATE TABLE conf_risorse_contenuti_init_seq (id BIGINT);
INSERT INTO conf_risorse_contenuti_init_seq VALUES (NEXT VALUE FOR seq_conf_risorse_contenuti);




-- govway
		  		
-- PLUGINS PER LE RICERCHE PERSONALIZZATE

CREATE SEQUENCE seq_ricerche_personalizzate AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE ricerche_personalizzate
(
	id VARCHAR(255) NOT NULL,
	enabled BOOLEAN NOT NULL,
	label VARCHAR(255) NOT NULL,
	-- fk/pk columns
	pid BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	id_configurazione BIGINT NOT NULL,
	id_plugin BIGINT NOT NULL,
	-- unique constraints
	CONSTRAINT uniq_ricerche_pers_1 UNIQUE (id_configurazione,id),
	CONSTRAINT uniq_ricerche_pers_2 UNIQUE (id_configurazione,id_plugin),
	-- fk/pk keys constraints
	CONSTRAINT fk_ricerche_personalizzate_1 FOREIGN KEY (id_configurazione) REFERENCES plugins_conf_azioni(id) ON DELETE CASCADE,
	CONSTRAINT fk_ricerche_personalizzate_2 FOREIGN KEY (id_plugin) REFERENCES plugins(id) ON DELETE CASCADE,
	CONSTRAINT pk_ricerche_personalizzate PRIMARY KEY (pid)
);

-- index
CREATE UNIQUE INDEX idx_ricerche_pers_1 ON ricerche_personalizzate (id_configurazione,id);
CREATE UNIQUE INDEX idx_ricerche_pers_2 ON ricerche_personalizzate (id_configurazione,id_plugin);
CREATE TABLE ricerche_personalizzate_init_seq (id BIGINT);
INSERT INTO ricerche_personalizzate_init_seq VALUES (NEXT VALUE FOR seq_ricerche_personalizzate);




-- govway
		  		
-- PLUGINS PER LE STATISTICHE PERSONALIZZATE

CREATE SEQUENCE seq_stat_personalizzate AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE stat_personalizzate
(
	id VARCHAR(255) NOT NULL,
	enabled BOOLEAN NOT NULL,
	label VARCHAR(255) NOT NULL,
	-- fk/pk columns
	pid BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	id_configurazione BIGINT NOT NULL,
	id_plugin BIGINT NOT NULL,
	-- unique constraints
	CONSTRAINT uniq_stat_pers_1 UNIQUE (id_configurazione,id),
	CONSTRAINT uniq_stat_pers_2 UNIQUE (id_configurazione,id_plugin),
	-- fk/pk keys constraints
	CONSTRAINT fk_stat_personalizzate_1 FOREIGN KEY (id_configurazione) REFERENCES plugins_conf_azioni(id) ON DELETE CASCADE,
	CONSTRAINT fk_stat_personalizzate_2 FOREIGN KEY (id_plugin) REFERENCES plugins(id) ON DELETE CASCADE,
	CONSTRAINT pk_stat_personalizzate PRIMARY KEY (pid)
);

-- index
CREATE UNIQUE INDEX idx_stat_pers_1 ON stat_personalizzate (id_configurazione,id);
CREATE UNIQUE INDEX idx_stat_pers_2 ON stat_personalizzate (id_configurazione,id_plugin);
CREATE TABLE stat_personalizzate_init_seq (id BIGINT);
INSERT INTO stat_personalizzate_init_seq VALUES (NEXT VALUE FOR seq_stat_personalizzate);




-- govway
		  		
-- **** Servizi Applicativi ****

CREATE SEQUENCE seq_servizi_applicativi AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE servizi_applicativi
(
	nome VARCHAR(2000) NOT NULL,
	tipo VARCHAR(255),
	as_client INT,
	descrizione VARCHAR(4000),
	-- * Risposta Asincrona *
	-- valori 0/1 indicano rispettivamente FALSE/TRUE
	sbustamentorisp INT,
	getmsgrisp VARCHAR(255),
	-- FOREIGN KEY (id_gestione_errore_risp) REFERENCES gestione_errore(id) Nota: indica un eventuale tipologia di gestione dell'errore per la risposta asincrona
	id_gestione_errore_risp BIGINT,
	tipoauthrisp VARCHAR(255),
	utenterisp VARCHAR(255),
	passwordrisp VARCHAR(255),
	enc_passwordrisp VARCHAR(65535),
	invio_x_rif_risp VARCHAR(255),
	risposta_x_rif_risp VARCHAR(255),
	id_connettore_risp BIGINT NOT NULL,
	sbustamento_protocol_info_risp INT,
	-- * Invocazione Servizio *
	-- valori 0/1 indicano rispettivamente FALSE/TRUE
	sbustamentoinv INT,
	getmsginv VARCHAR(255),
	-- FOREIGN KEY (id_gestione_errore_inv) REFERENCES gestione_errore(id) Nota: indica un eventuale tipologia di gestione dell'errore per l'invocazione servizio
	id_gestione_errore_inv BIGINT,
	tipoauthinv VARCHAR(255),
	utenteinv VARCHAR(255),
	passwordinv VARCHAR(255),
	enc_passwordinv VARCHAR(65535),
	invio_x_rif_inv VARCHAR(255),
	risposta_x_rif_inv VARCHAR(255),
	id_connettore_inv BIGINT NOT NULL,
	sbustamento_protocol_info_inv INT,
	-- * SoggettoErogatore *
	id_soggetto BIGINT NOT NULL,
	-- * Invocazione Porta *
	fault VARCHAR(255),
	fault_actor VARCHAR(255),
	generic_fault_code VARCHAR(255),
	prefix_fault_code VARCHAR(255),
	tipoauth VARCHAR(255),
	utente VARCHAR(2800),
	password VARCHAR(255),
	subject VARCHAR(2800),
	cn_subject VARCHAR(255),
	issuer VARCHAR(2800),
	cn_issuer VARCHAR(255),
	certificate VARBINARY(16777215),
	cert_strict_verification INT,
	token_policy VARCHAR(255),
	invio_x_rif VARCHAR(255),
	sbustamento_protocol_info INT,
	tipologia_fruizione VARCHAR(255),
	tipologia_erogazione VARCHAR(255),
	ora_registrazione TIMESTAMP,
	utente_richiedente VARCHAR(255),
	data_creazione TIMESTAMP,
	utente_ultima_modifica VARCHAR(255),
	data_ultima_modifica TIMESTAMP,
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	-- unique constraints
	CONSTRAINT unique_servizi_applicativi_1 UNIQUE (nome,id_soggetto),
	-- fk/pk keys constraints
	CONSTRAINT fk_servizi_applicativi_1 FOREIGN KEY (id_connettore_inv) REFERENCES connettori(id),
	CONSTRAINT fk_servizi_applicativi_2 FOREIGN KEY (id_soggetto) REFERENCES soggetti(id),
	CONSTRAINT pk_servizi_applicativi PRIMARY KEY (id)
);

-- index
CREATE UNIQUE INDEX index_servizi_applicativi_1 ON servizi_applicativi (nome,id_soggetto);
CREATE INDEX index_servizi_applicativi_2 ON servizi_applicativi (id_soggetto);

ALTER TABLE servizi_applicativi ALTER COLUMN sbustamentorisp SET DEFAULT 0;
ALTER TABLE servizi_applicativi ALTER COLUMN sbustamento_protocol_info_risp SET DEFAULT 1;
ALTER TABLE servizi_applicativi ALTER COLUMN sbustamentoinv SET DEFAULT 0;
ALTER TABLE servizi_applicativi ALTER COLUMN sbustamento_protocol_info_inv SET DEFAULT 1;
ALTER TABLE servizi_applicativi ALTER COLUMN sbustamento_protocol_info SET DEFAULT 1;
ALTER TABLE servizi_applicativi ALTER COLUMN ora_registrazione SET DEFAULT CURRENT_TIMESTAMP;

CREATE TABLE servizi_applicativi_init_seq (id BIGINT);
INSERT INTO servizi_applicativi_init_seq VALUES (NEXT VALUE FOR seq_servizi_applicativi);



CREATE SEQUENCE seq_sa_ruoli AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE sa_ruoli
(
	id_servizio_applicativo BIGINT NOT NULL,
	ruolo VARCHAR(255) NOT NULL,
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	-- unique constraints
	CONSTRAINT unique_sa_ruoli_1 UNIQUE (id_servizio_applicativo,ruolo),
	-- fk/pk keys constraints
	CONSTRAINT fk_sa_ruoli_1 FOREIGN KEY (id_servizio_applicativo) REFERENCES servizi_applicativi(id),
	CONSTRAINT pk_sa_ruoli PRIMARY KEY (id)
);

-- index
CREATE UNIQUE INDEX index_sa_ruoli_1 ON sa_ruoli (id_servizio_applicativo,ruolo);
CREATE INDEX INDEX_SA_RUOLI ON sa_ruoli (id_servizio_applicativo);
CREATE TABLE sa_ruoli_init_seq (id BIGINT);
INSERT INTO sa_ruoli_init_seq VALUES (NEXT VALUE FOR seq_sa_ruoli);



CREATE SEQUENCE seq_sa_credenziali AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE sa_credenziali
(
	id_servizio_applicativo BIGINT NOT NULL,
	subject VARCHAR(2800),
	cn_subject VARCHAR(255),
	issuer VARCHAR(2800),
	cn_issuer VARCHAR(255),
	certificate VARBINARY(16777215),
	cert_strict_verification INT,
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	-- fk/pk keys constraints
	CONSTRAINT fk_sa_credenziali_1 FOREIGN KEY (id_servizio_applicativo) REFERENCES servizi_applicativi(id),
	CONSTRAINT pk_sa_credenziali PRIMARY KEY (id)
);

-- index
CREATE INDEX INDEX_SA_CRED ON sa_credenziali (id_servizio_applicativo);
CREATE TABLE sa_credenziali_init_seq (id BIGINT);
INSERT INTO sa_credenziali_init_seq VALUES (NEXT VALUE FOR seq_sa_credenziali);



CREATE SEQUENCE seq_sa_properties AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE sa_properties
(
	id_servizio_applicativo BIGINT NOT NULL,
	nome VARCHAR(255) NOT NULL,
	valore VARCHAR(4000) NOT NULL,
	enc_value VARCHAR(65535),
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	-- unique constraints
	CONSTRAINT uniq_sa_properties_1 UNIQUE (id_servizio_applicativo,nome),
	-- fk/pk keys constraints
	CONSTRAINT fk_sa_properties_1 FOREIGN KEY (id_servizio_applicativo) REFERENCES servizi_applicativi(id),
	CONSTRAINT pk_sa_properties PRIMARY KEY (id)
);

-- index
CREATE INDEX INDEX_SA_PROP ON sa_properties (id_servizio_applicativo);
CREATE TABLE sa_properties_init_seq (id BIGINT);
INSERT INTO sa_properties_init_seq VALUES (NEXT VALUE FOR seq_sa_properties);




-- govway
		  		
-- **** Porte Delegate ****

CREATE SEQUENCE seq_porte_delegate AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE porte_delegate
(
	nome_porta VARCHAR(2000) NOT NULL,
	descrizione VARCHAR(4000),
	-- * Soggetto Erogatore *
	-- tipo/nome per le modalita static
	-- id utilizzato in caso di registryInput
	id_soggetto_erogatore BIGINT,
	tipo_soggetto_erogatore VARCHAR(255) NOT NULL,
	nome_soggetto_erogatore VARCHAR(255) NOT NULL,
	-- * Servizio *
	-- tipo/nome/versione per le modalita static
	-- id utilizzato in caso di registryInput
	id_servizio BIGINT,
	tipo_servizio VARCHAR(255) NOT NULL,
	nome_servizio VARCHAR(255) NOT NULL,
	versione_servizio INT NOT NULL,
	id_accordo BIGINT,
	id_port_type BIGINT,
	-- * Azione *
	-- tipo/nome per le modalita static
	-- tipo/pattern per la modalita contentBased/urlBased
	-- id utilizzato in caso di registryInput
	id_azione BIGINT,
	nome_azione VARCHAR(255),
	mode_azione VARCHAR(255),
	pattern_azione VARCHAR(255),
	nome_porta_delegante_azione VARCHAR(255),
	-- abilitato/disabilitato
	force_interface_based_azione VARCHAR(255),
	-- Controllo Accessi
	autenticazione VARCHAR(255),
	-- abilitato/disabilitato
	autenticazione_opzionale VARCHAR(255),
	-- Gestione Token
	token_policy VARCHAR(255),
	token_opzionale VARCHAR(255),
	token_validazione VARCHAR(255),
	token_introspection VARCHAR(255),
	token_user_info VARCHAR(255),
	token_forward VARCHAR(255),
	token_options VARCHAR(4000),
	token_authn_issuer VARCHAR(255),
	token_authn_client_id VARCHAR(255),
	token_authn_subject VARCHAR(255),
	token_authn_username VARCHAR(255),
	token_authn_email VARCHAR(255),
	-- Autorizzazione
	autorizzazione VARCHAR(255),
	autorizzazione_xacml LONGVARCHAR,
	autorizzazione_contenuto VARCHAR(255),
	-- disable/packaging/unpackaging/verify
	mtom_request_mode VARCHAR(255),
	-- disable/packaging/unpackaging/verify
	mtom_response_mode VARCHAR(255),
	-- abilitato/disabilitato (se abilitato le WSSproperties sono presenti nelle tabelle ...._security_request/response)
	security VARCHAR(255),
	-- abilitato/disabilitato
	security_mtom_req VARCHAR(255),
	-- abilitato/disabilitato
	security_mtom_res VARCHAR(255),
	security_request_mode VARCHAR(255),
	security_response_mode VARCHAR(255),
	-- abilitato/disabilitato
	ricevuta_asincrona_sim VARCHAR(255),
	-- abilitato/disabilitato
	ricevuta_asincrona_asim VARCHAR(255),
	-- abilitato/disabilitato/warningOnly
	validazione_contenuti_stato VARCHAR(255),
	-- wsdl/interface/xsd
	validazione_contenuti_tipo VARCHAR(255),
	-- abilitato/disabilitato
	validazione_contenuti_mtom VARCHAR(255),
	-- lista di tipi separati dalla virgola
	integrazione VARCHAR(4000),
	-- scadenza correlazione applicativa
	scadenza_correlazione_appl VARCHAR(255),
	-- abilitato/disabilitato
	allega_body VARCHAR(255),
	-- abilitato/disabilitato
	scarta_body VARCHAR(255),
	-- abilitato/disabilitato
	gestione_manifest VARCHAR(255),
	-- abilitato/disabilitato
	stateless VARCHAR(255),
	-- abilitato/disabilitato
	local_forward VARCHAR(255),
	-- Nome della PortaApplicativa
	local_forward_pa VARCHAR(255),
	-- all/any
	ruoli_match VARCHAR(255),
	token_sa_stato VARCHAR(255),
	token_ruoli_stato VARCHAR(255),
	-- all/any
	token_ruoli_match VARCHAR(255),
	token_ruoli_tipologia VARCHAR(255),
	scope_stato VARCHAR(255),
	-- all/any
	scope_match VARCHAR(255),
	-- abilitato/disabilitato
	ricerca_porta_azione_delegata VARCHAR(255),
	-- Livello Log Messaggi Diagnostici
	msg_diag_severita VARCHAR(255),
	-- Tracciamento delle transazioni
	tracciamento_stato VARCHAR(255),
	tracciamento_esiti VARCHAR(255),
	transazioni_tempi VARCHAR(255),
	transazioni_token VARCHAR(255),
	-- Gestione CORS
	cors_stato VARCHAR(255),
	cors_tipo VARCHAR(255),
	cors_all_allow_origins VARCHAR(255),
	cors_all_allow_methods VARCHAR(255),
	cors_all_allow_headers VARCHAR(255),
	cors_allow_credentials VARCHAR(255),
	cors_allow_max_age INT,
	cors_allow_max_age_seconds INT,
	cors_allow_origins VARCHAR(65535),
	cors_allow_headers VARCHAR(65535),
	cors_allow_methods VARCHAR(65535),
	cors_allow_expose_headers VARCHAR(65535),
	-- Response caching
	response_cache_stato VARCHAR(255),
	response_cache_seconds INT,
	response_cache_max_msg_size BIGINT,
	response_cache_hash_url VARCHAR(255),
	response_cache_hash_query VARCHAR(255),
	response_cache_hash_query_list VARCHAR(65535),
	response_cache_hash_headers VARCHAR(255),
	response_cache_hash_hdr_list VARCHAR(65535),
	response_cache_hash_payload VARCHAR(255),
	response_cache_control_nocache INT,
	response_cache_control_maxage INT,
	response_cache_control_nostore INT,
	-- Stato della porta: abilitato/disabilitato
	stato VARCHAR(255),
	-- proprietario porta delegata (Soggetto fruitore)
	id_soggetto BIGINT NOT NULL,
	ora_registrazione TIMESTAMP,
	options VARCHAR(4000),
	canale VARCHAR(255),
	utente_richiedente VARCHAR(255),
	data_creazione TIMESTAMP,
	utente_ultima_modifica VARCHAR(255),
	data_ultima_modifica TIMESTAMP,
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	-- unique constraints
	CONSTRAINT unique_porte_delegate_1 UNIQUE (nome_porta),
	-- fk/pk keys constraints
	CONSTRAINT fk_porte_delegate_1 FOREIGN KEY (id_soggetto) REFERENCES soggetti(id),
	CONSTRAINT pk_porte_delegate PRIMARY KEY (id)
);

-- index
CREATE UNIQUE INDEX index_porte_delegate_1 ON porte_delegate (nome_porta);
CREATE INDEX index_porte_delegate_2 ON porte_delegate (id_soggetto);
CREATE INDEX index_porte_delegate_3 ON porte_delegate (canale);

ALTER TABLE porte_delegate ALTER COLUMN versione_servizio SET DEFAULT 1;
ALTER TABLE porte_delegate ALTER COLUMN ora_registrazione SET DEFAULT CURRENT_TIMESTAMP;

CREATE TABLE porte_delegate_init_seq (id BIGINT);
INSERT INTO porte_delegate_init_seq VALUES (NEXT VALUE FOR seq_porte_delegate);



CREATE SEQUENCE seq_porte_delegate_sa AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE porte_delegate_sa
(
	id_porta BIGINT NOT NULL,
	id_servizio_applicativo BIGINT NOT NULL,
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	-- unique constraints
	CONSTRAINT unique_porte_delegate_sa_1 UNIQUE (id_porta,id_servizio_applicativo),
	-- fk/pk keys constraints
	CONSTRAINT fk_porte_delegate_sa_1 FOREIGN KEY (id_servizio_applicativo) REFERENCES servizi_applicativi(id),
	CONSTRAINT fk_porte_delegate_sa_2 FOREIGN KEY (id_porta) REFERENCES porte_delegate(id),
	CONSTRAINT pk_porte_delegate_sa PRIMARY KEY (id)
);

-- index
CREATE INDEX INDEX_PD_SA ON porte_delegate_sa (id_porta);
CREATE TABLE porte_delegate_sa_init_seq (id BIGINT);
INSERT INTO porte_delegate_sa_init_seq VALUES (NEXT VALUE FOR seq_porte_delegate_sa);



CREATE SEQUENCE seq_pd_auth_properties AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE pd_auth_properties
(
	id_porta BIGINT NOT NULL,
	nome VARCHAR(255) NOT NULL,
	valore VARCHAR(255) NOT NULL,
	enc_value VARCHAR(65535),
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	-- unique constraints
	CONSTRAINT uniq_pd_auth_props_1 UNIQUE (id_porta,nome),
	-- fk/pk keys constraints
	CONSTRAINT fk_pd_auth_properties_1 FOREIGN KEY (id_porta) REFERENCES porte_delegate(id),
	CONSTRAINT pk_pd_auth_properties PRIMARY KEY (id)
);

-- index
CREATE INDEX INDEX_PD_AUTH_PROP ON pd_auth_properties (id_porta);
CREATE TABLE pd_auth_properties_init_seq (id BIGINT);
INSERT INTO pd_auth_properties_init_seq VALUES (NEXT VALUE FOR seq_pd_auth_properties);



CREATE SEQUENCE seq_pd_authz_properties AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE pd_authz_properties
(
	id_porta BIGINT NOT NULL,
	nome VARCHAR(255) NOT NULL,
	valore VARCHAR(255) NOT NULL,
	enc_value VARCHAR(65535),
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	-- unique constraints
	CONSTRAINT uniq_pd_authz_props_1 UNIQUE (id_porta,nome),
	-- fk/pk keys constraints
	CONSTRAINT fk_pd_authz_properties_1 FOREIGN KEY (id_porta) REFERENCES porte_delegate(id),
	CONSTRAINT pk_pd_authz_properties PRIMARY KEY (id)
);

-- index
CREATE INDEX INDEX_PD_AUTHZ_PROP ON pd_authz_properties (id_porta);
CREATE TABLE pd_authz_properties_init_seq (id BIGINT);
INSERT INTO pd_authz_properties_init_seq VALUES (NEXT VALUE FOR seq_pd_authz_properties);



CREATE SEQUENCE seq_pd_authzc_properties AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE pd_authzc_properties
(
	id_porta BIGINT NOT NULL,
	nome VARCHAR(255) NOT NULL,
	valore VARCHAR(255) NOT NULL,
	enc_value VARCHAR(65535),
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	-- unique constraints
	CONSTRAINT uniq_pd_authzc_props_1 UNIQUE (id_porta,nome),
	-- fk/pk keys constraints
	CONSTRAINT fk_pd_authzc_properties_1 FOREIGN KEY (id_porta) REFERENCES porte_delegate(id),
	CONSTRAINT pk_pd_authzc_properties PRIMARY KEY (id)
);

-- index
CREATE INDEX INDEX_PD_AUTHZC_PROP ON pd_authzc_properties (id_porta);
CREATE TABLE pd_authzc_properties_init_seq (id BIGINT);
INSERT INTO pd_authzc_properties_init_seq VALUES (NEXT VALUE FOR seq_pd_authzc_properties);



CREATE SEQUENCE seq_pd_ct_properties AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE pd_ct_properties
(
	id_porta BIGINT NOT NULL,
	nome VARCHAR(255) NOT NULL,
	valore VARCHAR(255) NOT NULL,
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	-- unique constraints
	CONSTRAINT uniq_pd_ct_props_1 UNIQUE (id_porta,nome),
	-- fk/pk keys constraints
	CONSTRAINT fk_pd_ct_properties_1 FOREIGN KEY (id_porta) REFERENCES porte_delegate(id),
	CONSTRAINT pk_pd_ct_properties PRIMARY KEY (id)
);

-- index
CREATE INDEX idx_pd_ct_props_1 ON pd_ct_properties (valore);
CREATE INDEX INDEX_PD_CT_PROP ON pd_ct_properties (id_porta);
CREATE TABLE pd_ct_properties_init_seq (id BIGINT);
INSERT INTO pd_ct_properties_init_seq VALUES (NEXT VALUE FOR seq_pd_ct_properties);



CREATE SEQUENCE seq_pd_properties AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE pd_properties
(
	id_porta BIGINT NOT NULL,
	nome VARCHAR(255) NOT NULL,
	valore VARCHAR(255) NOT NULL,
	enc_value VARCHAR(65535),
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	-- unique constraints
	CONSTRAINT uniq_pd_properties_1 UNIQUE (id_porta,nome),
	-- fk/pk keys constraints
	CONSTRAINT fk_pd_properties_1 FOREIGN KEY (id_porta) REFERENCES porte_delegate(id),
	CONSTRAINT pk_pd_properties PRIMARY KEY (id)
);

-- index
CREATE INDEX INDEX_PD_PROP ON pd_properties (id_porta);
CREATE TABLE pd_properties_init_seq (id BIGINT);
INSERT INTO pd_properties_init_seq VALUES (NEXT VALUE FOR seq_pd_properties);



CREATE SEQUENCE seq_pd_mtom_request AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE pd_mtom_request
(
	id_porta BIGINT NOT NULL,
	nome VARCHAR(255) NOT NULL,
	pattern VARCHAR(65535) NOT NULL,
	content_type VARCHAR(255),
	required INT NOT NULL,
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	-- fk/pk keys constraints
	CONSTRAINT fk_pd_mtom_request_1 FOREIGN KEY (id_porta) REFERENCES porte_delegate(id),
	CONSTRAINT pk_pd_mtom_request PRIMARY KEY (id)
);

-- index
CREATE INDEX INDEX_PD_MTOMTREQ ON pd_mtom_request (id_porta);
CREATE TABLE pd_mtom_request_init_seq (id BIGINT);
INSERT INTO pd_mtom_request_init_seq VALUES (NEXT VALUE FOR seq_pd_mtom_request);



CREATE SEQUENCE seq_pd_mtom_response AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE pd_mtom_response
(
	id_porta BIGINT NOT NULL,
	nome VARCHAR(255) NOT NULL,
	pattern VARCHAR(65535) NOT NULL,
	content_type VARCHAR(255),
	required INT NOT NULL,
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	-- fk/pk keys constraints
	CONSTRAINT fk_pd_mtom_response_1 FOREIGN KEY (id_porta) REFERENCES porte_delegate(id),
	CONSTRAINT pk_pd_mtom_response PRIMARY KEY (id)
);

-- index
CREATE INDEX INDEX_PD_MTOMTRES ON pd_mtom_response (id_porta);
CREATE TABLE pd_mtom_response_init_seq (id BIGINT);
INSERT INTO pd_mtom_response_init_seq VALUES (NEXT VALUE FOR seq_pd_mtom_response);



CREATE SEQUENCE seq_pd_security_request AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE pd_security_request
(
	id_porta BIGINT NOT NULL,
	nome VARCHAR(255) NOT NULL,
	valore VARCHAR(65535) NOT NULL,
	enc_value VARCHAR(65535),
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	-- fk/pk keys constraints
	CONSTRAINT fk_pd_security_request_1 FOREIGN KEY (id_porta) REFERENCES porte_delegate(id),
	CONSTRAINT pk_pd_security_request PRIMARY KEY (id)
);

-- index
CREATE INDEX INDEX_PD_WSSREQ ON pd_security_request (id_porta);
CREATE TABLE pd_security_request_init_seq (id BIGINT);
INSERT INTO pd_security_request_init_seq VALUES (NEXT VALUE FOR seq_pd_security_request);



CREATE SEQUENCE seq_pd_security_response AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE pd_security_response
(
	id_porta BIGINT NOT NULL,
	nome VARCHAR(255) NOT NULL,
	valore VARCHAR(65535) NOT NULL,
	enc_value VARCHAR(65535),
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	-- fk/pk keys constraints
	CONSTRAINT fk_pd_security_response_1 FOREIGN KEY (id_porta) REFERENCES porte_delegate(id),
	CONSTRAINT pk_pd_security_response PRIMARY KEY (id)
);

-- index
CREATE INDEX INDEX_PD_WSSRES ON pd_security_response (id_porta);
CREATE TABLE pd_security_response_init_seq (id BIGINT);
INSERT INTO pd_security_response_init_seq VALUES (NEXT VALUE FOR seq_pd_security_response);



CREATE SEQUENCE seq_pd_correlazione AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE pd_correlazione
(
	id_porta BIGINT NOT NULL,
	nome_elemento VARCHAR(255),
	-- modalita di scelta user input, content-based, url-based, disabilitato
	mode_correlazione VARCHAR(255),
	-- pattern utilizzato solo per content/url based
	pattern VARCHAR(65535),
	-- blocca/accetta
	identificazione_fallita VARCHAR(255),
	-- abilitato/disabilitato
	riuso_id VARCHAR(255),
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	-- fk/pk keys constraints
	CONSTRAINT fk_pd_correlazione_1 FOREIGN KEY (id_porta) REFERENCES porte_delegate(id),
	CONSTRAINT pk_pd_correlazione PRIMARY KEY (id)
);

-- index
CREATE INDEX INDEX_PD_CORR_REQ ON pd_correlazione (id_porta);
CREATE TABLE pd_correlazione_init_seq (id BIGINT);
INSERT INTO pd_correlazione_init_seq VALUES (NEXT VALUE FOR seq_pd_correlazione);



CREATE SEQUENCE seq_pd_correlazione_risposta AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE pd_correlazione_risposta
(
	id_porta BIGINT NOT NULL,
	nome_elemento VARCHAR(255),
	-- modalita di scelta user input, content-based, url-based, disabilitato
	mode_correlazione VARCHAR(255),
	-- pattern utilizzato solo per content/url based
	pattern VARCHAR(65535),
	-- blocca/accetta
	identificazione_fallita VARCHAR(255),
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	-- fk/pk keys constraints
	CONSTRAINT fk_pd_correlazione_risposta_1 FOREIGN KEY (id_porta) REFERENCES porte_delegate(id),
	CONSTRAINT pk_pd_correlazione_risposta PRIMARY KEY (id)
);

-- index
CREATE INDEX INDEX_PD_CORR_RES ON pd_correlazione_risposta (id_porta);
CREATE TABLE pd_correlazione_risposta_init_seq (id BIGINT);
INSERT INTO pd_correlazione_risposta_init_seq VALUES (NEXT VALUE FOR seq_pd_correlazione_risposta);



CREATE SEQUENCE seq_pd_ruoli AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE pd_ruoli
(
	id_porta BIGINT NOT NULL,
	ruolo VARCHAR(255) NOT NULL,
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	-- unique constraints
	CONSTRAINT unique_pd_ruoli_1 UNIQUE (id_porta,ruolo),
	-- fk/pk keys constraints
	CONSTRAINT fk_pd_ruoli_1 FOREIGN KEY (id_porta) REFERENCES porte_delegate(id),
	CONSTRAINT pk_pd_ruoli PRIMARY KEY (id)
);

-- index
CREATE UNIQUE INDEX index_pd_ruoli_1 ON pd_ruoli (id_porta,ruolo);
CREATE TABLE pd_ruoli_init_seq (id BIGINT);
INSERT INTO pd_ruoli_init_seq VALUES (NEXT VALUE FOR seq_pd_ruoli);



CREATE SEQUENCE seq_pd_scope AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE pd_scope
(
	id_porta BIGINT NOT NULL,
	scope VARCHAR(255) NOT NULL,
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	-- unique constraints
	CONSTRAINT unique_pd_scope_1 UNIQUE (id_porta,scope),
	-- fk/pk keys constraints
	CONSTRAINT fk_pd_scope_1 FOREIGN KEY (id_porta) REFERENCES porte_delegate(id),
	CONSTRAINT pk_pd_scope PRIMARY KEY (id)
);

-- index
CREATE UNIQUE INDEX index_pd_scope_1 ON pd_scope (id_porta,scope);
CREATE TABLE pd_scope_init_seq (id BIGINT);
INSERT INTO pd_scope_init_seq VALUES (NEXT VALUE FOR seq_pd_scope);



CREATE SEQUENCE seq_pd_azioni AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE pd_azioni
(
	id_porta BIGINT NOT NULL,
	azione VARCHAR(255) NOT NULL,
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	-- unique constraints
	CONSTRAINT unique_pd_azioni_1 UNIQUE (id_porta,azione),
	-- fk/pk keys constraints
	CONSTRAINT fk_pd_azioni_1 FOREIGN KEY (id_porta) REFERENCES porte_delegate(id),
	CONSTRAINT pk_pd_azioni PRIMARY KEY (id)
);

-- index
CREATE UNIQUE INDEX index_pd_azioni_1 ON pd_azioni (id_porta,azione);
CREATE TABLE pd_azioni_init_seq (id BIGINT);
INSERT INTO pd_azioni_init_seq VALUES (NEXT VALUE FOR seq_pd_azioni);



CREATE SEQUENCE seq_pd_cache_regole AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE pd_cache_regole
(
	id_porta BIGINT NOT NULL,
	status_min INT,
	status_max INT,
	fault INT,
	cache_seconds INT,
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	-- fk/pk keys constraints
	CONSTRAINT fk_pd_cache_regole_1 FOREIGN KEY (id_porta) REFERENCES porte_delegate(id),
	CONSTRAINT pk_pd_cache_regole PRIMARY KEY (id)
);

-- index
CREATE INDEX INDEX_PD_CACHE ON pd_cache_regole (id_porta);

ALTER TABLE pd_cache_regole ALTER COLUMN fault SET DEFAULT 0;

CREATE TABLE pd_cache_regole_init_seq (id BIGINT);
INSERT INTO pd_cache_regole_init_seq VALUES (NEXT VALUE FOR seq_pd_cache_regole);



CREATE SEQUENCE seq_pd_transform AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE pd_transform
(
	id_porta BIGINT NOT NULL,
	nome VARCHAR(255) NOT NULL,
	posizione INT NOT NULL,
	stato VARCHAR(255),
	applicabilita_azioni VARCHAR(65535),
	applicabilita_ct VARCHAR(65535),
	applicabilita_pattern VARCHAR(65535),
	applicabilita_connettori VARCHAR(65535),
	req_conversione_enabled INT NOT NULL,
	req_conversione_tipo VARCHAR(255),
	-- In hsql 2.x usare il tipo BLOB al posto di VARBINARY
	req_conversione_template VARBINARY(1073741823),
	req_content_type VARCHAR(255),
	rest_transformation INT NOT NULL,
	rest_method VARCHAR(255),
	rest_path VARCHAR(4000),
	soap_transformation INT NOT NULL,
	soap_version VARCHAR(255),
	soap_action VARCHAR(255),
	soap_envelope INT NOT NULL,
	soap_envelope_as_attach INT NOT NULL,
	soap_envelope_tipo VARCHAR(255),
	-- In hsql 2.x usare il tipo BLOB al posto di VARBINARY
	soap_envelope_template VARBINARY(1073741823),
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	-- unique constraints
	CONSTRAINT unique_pd_transform_1 UNIQUE (id_porta,nome),
	CONSTRAINT unique_pd_transform_2 UNIQUE (id_porta,posizione),
	-- fk/pk keys constraints
	CONSTRAINT fk_pd_transform_1 FOREIGN KEY (id_porta) REFERENCES porte_delegate(id),
	CONSTRAINT pk_pd_transform PRIMARY KEY (id)
);

-- index
CREATE UNIQUE INDEX index_pd_transform_1 ON pd_transform (id_porta,nome);
CREATE UNIQUE INDEX index_pd_transform_2 ON pd_transform (id_porta,posizione);

ALTER TABLE pd_transform ALTER COLUMN req_conversione_enabled SET DEFAULT 0;
ALTER TABLE pd_transform ALTER COLUMN rest_transformation SET DEFAULT 0;
ALTER TABLE pd_transform ALTER COLUMN soap_transformation SET DEFAULT 0;
ALTER TABLE pd_transform ALTER COLUMN soap_envelope SET DEFAULT 0;
ALTER TABLE pd_transform ALTER COLUMN soap_envelope_as_attach SET DEFAULT 0;

CREATE TABLE pd_transform_init_seq (id BIGINT);
INSERT INTO pd_transform_init_seq VALUES (NEXT VALUE FOR seq_pd_transform);



CREATE SEQUENCE seq_pd_transform_sa AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE pd_transform_sa
(
	id_trasformazione BIGINT NOT NULL,
	id_servizio_applicativo BIGINT NOT NULL,
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	-- unique constraints
	CONSTRAINT unique_pd_transform_sa_1 UNIQUE (id_trasformazione,id_servizio_applicativo),
	-- fk/pk keys constraints
	CONSTRAINT fk_pd_transform_sa_1 FOREIGN KEY (id_servizio_applicativo) REFERENCES servizi_applicativi(id),
	CONSTRAINT fk_pd_transform_sa_2 FOREIGN KEY (id_trasformazione) REFERENCES pd_transform(id),
	CONSTRAINT pk_pd_transform_sa PRIMARY KEY (id)
);

-- index
CREATE UNIQUE INDEX index_pd_transform_sa_1 ON pd_transform_sa (id_trasformazione,id_servizio_applicativo);
CREATE TABLE pd_transform_sa_init_seq (id BIGINT);
INSERT INTO pd_transform_sa_init_seq VALUES (NEXT VALUE FOR seq_pd_transform_sa);



CREATE SEQUENCE seq_pd_transform_hdr AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE pd_transform_hdr
(
	id_trasformazione BIGINT NOT NULL,
	tipo VARCHAR(255) NOT NULL,
	nome VARCHAR(255) NOT NULL,
	valore VARCHAR(65535),
	identificazione_fallita VARCHAR(255),
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	-- fk/pk keys constraints
	CONSTRAINT fk_pd_transform_hdr_1 FOREIGN KEY (id_trasformazione) REFERENCES pd_transform(id),
	CONSTRAINT pk_pd_transform_hdr PRIMARY KEY (id)
);

-- index
CREATE INDEX idx_pd_trasf_hdr_1 ON pd_transform_hdr (id_trasformazione);
CREATE TABLE pd_transform_hdr_init_seq (id BIGINT);
INSERT INTO pd_transform_hdr_init_seq VALUES (NEXT VALUE FOR seq_pd_transform_hdr);



CREATE SEQUENCE seq_pd_transform_url AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE pd_transform_url
(
	id_trasformazione BIGINT NOT NULL,
	tipo VARCHAR(255) NOT NULL,
	nome VARCHAR(255) NOT NULL,
	valore VARCHAR(65535),
	identificazione_fallita VARCHAR(255),
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	-- fk/pk keys constraints
	CONSTRAINT fk_pd_transform_url_1 FOREIGN KEY (id_trasformazione) REFERENCES pd_transform(id),
	CONSTRAINT pk_pd_transform_url PRIMARY KEY (id)
);

-- index
CREATE INDEX idx_pd_trasf_url_1 ON pd_transform_url (id_trasformazione);
CREATE TABLE pd_transform_url_init_seq (id BIGINT);
INSERT INTO pd_transform_url_init_seq VALUES (NEXT VALUE FOR seq_pd_transform_url);



CREATE SEQUENCE seq_pd_transform_risp AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE pd_transform_risp
(
	id_trasformazione BIGINT NOT NULL,
	nome VARCHAR(255) NOT NULL,
	posizione INT NOT NULL,
	applicabilita_status_min INT,
	applicabilita_status_max INT,
	applicabilita_ct VARCHAR(65535),
	applicabilita_pattern VARCHAR(65535),
	conversione_enabled INT NOT NULL,
	conversione_tipo VARCHAR(255),
	-- In hsql 2.x usare il tipo BLOB al posto di VARBINARY
	conversione_template VARBINARY(1073741823),
	content_type VARCHAR(255),
	return_code VARCHAR(255),
	soap_envelope INT NOT NULL,
	soap_envelope_as_attach INT NOT NULL,
	soap_envelope_tipo VARCHAR(255),
	-- In hsql 2.x usare il tipo BLOB al posto di VARBINARY
	soap_envelope_template VARBINARY(1073741823),
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	-- unique constraints
	CONSTRAINT uniq_pd_trasf_resp_1 UNIQUE (id_trasformazione,nome),
	CONSTRAINT uniq_pd_trasf_resp_2 UNIQUE (id_trasformazione,posizione),
	-- fk/pk keys constraints
	CONSTRAINT fk_pd_transform_risp_1 FOREIGN KEY (id_trasformazione) REFERENCES pd_transform(id),
	CONSTRAINT pk_pd_transform_risp PRIMARY KEY (id)
);

-- index
CREATE UNIQUE INDEX idx_pd_trasf_resp_1 ON pd_transform_risp (id_trasformazione,nome);
CREATE UNIQUE INDEX idx_pd_trasf_resp_2 ON pd_transform_risp (id_trasformazione,posizione);

ALTER TABLE pd_transform_risp ALTER COLUMN conversione_enabled SET DEFAULT 0;
ALTER TABLE pd_transform_risp ALTER COLUMN soap_envelope SET DEFAULT 0;
ALTER TABLE pd_transform_risp ALTER COLUMN soap_envelope_as_attach SET DEFAULT 0;

CREATE TABLE pd_transform_risp_init_seq (id BIGINT);
INSERT INTO pd_transform_risp_init_seq VALUES (NEXT VALUE FOR seq_pd_transform_risp);



CREATE SEQUENCE seq_pd_transform_risp_hdr AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE pd_transform_risp_hdr
(
	id_transform_risp BIGINT NOT NULL,
	tipo VARCHAR(255) NOT NULL,
	nome VARCHAR(255) NOT NULL,
	valore VARCHAR(65535),
	identificazione_fallita VARCHAR(255),
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	-- fk/pk keys constraints
	CONSTRAINT fk_pd_transform_risp_hdr_1 FOREIGN KEY (id_transform_risp) REFERENCES pd_transform_risp(id),
	CONSTRAINT pk_pd_transform_risp_hdr PRIMARY KEY (id)
);

-- index
CREATE INDEX idx_pd_trasf_hdr_resp_1 ON pd_transform_risp_hdr (id_transform_risp);
CREATE TABLE pd_transform_risp_hdr_init_seq (id BIGINT);
INSERT INTO pd_transform_risp_hdr_init_seq VALUES (NEXT VALUE FOR seq_pd_transform_risp_hdr);



CREATE SEQUENCE seq_pd_handlers AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE pd_handlers
(
	id_porta BIGINT NOT NULL,
	tipologia VARCHAR(255) NOT NULL,
	tipo VARCHAR(255) NOT NULL,
	posizione INT NOT NULL,
	stato VARCHAR(255),
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	-- unique constraints
	CONSTRAINT unique_pd_handlers_1 UNIQUE (id_porta,tipologia,tipo),
	-- fk/pk keys constraints
	CONSTRAINT fk_pd_handlers_1 FOREIGN KEY (id_porta) REFERENCES porte_delegate(id),
	CONSTRAINT pk_pd_handlers PRIMARY KEY (id)
);

-- index
CREATE UNIQUE INDEX index_pd_handlers_1 ON pd_handlers (id_porta,tipologia,tipo);
CREATE TABLE pd_handlers_init_seq (id BIGINT);
INSERT INTO pd_handlers_init_seq VALUES (NEXT VALUE FOR seq_pd_handlers);



CREATE SEQUENCE seq_pd_aa AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE pd_aa
(
	id_porta BIGINT NOT NULL,
	nome VARCHAR(255) NOT NULL,
	attributi VARCHAR(65535),
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	-- unique constraints
	CONSTRAINT unique_pd_aa_1 UNIQUE (id_porta,nome),
	-- fk/pk keys constraints
	CONSTRAINT fk_pd_aa_1 FOREIGN KEY (id_porta) REFERENCES porte_delegate(id),
	CONSTRAINT pk_pd_aa PRIMARY KEY (id)
);

-- index
CREATE UNIQUE INDEX index_pd_aa_1 ON pd_aa (id_porta,nome);
CREATE TABLE pd_aa_init_seq (id BIGINT);
INSERT INTO pd_aa_init_seq VALUES (NEXT VALUE FOR seq_pd_aa);



CREATE SEQUENCE seq_pd_token_sa AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE pd_token_sa
(
	id_porta BIGINT NOT NULL,
	id_servizio_applicativo BIGINT NOT NULL,
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	-- unique constraints
	CONSTRAINT uniq_pd_token_sa_1 UNIQUE (id_porta,id_servizio_applicativo),
	-- fk/pk keys constraints
	CONSTRAINT fk_pd_token_sa_1 FOREIGN KEY (id_servizio_applicativo) REFERENCES servizi_applicativi(id),
	CONSTRAINT fk_pd_token_sa_2 FOREIGN KEY (id_porta) REFERENCES porte_delegate(id),
	CONSTRAINT pk_pd_token_sa PRIMARY KEY (id)
);

-- index
CREATE INDEX INDEX_PD_TOKEN_SA ON pd_token_sa (id_porta);
CREATE TABLE pd_token_sa_init_seq (id BIGINT);
INSERT INTO pd_token_sa_init_seq VALUES (NEXT VALUE FOR seq_pd_token_sa);



CREATE SEQUENCE seq_pd_token_ruoli AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE pd_token_ruoli
(
	id_porta BIGINT NOT NULL,
	ruolo VARCHAR(255) NOT NULL,
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	-- unique constraints
	CONSTRAINT unique_pd_token_ruoli_1 UNIQUE (id_porta,ruolo),
	-- fk/pk keys constraints
	CONSTRAINT fk_pd_token_ruoli_1 FOREIGN KEY (id_porta) REFERENCES porte_delegate(id),
	CONSTRAINT pk_pd_token_ruoli PRIMARY KEY (id)
);

-- index
CREATE UNIQUE INDEX index_pd_token_ruoli_1 ON pd_token_ruoli (id_porta,ruolo);
CREATE TABLE pd_token_ruoli_init_seq (id BIGINT);
INSERT INTO pd_token_ruoli_init_seq VALUES (NEXT VALUE FOR seq_pd_token_ruoli);




-- govway
		  		
-- **** Porte Applicative ****

CREATE SEQUENCE seq_porte_applicative AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE porte_applicative
(
	nome_porta VARCHAR(2000) NOT NULL,
	descrizione VARCHAR(4000),
	-- Soggetto Virtuale
	id_soggetto_virtuale BIGINT,
	tipo_soggetto_virtuale VARCHAR(255),
	nome_soggetto_virtuale VARCHAR(255),
	-- Servizio
	id_servizio BIGINT,
	tipo_servizio VARCHAR(255) NOT NULL,
	servizio VARCHAR(255) NOT NULL,
	versione_servizio INT NOT NULL,
	id_accordo BIGINT,
	id_port_type BIGINT,
	-- azione
	azione VARCHAR(255),
	mode_azione VARCHAR(255),
	pattern_azione VARCHAR(255),
	nome_porta_delegante_azione VARCHAR(255),
	-- abilitato/disabilitato
	force_interface_based_azione VARCHAR(255),
	-- disable/packaging/unpackaging/verify
	mtom_request_mode VARCHAR(255),
	-- disable/packaging/unpackaging/verify
	mtom_response_mode VARCHAR(255),
	-- abilitato/disabilitato (se abilitato le WSSproperties sono presenti nelle tabelle ...._security_request/response)
	security VARCHAR(255),
	-- abilitato/disabilitato
	security_mtom_req VARCHAR(255),
	-- abilitato/disabilitato
	security_mtom_res VARCHAR(255),
	security_request_mode VARCHAR(255),
	security_response_mode VARCHAR(255),
	-- abilitato/disabilitato
	ricevuta_asincrona_sim VARCHAR(255),
	-- abilitato/disabilitato
	ricevuta_asincrona_asim VARCHAR(255),
	-- abilitato/disabilitato/warningOnly
	validazione_contenuti_stato VARCHAR(255),
	-- wsdl/interface/xsd
	validazione_contenuti_tipo VARCHAR(255),
	-- abilitato/disabilitato
	validazione_contenuti_mtom VARCHAR(255),
	-- lista di tipi separati dalla virgola
	integrazione VARCHAR(4000),
	-- scadenza correlazione applicativa
	scadenza_correlazione_appl VARCHAR(255),
	-- abilitato/disabilitato
	allega_body VARCHAR(255),
	-- abilitato/disabilitato
	scarta_body VARCHAR(255),
	-- abilitato/disabilitato
	gestione_manifest VARCHAR(255),
	-- abilitato/disabilitato
	stateless VARCHAR(255),
	behaviour VARCHAR(255),
	-- Controllo Accessi
	autenticazione VARCHAR(255),
	-- abilitato/disabilitato
	autenticazione_opzionale VARCHAR(255),
	-- Gestione Token
	token_policy VARCHAR(255),
	token_opzionale VARCHAR(255),
	token_validazione VARCHAR(255),
	token_introspection VARCHAR(255),
	token_user_info VARCHAR(255),
	token_forward VARCHAR(255),
	token_options VARCHAR(4000),
	token_authn_issuer VARCHAR(255),
	token_authn_client_id VARCHAR(255),
	token_authn_subject VARCHAR(255),
	token_authn_username VARCHAR(255),
	token_authn_email VARCHAR(255),
	-- Autorizzazione
	autorizzazione VARCHAR(255),
	autorizzazione_xacml LONGVARCHAR,
	autorizzazione_contenuto VARCHAR(255),
	-- all/any
	ruoli_match VARCHAR(255),
	token_sa_stato VARCHAR(255),
	token_ruoli_stato VARCHAR(255),
	-- all/any
	token_ruoli_match VARCHAR(255),
	token_ruoli_tipologia VARCHAR(255),
	scope_stato VARCHAR(255),
	-- all/any
	scope_match VARCHAR(255),
	-- abilitato/disabilitato
	ricerca_porta_azione_delegata VARCHAR(255),
	-- Livello Log Messaggi Diagnostici
	msg_diag_severita VARCHAR(255),
	-- Tracciamento delle transazioni
	tracciamento_stato VARCHAR(255),
	tracciamento_esiti VARCHAR(255),
	transazioni_tempi VARCHAR(255),
	transazioni_token VARCHAR(255),
	-- Gestione CORS
	cors_stato VARCHAR(255),
	cors_tipo VARCHAR(255),
	cors_all_allow_origins VARCHAR(255),
	cors_all_allow_methods VARCHAR(255),
	cors_all_allow_headers VARCHAR(255),
	cors_allow_credentials VARCHAR(255),
	cors_allow_max_age INT,
	cors_allow_max_age_seconds INT,
	cors_allow_origins VARCHAR(65535),
	cors_allow_headers VARCHAR(65535),
	cors_allow_methods VARCHAR(65535),
	cors_allow_expose_headers VARCHAR(65535),
	-- Response caching
	response_cache_stato VARCHAR(255),
	response_cache_seconds INT,
	response_cache_max_msg_size BIGINT,
	response_cache_hash_url VARCHAR(255),
	response_cache_hash_query VARCHAR(255),
	response_cache_hash_query_list VARCHAR(65535),
	response_cache_hash_headers VARCHAR(255),
	response_cache_hash_hdr_list VARCHAR(65535),
	response_cache_hash_payload VARCHAR(255),
	response_cache_control_nocache INT,
	response_cache_control_maxage INT,
	response_cache_control_nostore INT,
	id_sa_default BIGINT,
	-- Stato della porta: abilitato/disabilitato
	stato VARCHAR(255),
	-- proprietario porta applicativa
	id_soggetto BIGINT NOT NULL,
	ora_registrazione TIMESTAMP,
	options VARCHAR(4000),
	canale VARCHAR(255),
	utente_richiedente VARCHAR(255),
	data_creazione TIMESTAMP,
	utente_ultima_modifica VARCHAR(255),
	data_ultima_modifica TIMESTAMP,
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	-- unique constraints
	CONSTRAINT unique_porte_applicative_1 UNIQUE (nome_porta),
	-- fk/pk keys constraints
	CONSTRAINT fk_porte_applicative_1 FOREIGN KEY (id_soggetto) REFERENCES soggetti(id),
	CONSTRAINT pk_porte_applicative PRIMARY KEY (id)
);

-- index
CREATE UNIQUE INDEX index_porte_applicative_1 ON porte_applicative (nome_porta);
CREATE INDEX index_porte_applicative_2 ON porte_applicative (id_soggetto);
CREATE INDEX index_porte_applicative_3 ON porte_applicative (canale);

ALTER TABLE porte_applicative ALTER COLUMN versione_servizio SET DEFAULT 1;
ALTER TABLE porte_applicative ALTER COLUMN ora_registrazione SET DEFAULT CURRENT_TIMESTAMP;

CREATE TABLE porte_applicative_init_seq (id BIGINT);
INSERT INTO porte_applicative_init_seq VALUES (NEXT VALUE FOR seq_porte_applicative);



CREATE SEQUENCE seq_porte_applicative_sa AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE porte_applicative_sa
(
	id_porta BIGINT NOT NULL,
	id_servizio_applicativo BIGINT NOT NULL,
	-- Dati Connettore
	connettore_nome VARCHAR(255),
	connettore_notifica INT,
	connettore_descrizione VARCHAR(4000),
	connettore_stato VARCHAR(255),
	connettore_scheduling VARCHAR(255),
	connettore_filtri VARCHAR(65535),
	connettore_coda VARCHAR(10),
	connettore_priorita VARCHAR(10),
	connettore_max_priorita INT,
	utente_richiedente VARCHAR(255),
	data_creazione TIMESTAMP,
	utente_ultima_modifica VARCHAR(255),
	data_ultima_modifica TIMESTAMP,
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	-- unique constraints
	CONSTRAINT unique_porte_applicative_sa_1 UNIQUE (id_porta,id_servizio_applicativo),
	-- fk/pk keys constraints
	CONSTRAINT fk_porte_applicative_sa_1 FOREIGN KEY (id_servizio_applicativo) REFERENCES servizi_applicativi(id),
	CONSTRAINT fk_porte_applicative_sa_2 FOREIGN KEY (id_porta) REFERENCES porte_applicative(id),
	CONSTRAINT pk_porte_applicative_sa PRIMARY KEY (id)
);

-- index
CREATE INDEX INDEX_PA_SA ON porte_applicative_sa (id_porta);
CREATE INDEX INDEX_PA_SA_CODA ON porte_applicative_sa (connettore_coda,connettore_priorita,id_servizio_applicativo);
CREATE INDEX INDEX_PA_SA_CODA_MAX ON porte_applicative_sa (connettore_coda,connettore_max_priorita,id_servizio_applicativo);

ALTER TABLE porte_applicative_sa ALTER COLUMN connettore_coda SET DEFAULT 'DEFAULT';
ALTER TABLE porte_applicative_sa ALTER COLUMN connettore_priorita SET DEFAULT 'DEFAULT';
ALTER TABLE porte_applicative_sa ALTER COLUMN connettore_max_priorita SET DEFAULT 0;

CREATE TABLE porte_applicative_sa_init_seq (id BIGINT);
INSERT INTO porte_applicative_sa_init_seq VALUES (NEXT VALUE FOR seq_porte_applicative_sa);



CREATE SEQUENCE seq_pa_sa_properties AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE pa_sa_properties
(
	id_porta BIGINT NOT NULL,
	nome VARCHAR(255) NOT NULL,
	valore VARCHAR(255) NOT NULL,
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	-- unique constraints
	CONSTRAINT uniq_pa_sa_props_1 UNIQUE (id_porta,nome),
	-- fk/pk keys constraints
	CONSTRAINT fk_pa_sa_properties_1 FOREIGN KEY (id_porta) REFERENCES porte_applicative_sa(id),
	CONSTRAINT pk_pa_sa_properties PRIMARY KEY (id)
);

-- index
CREATE INDEX INDEX_PA_SA_PROP ON pa_sa_properties (id_porta);
CREATE TABLE pa_sa_properties_init_seq (id BIGINT);
INSERT INTO pa_sa_properties_init_seq VALUES (NEXT VALUE FOR seq_pa_sa_properties);



CREATE SEQUENCE seq_pa_behaviour_props AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE pa_behaviour_props
(
	id_porta BIGINT NOT NULL,
	nome VARCHAR(255) NOT NULL,
	valore VARCHAR(4000) NOT NULL,
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	-- unique constraints
	CONSTRAINT uniq_pa_behaviour_props_1 UNIQUE (id_porta,nome),
	-- fk/pk keys constraints
	CONSTRAINT fk_pa_behaviour_props_1 FOREIGN KEY (id_porta) REFERENCES porte_applicative(id),
	CONSTRAINT pk_pa_behaviour_props PRIMARY KEY (id)
);

-- index
CREATE INDEX INDEX_PA_BEHAVIOUR_PROP ON pa_behaviour_props (id_porta);
CREATE TABLE pa_behaviour_props_init_seq (id BIGINT);
INSERT INTO pa_behaviour_props_init_seq VALUES (NEXT VALUE FOR seq_pa_behaviour_props);



CREATE SEQUENCE seq_pa_auth_properties AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE pa_auth_properties
(
	id_porta BIGINT NOT NULL,
	nome VARCHAR(255) NOT NULL,
	valore VARCHAR(255) NOT NULL,
	enc_value VARCHAR(65535),
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	-- unique constraints
	CONSTRAINT uniq_pa_auth_props_1 UNIQUE (id_porta,nome),
	-- fk/pk keys constraints
	CONSTRAINT fk_pa_auth_properties_1 FOREIGN KEY (id_porta) REFERENCES porte_applicative(id),
	CONSTRAINT pk_pa_auth_properties PRIMARY KEY (id)
);

-- index
CREATE INDEX INDEX_PA_AUTH_PROP ON pa_auth_properties (id_porta);
CREATE TABLE pa_auth_properties_init_seq (id BIGINT);
INSERT INTO pa_auth_properties_init_seq VALUES (NEXT VALUE FOR seq_pa_auth_properties);



CREATE SEQUENCE seq_pa_authz_properties AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE pa_authz_properties
(
	id_porta BIGINT NOT NULL,
	nome VARCHAR(255) NOT NULL,
	valore VARCHAR(255) NOT NULL,
	enc_value VARCHAR(65535),
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	-- unique constraints
	CONSTRAINT uniq_pa_authz_props_1 UNIQUE (id_porta,nome),
	-- fk/pk keys constraints
	CONSTRAINT fk_pa_authz_properties_1 FOREIGN KEY (id_porta) REFERENCES porte_applicative(id),
	CONSTRAINT pk_pa_authz_properties PRIMARY KEY (id)
);

-- index
CREATE INDEX INDEX_PA_AUTHZ_PROP ON pa_authz_properties (id_porta);
CREATE TABLE pa_authz_properties_init_seq (id BIGINT);
INSERT INTO pa_authz_properties_init_seq VALUES (NEXT VALUE FOR seq_pa_authz_properties);



CREATE SEQUENCE seq_pa_authzc_properties AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE pa_authzc_properties
(
	id_porta BIGINT NOT NULL,
	nome VARCHAR(255) NOT NULL,
	valore VARCHAR(255) NOT NULL,
	enc_value VARCHAR(65535),
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	-- unique constraints
	CONSTRAINT uniq_pa_authzc_props_1 UNIQUE (id_porta,nome),
	-- fk/pk keys constraints
	CONSTRAINT fk_pa_authzc_properties_1 FOREIGN KEY (id_porta) REFERENCES porte_applicative(id),
	CONSTRAINT pk_pa_authzc_properties PRIMARY KEY (id)
);

-- index
CREATE INDEX INDEX_PA_AUTHZC_PROP ON pa_authzc_properties (id_porta);
CREATE TABLE pa_authzc_properties_init_seq (id BIGINT);
INSERT INTO pa_authzc_properties_init_seq VALUES (NEXT VALUE FOR seq_pa_authzc_properties);



CREATE SEQUENCE seq_pa_ct_properties AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE pa_ct_properties
(
	id_porta BIGINT NOT NULL,
	nome VARCHAR(255) NOT NULL,
	valore VARCHAR(255) NOT NULL,
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	-- unique constraints
	CONSTRAINT uniq_pa_ct_props_1 UNIQUE (id_porta,nome),
	-- fk/pk keys constraints
	CONSTRAINT fk_pa_ct_properties_1 FOREIGN KEY (id_porta) REFERENCES porte_applicative(id),
	CONSTRAINT pk_pa_ct_properties PRIMARY KEY (id)
);

-- index
CREATE INDEX idx_pa_ct_props_1 ON pa_ct_properties (valore);
CREATE INDEX INDEX_PA_CT_PROP ON pa_ct_properties (id_porta);
CREATE TABLE pa_ct_properties_init_seq (id BIGINT);
INSERT INTO pa_ct_properties_init_seq VALUES (NEXT VALUE FOR seq_pa_ct_properties);



CREATE SEQUENCE seq_pa_properties AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE pa_properties
(
	id_porta BIGINT NOT NULL,
	nome VARCHAR(255) NOT NULL,
	valore VARCHAR(255) NOT NULL,
	enc_value VARCHAR(65535),
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	-- unique constraints
	CONSTRAINT uniq_pa_properties_1 UNIQUE (id_porta,nome),
	-- fk/pk keys constraints
	CONSTRAINT fk_pa_properties_1 FOREIGN KEY (id_porta) REFERENCES porte_applicative(id),
	CONSTRAINT pk_pa_properties PRIMARY KEY (id)
);

-- index
CREATE INDEX INDEX_PA_PROP ON pa_properties (id_porta);
CREATE TABLE pa_properties_init_seq (id BIGINT);
INSERT INTO pa_properties_init_seq VALUES (NEXT VALUE FOR seq_pa_properties);



CREATE SEQUENCE seq_pa_mtom_request AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE pa_mtom_request
(
	id_porta BIGINT NOT NULL,
	nome VARCHAR(255) NOT NULL,
	pattern VARCHAR(65535) NOT NULL,
	content_type VARCHAR(255),
	required INT NOT NULL,
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	-- fk/pk keys constraints
	CONSTRAINT fk_pa_mtom_request_1 FOREIGN KEY (id_porta) REFERENCES porte_applicative(id),
	CONSTRAINT pk_pa_mtom_request PRIMARY KEY (id)
);

-- index
CREATE INDEX INDEX_PA_MTOMTREQ ON pa_mtom_request (id_porta);
CREATE TABLE pa_mtom_request_init_seq (id BIGINT);
INSERT INTO pa_mtom_request_init_seq VALUES (NEXT VALUE FOR seq_pa_mtom_request);



CREATE SEQUENCE seq_pa_mtom_response AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE pa_mtom_response
(
	id_porta BIGINT NOT NULL,
	nome VARCHAR(255) NOT NULL,
	pattern VARCHAR(65535) NOT NULL,
	content_type VARCHAR(255),
	required INT NOT NULL,
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	-- fk/pk keys constraints
	CONSTRAINT fk_pa_mtom_response_1 FOREIGN KEY (id_porta) REFERENCES porte_applicative(id),
	CONSTRAINT pk_pa_mtom_response PRIMARY KEY (id)
);

-- index
CREATE INDEX INDEX_PA_MTOMTRES ON pa_mtom_response (id_porta);
CREATE TABLE pa_mtom_response_init_seq (id BIGINT);
INSERT INTO pa_mtom_response_init_seq VALUES (NEXT VALUE FOR seq_pa_mtom_response);



CREATE SEQUENCE seq_pa_security_request AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE pa_security_request
(
	id_porta BIGINT NOT NULL,
	nome VARCHAR(255) NOT NULL,
	valore VARCHAR(65535) NOT NULL,
	enc_value VARCHAR(65535),
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	-- fk/pk keys constraints
	CONSTRAINT fk_pa_security_request_1 FOREIGN KEY (id_porta) REFERENCES porte_applicative(id),
	CONSTRAINT pk_pa_security_request PRIMARY KEY (id)
);

-- index
CREATE INDEX INDEX_PA_WSSREQ ON pa_security_request (id_porta);
CREATE TABLE pa_security_request_init_seq (id BIGINT);
INSERT INTO pa_security_request_init_seq VALUES (NEXT VALUE FOR seq_pa_security_request);



CREATE SEQUENCE seq_pa_security_response AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE pa_security_response
(
	id_porta BIGINT NOT NULL,
	nome VARCHAR(255) NOT NULL,
	valore VARCHAR(65535) NOT NULL,
	enc_value VARCHAR(65535),
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	-- fk/pk keys constraints
	CONSTRAINT fk_pa_security_response_1 FOREIGN KEY (id_porta) REFERENCES porte_applicative(id),
	CONSTRAINT pk_pa_security_response PRIMARY KEY (id)
);

-- index
CREATE INDEX INDEX_PA_WSSRES ON pa_security_response (id_porta);
CREATE TABLE pa_security_response_init_seq (id BIGINT);
INSERT INTO pa_security_response_init_seq VALUES (NEXT VALUE FOR seq_pa_security_response);



CREATE SEQUENCE seq_pa_correlazione AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE pa_correlazione
(
	id_porta BIGINT NOT NULL,
	nome_elemento VARCHAR(255),
	-- modalita di scelta user input, content-based, url-based, disabilitato
	mode_correlazione VARCHAR(255),
	-- pattern utilizzato solo per content/url based
	pattern VARCHAR(65535),
	-- blocca/accetta
	identificazione_fallita VARCHAR(255),
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	-- fk/pk keys constraints
	CONSTRAINT fk_pa_correlazione_1 FOREIGN KEY (id_porta) REFERENCES porte_applicative(id),
	CONSTRAINT pk_pa_correlazione PRIMARY KEY (id)
);

-- index
CREATE INDEX INDEX_PA_CORR_REQ ON pa_correlazione (id_porta);
CREATE TABLE pa_correlazione_init_seq (id BIGINT);
INSERT INTO pa_correlazione_init_seq VALUES (NEXT VALUE FOR seq_pa_correlazione);



CREATE SEQUENCE seq_pa_correlazione_risposta AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE pa_correlazione_risposta
(
	id_porta BIGINT NOT NULL,
	nome_elemento VARCHAR(255),
	-- modalita di scelta user input, content-based, url-based, disabilitato
	mode_correlazione VARCHAR(255),
	-- pattern utilizzato solo per content/url based
	pattern VARCHAR(65535),
	-- blocca/accetta
	identificazione_fallita VARCHAR(255),
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	-- fk/pk keys constraints
	CONSTRAINT fk_pa_correlazione_risposta_1 FOREIGN KEY (id_porta) REFERENCES porte_applicative(id),
	CONSTRAINT pk_pa_correlazione_risposta PRIMARY KEY (id)
);

-- index
CREATE INDEX INDEX_PA_CORR_RES ON pa_correlazione_risposta (id_porta);
CREATE TABLE pa_correlazione_risposta_init_seq (id BIGINT);
INSERT INTO pa_correlazione_risposta_init_seq VALUES (NEXT VALUE FOR seq_pa_correlazione_risposta);



CREATE SEQUENCE seq_pa_ruoli AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE pa_ruoli
(
	id_porta BIGINT NOT NULL,
	ruolo VARCHAR(255) NOT NULL,
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	-- unique constraints
	CONSTRAINT unique_pa_ruoli_1 UNIQUE (id_porta,ruolo),
	-- fk/pk keys constraints
	CONSTRAINT fk_pa_ruoli_1 FOREIGN KEY (id_porta) REFERENCES porte_applicative(id),
	CONSTRAINT pk_pa_ruoli PRIMARY KEY (id)
);

-- index
CREATE UNIQUE INDEX index_pa_ruoli_1 ON pa_ruoli (id_porta,ruolo);
CREATE TABLE pa_ruoli_init_seq (id BIGINT);
INSERT INTO pa_ruoli_init_seq VALUES (NEXT VALUE FOR seq_pa_ruoli);



CREATE SEQUENCE seq_pa_scope AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE pa_scope
(
	id_porta BIGINT NOT NULL,
	scope VARCHAR(255) NOT NULL,
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	-- unique constraints
	CONSTRAINT unique_pa_scope_1 UNIQUE (id_porta,scope),
	-- fk/pk keys constraints
	CONSTRAINT fk_pa_scope_1 FOREIGN KEY (id_porta) REFERENCES porte_applicative(id),
	CONSTRAINT pk_pa_scope PRIMARY KEY (id)
);

-- index
CREATE UNIQUE INDEX index_pa_scope_1 ON pa_scope (id_porta,scope);
CREATE TABLE pa_scope_init_seq (id BIGINT);
INSERT INTO pa_scope_init_seq VALUES (NEXT VALUE FOR seq_pa_scope);



CREATE SEQUENCE seq_pa_soggetti AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE pa_soggetti
(
	id_porta BIGINT NOT NULL,
	tipo_soggetto VARCHAR(255) NOT NULL,
	nome_soggetto VARCHAR(255) NOT NULL,
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	-- unique constraints
	CONSTRAINT unique_pa_soggetti_1 UNIQUE (id_porta,tipo_soggetto,nome_soggetto),
	-- fk/pk keys constraints
	CONSTRAINT fk_pa_soggetti_1 FOREIGN KEY (id_porta) REFERENCES porte_applicative(id),
	CONSTRAINT pk_pa_soggetti PRIMARY KEY (id)
);

-- index
CREATE UNIQUE INDEX index_pa_soggetti_1 ON pa_soggetti (id_porta,tipo_soggetto,nome_soggetto);
CREATE TABLE pa_soggetti_init_seq (id BIGINT);
INSERT INTO pa_soggetti_init_seq VALUES (NEXT VALUE FOR seq_pa_soggetti);



CREATE SEQUENCE seq_porte_applicative_sa_auth AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE porte_applicative_sa_auth
(
	id_porta BIGINT NOT NULL,
	id_servizio_applicativo BIGINT NOT NULL,
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	-- unique constraints
	CONSTRAINT uniq_pa_sa_auth_1 UNIQUE (id_porta,id_servizio_applicativo),
	-- fk/pk keys constraints
	CONSTRAINT fk_porte_applicative_sa_auth_1 FOREIGN KEY (id_servizio_applicativo) REFERENCES servizi_applicativi(id),
	CONSTRAINT fk_porte_applicative_sa_auth_2 FOREIGN KEY (id_porta) REFERENCES porte_applicative(id),
	CONSTRAINT pk_porte_applicative_sa_auth PRIMARY KEY (id)
);

-- index
CREATE INDEX INDEX_PA_SA_AUTH ON porte_applicative_sa_auth (id_porta);
CREATE TABLE porte_applicative_sa_auth_init_seq (id BIGINT);
INSERT INTO porte_applicative_sa_auth_init_seq VALUES (NEXT VALUE FOR seq_porte_applicative_sa_auth);



CREATE SEQUENCE seq_pa_azioni AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE pa_azioni
(
	id_porta BIGINT NOT NULL,
	azione VARCHAR(255) NOT NULL,
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	-- unique constraints
	CONSTRAINT unique_pa_azioni_1 UNIQUE (id_porta,azione),
	-- fk/pk keys constraints
	CONSTRAINT fk_pa_azioni_1 FOREIGN KEY (id_porta) REFERENCES porte_applicative(id),
	CONSTRAINT pk_pa_azioni PRIMARY KEY (id)
);

-- index
CREATE UNIQUE INDEX index_pa_azioni_1 ON pa_azioni (id_porta,azione);
CREATE TABLE pa_azioni_init_seq (id BIGINT);
INSERT INTO pa_azioni_init_seq VALUES (NEXT VALUE FOR seq_pa_azioni);



CREATE SEQUENCE seq_pa_cache_regole AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE pa_cache_regole
(
	id_porta BIGINT NOT NULL,
	status_min INT,
	status_max INT,
	fault INT,
	cache_seconds INT,
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	-- fk/pk keys constraints
	CONSTRAINT fk_pa_cache_regole_1 FOREIGN KEY (id_porta) REFERENCES porte_applicative(id),
	CONSTRAINT pk_pa_cache_regole PRIMARY KEY (id)
);

-- index
CREATE INDEX INDEX_PA_CACHE ON pa_cache_regole (id_porta);

ALTER TABLE pa_cache_regole ALTER COLUMN fault SET DEFAULT 0;

CREATE TABLE pa_cache_regole_init_seq (id BIGINT);
INSERT INTO pa_cache_regole_init_seq VALUES (NEXT VALUE FOR seq_pa_cache_regole);



CREATE SEQUENCE seq_pa_transform AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE pa_transform
(
	id_porta BIGINT NOT NULL,
	nome VARCHAR(255) NOT NULL,
	posizione INT NOT NULL,
	stato VARCHAR(255),
	applicabilita_azioni VARCHAR(65535),
	applicabilita_ct VARCHAR(65535),
	applicabilita_pattern VARCHAR(65535),
	applicabilita_connettori VARCHAR(65535),
	req_conversione_enabled INT NOT NULL,
	req_conversione_tipo VARCHAR(255),
	-- In hsql 2.x usare il tipo BLOB al posto di VARBINARY
	req_conversione_template VARBINARY(1073741823),
	req_content_type VARCHAR(255),
	rest_transformation INT NOT NULL,
	rest_method VARCHAR(255),
	rest_path VARCHAR(4000),
	soap_transformation INT NOT NULL,
	soap_version VARCHAR(255),
	soap_action VARCHAR(255),
	soap_envelope INT NOT NULL,
	soap_envelope_as_attach INT NOT NULL,
	soap_envelope_tipo VARCHAR(255),
	-- In hsql 2.x usare il tipo BLOB al posto di VARBINARY
	soap_envelope_template VARBINARY(1073741823),
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	-- unique constraints
	CONSTRAINT unique_pa_transform_1 UNIQUE (id_porta,nome),
	CONSTRAINT unique_pa_transform_2 UNIQUE (id_porta,posizione),
	-- fk/pk keys constraints
	CONSTRAINT fk_pa_transform_1 FOREIGN KEY (id_porta) REFERENCES porte_applicative(id),
	CONSTRAINT pk_pa_transform PRIMARY KEY (id)
);

-- index
CREATE UNIQUE INDEX index_pa_transform_1 ON pa_transform (id_porta,nome);
CREATE UNIQUE INDEX index_pa_transform_2 ON pa_transform (id_porta,posizione);

ALTER TABLE pa_transform ALTER COLUMN req_conversione_enabled SET DEFAULT 0;
ALTER TABLE pa_transform ALTER COLUMN rest_transformation SET DEFAULT 0;
ALTER TABLE pa_transform ALTER COLUMN soap_transformation SET DEFAULT 0;
ALTER TABLE pa_transform ALTER COLUMN soap_envelope SET DEFAULT 0;
ALTER TABLE pa_transform ALTER COLUMN soap_envelope_as_attach SET DEFAULT 0;

CREATE TABLE pa_transform_init_seq (id BIGINT);
INSERT INTO pa_transform_init_seq VALUES (NEXT VALUE FOR seq_pa_transform);



CREATE SEQUENCE seq_pa_transform_soggetti AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE pa_transform_soggetti
(
	id_trasformazione BIGINT NOT NULL,
	tipo_soggetto VARCHAR(255) NOT NULL,
	nome_soggetto VARCHAR(255) NOT NULL,
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	-- unique constraints
	CONSTRAINT unique_pa_transform_soggetti_1 UNIQUE (id_trasformazione,tipo_soggetto,nome_soggetto),
	-- fk/pk keys constraints
	CONSTRAINT fk_pa_transform_soggetti_1 FOREIGN KEY (id_trasformazione) REFERENCES pa_transform(id),
	CONSTRAINT pk_pa_transform_soggetti PRIMARY KEY (id)
);

-- index
CREATE UNIQUE INDEX index_pa_transform_soggetti_1 ON pa_transform_soggetti (id_trasformazione,tipo_soggetto,nome_soggetto);
CREATE TABLE pa_transform_soggetti_init_seq (id BIGINT);
INSERT INTO pa_transform_soggetti_init_seq VALUES (NEXT VALUE FOR seq_pa_transform_soggetti);



CREATE SEQUENCE seq_pa_transform_sa AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE pa_transform_sa
(
	id_trasformazione BIGINT NOT NULL,
	id_servizio_applicativo BIGINT NOT NULL,
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	-- unique constraints
	CONSTRAINT unique_pa_transform_sa_1 UNIQUE (id_trasformazione,id_servizio_applicativo),
	-- fk/pk keys constraints
	CONSTRAINT fk_pa_transform_sa_1 FOREIGN KEY (id_servizio_applicativo) REFERENCES servizi_applicativi(id),
	CONSTRAINT fk_pa_transform_sa_2 FOREIGN KEY (id_trasformazione) REFERENCES pa_transform(id),
	CONSTRAINT pk_pa_transform_sa PRIMARY KEY (id)
);

-- index
CREATE UNIQUE INDEX index_pa_transform_sa_1 ON pa_transform_sa (id_trasformazione,id_servizio_applicativo);
CREATE TABLE pa_transform_sa_init_seq (id BIGINT);
INSERT INTO pa_transform_sa_init_seq VALUES (NEXT VALUE FOR seq_pa_transform_sa);



CREATE SEQUENCE seq_pa_transform_hdr AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE pa_transform_hdr
(
	id_trasformazione BIGINT NOT NULL,
	tipo VARCHAR(255) NOT NULL,
	nome VARCHAR(255) NOT NULL,
	valore VARCHAR(65535),
	identificazione_fallita VARCHAR(255),
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	-- fk/pk keys constraints
	CONSTRAINT fk_pa_transform_hdr_1 FOREIGN KEY (id_trasformazione) REFERENCES pa_transform(id),
	CONSTRAINT pk_pa_transform_hdr PRIMARY KEY (id)
);

-- index
CREATE INDEX idx_pa_trasf_hdr_1 ON pa_transform_hdr (id_trasformazione);
CREATE TABLE pa_transform_hdr_init_seq (id BIGINT);
INSERT INTO pa_transform_hdr_init_seq VALUES (NEXT VALUE FOR seq_pa_transform_hdr);



CREATE SEQUENCE seq_pa_transform_url AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE pa_transform_url
(
	id_trasformazione BIGINT NOT NULL,
	tipo VARCHAR(255) NOT NULL,
	nome VARCHAR(255) NOT NULL,
	valore VARCHAR(65535),
	identificazione_fallita VARCHAR(255),
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	-- fk/pk keys constraints
	CONSTRAINT fk_pa_transform_url_1 FOREIGN KEY (id_trasformazione) REFERENCES pa_transform(id),
	CONSTRAINT pk_pa_transform_url PRIMARY KEY (id)
);

-- index
CREATE INDEX idx_pa_trasf_url_1 ON pa_transform_url (id_trasformazione);
CREATE TABLE pa_transform_url_init_seq (id BIGINT);
INSERT INTO pa_transform_url_init_seq VALUES (NEXT VALUE FOR seq_pa_transform_url);



CREATE SEQUENCE seq_pa_transform_risp AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE pa_transform_risp
(
	id_trasformazione BIGINT NOT NULL,
	nome VARCHAR(255) NOT NULL,
	posizione INT NOT NULL,
	applicabilita_status_min INT,
	applicabilita_status_max INT,
	applicabilita_ct VARCHAR(65535),
	applicabilita_pattern VARCHAR(65535),
	conversione_enabled INT NOT NULL,
	conversione_tipo VARCHAR(255),
	-- In hsql 2.x usare il tipo BLOB al posto di VARBINARY
	conversione_template VARBINARY(1073741823),
	content_type VARCHAR(255),
	return_code VARCHAR(255),
	soap_envelope INT NOT NULL,
	soap_envelope_as_attach INT NOT NULL,
	soap_envelope_tipo VARCHAR(255),
	-- In hsql 2.x usare il tipo BLOB al posto di VARBINARY
	soap_envelope_template VARBINARY(1073741823),
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	-- unique constraints
	CONSTRAINT uniq_pa_trasf_resp_1 UNIQUE (id_trasformazione,nome),
	CONSTRAINT uniq_pa_trasf_resp_2 UNIQUE (id_trasformazione,posizione),
	-- fk/pk keys constraints
	CONSTRAINT fk_pa_transform_risp_1 FOREIGN KEY (id_trasformazione) REFERENCES pa_transform(id),
	CONSTRAINT pk_pa_transform_risp PRIMARY KEY (id)
);

-- index
CREATE UNIQUE INDEX idx_pa_trasf_resp_1 ON pa_transform_risp (id_trasformazione,nome);
CREATE UNIQUE INDEX idx_pa_trasf_resp_2 ON pa_transform_risp (id_trasformazione,posizione);

ALTER TABLE pa_transform_risp ALTER COLUMN conversione_enabled SET DEFAULT 0;
ALTER TABLE pa_transform_risp ALTER COLUMN soap_envelope SET DEFAULT 0;
ALTER TABLE pa_transform_risp ALTER COLUMN soap_envelope_as_attach SET DEFAULT 0;

CREATE TABLE pa_transform_risp_init_seq (id BIGINT);
INSERT INTO pa_transform_risp_init_seq VALUES (NEXT VALUE FOR seq_pa_transform_risp);



CREATE SEQUENCE seq_pa_transform_risp_hdr AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE pa_transform_risp_hdr
(
	id_transform_risp BIGINT NOT NULL,
	tipo VARCHAR(255) NOT NULL,
	nome VARCHAR(255) NOT NULL,
	valore VARCHAR(65535),
	identificazione_fallita VARCHAR(255),
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	-- fk/pk keys constraints
	CONSTRAINT fk_pa_transform_risp_hdr_1 FOREIGN KEY (id_transform_risp) REFERENCES pa_transform_risp(id),
	CONSTRAINT pk_pa_transform_risp_hdr PRIMARY KEY (id)
);

-- index
CREATE INDEX idx_pa_trasf_hdr_resp_1 ON pa_transform_risp_hdr (id_transform_risp);
CREATE TABLE pa_transform_risp_hdr_init_seq (id BIGINT);
INSERT INTO pa_transform_risp_hdr_init_seq VALUES (NEXT VALUE FOR seq_pa_transform_risp_hdr);



CREATE SEQUENCE seq_pa_handlers AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE pa_handlers
(
	id_porta BIGINT NOT NULL,
	tipologia VARCHAR(255) NOT NULL,
	tipo VARCHAR(255) NOT NULL,
	posizione INT NOT NULL,
	stato VARCHAR(255),
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	-- unique constraints
	CONSTRAINT unique_pa_handlers_1 UNIQUE (id_porta,tipologia,tipo),
	-- fk/pk keys constraints
	CONSTRAINT fk_pa_handlers_1 FOREIGN KEY (id_porta) REFERENCES porte_applicative(id),
	CONSTRAINT pk_pa_handlers PRIMARY KEY (id)
);

-- index
CREATE UNIQUE INDEX index_pa_handlers_1 ON pa_handlers (id_porta,tipologia,tipo);
CREATE TABLE pa_handlers_init_seq (id BIGINT);
INSERT INTO pa_handlers_init_seq VALUES (NEXT VALUE FOR seq_pa_handlers);



CREATE SEQUENCE seq_pa_aa AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE pa_aa
(
	id_porta BIGINT NOT NULL,
	nome VARCHAR(255) NOT NULL,
	attributi VARCHAR(65535),
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	-- unique constraints
	CONSTRAINT unique_pa_aa_1 UNIQUE (id_porta,nome),
	-- fk/pk keys constraints
	CONSTRAINT fk_pa_aa_1 FOREIGN KEY (id_porta) REFERENCES porte_applicative(id),
	CONSTRAINT pk_pa_aa PRIMARY KEY (id)
);

-- index
CREATE UNIQUE INDEX index_pa_aa_1 ON pa_aa (id_porta,nome);
CREATE TABLE pa_aa_init_seq (id BIGINT);
INSERT INTO pa_aa_init_seq VALUES (NEXT VALUE FOR seq_pa_aa);



CREATE SEQUENCE seq_pa_token_sa AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE pa_token_sa
(
	id_porta BIGINT NOT NULL,
	id_servizio_applicativo BIGINT NOT NULL,
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	-- unique constraints
	CONSTRAINT uniq_pa_token_sa_1 UNIQUE (id_porta,id_servizio_applicativo),
	-- fk/pk keys constraints
	CONSTRAINT fk_pa_token_sa_1 FOREIGN KEY (id_servizio_applicativo) REFERENCES servizi_applicativi(id),
	CONSTRAINT fk_pa_token_sa_2 FOREIGN KEY (id_porta) REFERENCES porte_applicative(id),
	CONSTRAINT pk_pa_token_sa PRIMARY KEY (id)
);

-- index
CREATE INDEX INDEX_PA_TOKEN_SA ON pa_token_sa (id_porta);
CREATE TABLE pa_token_sa_init_seq (id BIGINT);
INSERT INTO pa_token_sa_init_seq VALUES (NEXT VALUE FOR seq_pa_token_sa);



CREATE SEQUENCE seq_pa_token_ruoli AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE pa_token_ruoli
(
	id_porta BIGINT NOT NULL,
	ruolo VARCHAR(255) NOT NULL,
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	-- unique constraints
	CONSTRAINT unique_pa_token_ruoli_1 UNIQUE (id_porta,ruolo),
	-- fk/pk keys constraints
	CONSTRAINT fk_pa_token_ruoli_1 FOREIGN KEY (id_porta) REFERENCES porte_applicative(id),
	CONSTRAINT pk_pa_token_ruoli PRIMARY KEY (id)
);

-- index
CREATE UNIQUE INDEX index_pa_token_ruoli_1 ON pa_token_ruoli (id_porta,ruolo);
CREATE TABLE pa_token_ruoli_init_seq (id BIGINT);
INSERT INTO pa_token_ruoli_init_seq VALUES (NEXT VALUE FOR seq_pa_token_ruoli);




-- govway
		  		
-- **** Mapping con Porte ****

CREATE SEQUENCE seq_mapping_fruizione_pd AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE mapping_fruizione_pd
(
	id_fruizione BIGINT NOT NULL,
	id_porta BIGINT NOT NULL,
	is_default INT NOT NULL,
	nome VARCHAR(255) NOT NULL,
	descrizione VARCHAR(255) NOT NULL,
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	-- fk/pk keys constraints
	CONSTRAINT fk_mapping_fruizione_pd_1 FOREIGN KEY (id_fruizione) REFERENCES servizi_fruitori(id),
	CONSTRAINT fk_mapping_fruizione_pd_2 FOREIGN KEY (id_porta) REFERENCES porte_delegate(id),
	CONSTRAINT pk_mapping_fruizione_pd PRIMARY KEY (id)
);

-- index
CREATE INDEX index_mapping_fruizione_pd_1 ON mapping_fruizione_pd (id_fruizione);
CREATE INDEX index_mapping_fruizione_pd_2 ON mapping_fruizione_pd (id_porta);
CREATE TABLE mapping_fruizione_pd_init_seq (id BIGINT);
INSERT INTO mapping_fruizione_pd_init_seq VALUES (NEXT VALUE FOR seq_mapping_fruizione_pd);



CREATE SEQUENCE seq_mapping_erogazione_pa AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE mapping_erogazione_pa
(
	id_erogazione BIGINT NOT NULL,
	id_porta BIGINT NOT NULL,
	is_default INT NOT NULL,
	nome VARCHAR(255) NOT NULL,
	descrizione VARCHAR(255) NOT NULL,
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	-- fk/pk keys constraints
	CONSTRAINT fk_mapping_erogazione_pa_1 FOREIGN KEY (id_erogazione) REFERENCES servizi(id),
	CONSTRAINT fk_mapping_erogazione_pa_2 FOREIGN KEY (id_porta) REFERENCES porte_applicative(id),
	CONSTRAINT pk_mapping_erogazione_pa PRIMARY KEY (id)
);

-- index
CREATE INDEX index_mapping_erogazione_pa_1 ON mapping_erogazione_pa (id_erogazione);
CREATE INDEX index_mapping_erogazione_pa_2 ON mapping_erogazione_pa (id_porta);
CREATE TABLE mapping_erogazione_pa_init_seq (id BIGINT);
INSERT INTO mapping_erogazione_pa_init_seq VALUES (NEXT VALUE FOR seq_mapping_erogazione_pa);




-- govway
		  		
-- **** Audit Appenders ****

CREATE SEQUENCE seq_audit_operations AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) CYCLE;

CREATE TABLE audit_operations
(
	tipo_operazione VARCHAR(255) NOT NULL,
	-- non definito in caso di LOGIN/LOGOUT
	tipo VARCHAR(255),
	-- non definito in caso di LOGIN/LOGOUT
	object_id VARCHAR(2000),
	object_old_id VARCHAR(2000),
	utente VARCHAR(255) NOT NULL,
	stato VARCHAR(255) NOT NULL,
	-- Dettaglio oggetto in gestione (Rappresentazione tramite JSON o XML format)
	object_details LONGVARCHAR,
	-- Class, serve eventualmente per riottenere l'oggetto dal dettaglio
	-- non definito in caso di LOGIN/LOGOUT
	object_class VARCHAR(255),
	-- Eventuale messaggio di errore
	error VARCHAR(65535),
	time_request TIMESTAMP NOT NULL,
	time_execute TIMESTAMP NOT NULL,
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	-- check constraints
	CONSTRAINT chk_audit_operations_1 CHECK (tipo_operazione IN ('LOGIN','LOGOUT','ADD','CHANGE','DEL')),
	CONSTRAINT chk_audit_operations_2 CHECK (stato IN ('requesting','error','completed')),
	-- fk/pk keys constraints
	CONSTRAINT pk_audit_operations PRIMARY KEY (id)
);

-- index
CREATE INDEX audit_filter_time ON audit_operations (time_request DESC);
CREATE INDEX audit_object_id ON audit_operations (object_id);
CREATE INDEX audit_object_old_id ON audit_operations (object_old_id);
CREATE INDEX audit_filter ON audit_operations (tipo_operazione,tipo,object_id,utente,stato);
CREATE TABLE audit_operations_init_seq (id BIGINT);
INSERT INTO audit_operations_init_seq VALUES (NEXT VALUE FOR seq_audit_operations);



CREATE SEQUENCE seq_audit_binaries AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) CYCLE;

CREATE TABLE audit_binaries
(
	binary_id VARCHAR(255) NOT NULL,
	checksum BIGINT NOT NULL,
	id_audit_operation BIGINT NOT NULL,
	time_rec TIMESTAMP,
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	-- unique constraints
	CONSTRAINT unique_audit_binaries_1 UNIQUE (binary_id,id_audit_operation),
	-- fk/pk keys constraints
	CONSTRAINT fk_audit_binaries_1 FOREIGN KEY (id_audit_operation) REFERENCES audit_operations(id),
	CONSTRAINT pk_audit_binaries PRIMARY KEY (id)
);

-- index
CREATE UNIQUE INDEX index_audit_binaries_1 ON audit_binaries (binary_id,id_audit_operation);

ALTER TABLE audit_binaries ALTER COLUMN time_rec SET DEFAULT CURRENT_TIMESTAMP;

CREATE TABLE audit_binaries_init_seq (id BIGINT);
INSERT INTO audit_binaries_init_seq VALUES (NEXT VALUE FOR seq_audit_binaries);




-- govway
		  		
-- **** Audit Configuration ****

CREATE SEQUENCE seq_audit_conf AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE audit_conf
(
	audit_engine INT NOT NULL,
	enabled INT NOT NULL,
	dump INT NOT NULL,
	dump_format VARCHAR(255) NOT NULL,
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	-- fk/pk keys constraints
	CONSTRAINT pk_audit_conf PRIMARY KEY (id)
);


ALTER TABLE audit_conf ALTER COLUMN audit_engine SET DEFAULT 0;
ALTER TABLE audit_conf ALTER COLUMN enabled SET DEFAULT 0;
ALTER TABLE audit_conf ALTER COLUMN dump SET DEFAULT 0;
ALTER TABLE audit_conf ALTER COLUMN dump_format SET DEFAULT 'JSON';

CREATE TABLE audit_conf_init_seq (id BIGINT);
INSERT INTO audit_conf_init_seq VALUES (NEXT VALUE FOR seq_audit_conf);



CREATE SEQUENCE seq_audit_filters AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE audit_filters
(
	-- Filter
	username VARCHAR(255),
	tipo_operazione VARCHAR(255),
	tipo VARCHAR(255),
	stato VARCHAR(255),
	-- Filtri basati su contenuto utilizzabili solo se il dump della configurazione generale e' abilitato
	dump_search VARCHAR(255),
	-- 1(espressione regolare)/0(semplice stringa da ricercare)
	dump_expr INT NOT NULL,
	-- Action
	enabled INT NOT NULL,
	dump INT NOT NULL,
	-- Tempo di registrazione
	ora_registrazione TIMESTAMP,
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	-- check constraints
	CONSTRAINT chk_audit_filters_1 CHECK (tipo_operazione IN ('ADD','CHANGE','DEL')),
	CONSTRAINT chk_audit_filters_2 CHECK (stato IN ('requesting','error','completed')),
	-- fk/pk keys constraints
	CONSTRAINT pk_audit_filters PRIMARY KEY (id)
);


ALTER TABLE audit_filters ALTER COLUMN dump_expr SET DEFAULT 0;
ALTER TABLE audit_filters ALTER COLUMN enabled SET DEFAULT 0;
ALTER TABLE audit_filters ALTER COLUMN dump SET DEFAULT 0;
ALTER TABLE audit_filters ALTER COLUMN ora_registrazione SET DEFAULT CURRENT_TIMESTAMP;

CREATE TABLE audit_filters_init_seq (id BIGINT);
INSERT INTO audit_filters_init_seq VALUES (NEXT VALUE FOR seq_audit_filters);



CREATE SEQUENCE seq_audit_appender AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE audit_appender
(
	name VARCHAR(255) NOT NULL,
	class VARCHAR(255) NOT NULL,
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	-- unique constraints
	CONSTRAINT unique_audit_appender_1 UNIQUE (name),
	-- fk/pk keys constraints
	CONSTRAINT pk_audit_appender PRIMARY KEY (id)
);

CREATE TABLE audit_appender_init_seq (id BIGINT);
INSERT INTO audit_appender_init_seq VALUES (NEXT VALUE FOR seq_audit_appender);



CREATE SEQUENCE seq_audit_appender_prop AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE audit_appender_prop
(
	name VARCHAR(255) NOT NULL,
	value VARCHAR(255) NOT NULL,
	id_audit_appender BIGINT NOT NULL,
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	-- fk/pk keys constraints
	CONSTRAINT fk_audit_appender_prop_1 FOREIGN KEY (id_audit_appender) REFERENCES audit_appender(id),
	CONSTRAINT pk_audit_appender_prop PRIMARY KEY (id)
);

CREATE TABLE audit_appender_prop_init_seq (id BIGINT);
INSERT INTO audit_appender_prop_init_seq VALUES (NEXT VALUE FOR seq_audit_appender_prop);




-- govway
		  		
-- **** Utenti ****

CREATE SEQUENCE seq_users AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE users
(
	login VARCHAR(255) NOT NULL,
	password VARCHAR(255) NOT NULL,
	data_password TIMESTAMP NOT NULL,
	check_data_password INT NOT NULL,
	tipo_interfaccia VARCHAR(255) NOT NULL,
	interfaccia_completa INT,
	permessi VARCHAR(255) NOT NULL,
	protocolli VARCHAR(65535),
	protocollo_pddconsole VARCHAR(255),
	protocollo_pddmonitor VARCHAR(255),
	soggetto_pddconsole VARCHAR(255),
	soggetto_pddmonitor VARCHAR(255),
	soggetti_all INT,
	servizi_all INT,
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	-- unique constraints
	CONSTRAINT unique_users_1 UNIQUE (login),
	-- fk/pk keys constraints
	CONSTRAINT pk_users PRIMARY KEY (id)
);


ALTER TABLE users ALTER COLUMN data_password SET DEFAULT CURRENT_TIMESTAMP;
ALTER TABLE users ALTER COLUMN check_data_password SET DEFAULT 1;

CREATE TABLE users_init_seq (id BIGINT);
INSERT INTO users_init_seq VALUES (NEXT VALUE FOR seq_users);



CREATE SEQUENCE seq_users_stati AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE users_stati
(
	oggetto VARCHAR(255) NOT NULL,
	stato LONGVARCHAR NOT NULL,
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	id_utente BIGINT NOT NULL,
	-- fk/pk keys constraints
	CONSTRAINT fk_users_stati_1 FOREIGN KEY (id_utente) REFERENCES users(id),
	CONSTRAINT pk_users_stati PRIMARY KEY (id)
);

CREATE TABLE users_stati_init_seq (id BIGINT);
INSERT INTO users_stati_init_seq VALUES (NEXT VALUE FOR seq_users_stati);



CREATE SEQUENCE seq_users_password AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE users_password
(
	password VARCHAR(255) NOT NULL,
	data_password TIMESTAMP NOT NULL,
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	id_utente BIGINT NOT NULL,
	-- fk/pk keys constraints
	CONSTRAINT fk_users_password_1 FOREIGN KEY (id_utente) REFERENCES users(id),
	CONSTRAINT pk_users_password PRIMARY KEY (id)
);

CREATE TABLE users_password_init_seq (id BIGINT);
INSERT INTO users_password_init_seq VALUES (NEXT VALUE FOR seq_users_password);



CREATE SEQUENCE seq_users_soggetti AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE users_soggetti
(
	id_utente BIGINT NOT NULL,
	id_soggetto BIGINT NOT NULL,
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	-- unique constraints
	CONSTRAINT unique_users_soggetti_1 UNIQUE (id_utente,id_soggetto),
	-- fk/pk keys constraints
	CONSTRAINT fk_users_soggetti_1 FOREIGN KEY (id_soggetto) REFERENCES soggetti(id),
	CONSTRAINT fk_users_soggetti_2 FOREIGN KEY (id_utente) REFERENCES users(id),
	CONSTRAINT pk_users_soggetti PRIMARY KEY (id)
);

-- index
CREATE UNIQUE INDEX index_users_soggetti_1 ON users_soggetti (id_utente,id_soggetto);
CREATE TABLE users_soggetti_init_seq (id BIGINT);
INSERT INTO users_soggetti_init_seq VALUES (NEXT VALUE FOR seq_users_soggetti);



CREATE SEQUENCE seq_users_servizi AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE users_servizi
(
	id_utente BIGINT NOT NULL,
	id_servizio BIGINT NOT NULL,
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	-- unique constraints
	CONSTRAINT unique_users_servizi_1 UNIQUE (id_utente,id_servizio),
	-- fk/pk keys constraints
	CONSTRAINT fk_users_servizi_1 FOREIGN KEY (id_servizio) REFERENCES servizi(id),
	CONSTRAINT fk_users_servizi_2 FOREIGN KEY (id_utente) REFERENCES users(id),
	CONSTRAINT pk_users_servizi PRIMARY KEY (id)
);

-- index
CREATE UNIQUE INDEX index_users_servizi_1 ON users_servizi (id_utente,id_servizio);
CREATE TABLE users_servizi_init_seq (id BIGINT);
INSERT INTO users_servizi_init_seq VALUES (NEXT VALUE FOR seq_users_servizi);



CREATE SEQUENCE seq_users_ricerche AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE users_ricerche
(
	id_utente BIGINT NOT NULL,
	label VARCHAR(255) NOT NULL,
	descrizione VARCHAR(4000) NOT NULL,
	data_creazione TIMESTAMP NOT NULL,
	modulo VARCHAR(255) NOT NULL,
	modalita_ricerca VARCHAR(255) NOT NULL,
	visibilita VARCHAR(255) NOT NULL,
	ricerca LONGVARCHAR NOT NULL,
	protocollo VARCHAR(255),
	soggetto VARCHAR(255),
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	-- unique constraints
	CONSTRAINT unique_users_ricerche_1 UNIQUE (id_utente,label,modulo,modalita_ricerca),
	-- fk/pk keys constraints
	CONSTRAINT fk_users_ricerche_1 FOREIGN KEY (id_utente) REFERENCES users(id),
	CONSTRAINT pk_users_ricerche PRIMARY KEY (id)
);

-- index
CREATE UNIQUE INDEX index_users_ricerche_1 ON users_ricerche (id_utente,label,modulo,modalita_ricerca);

ALTER TABLE users_ricerche ALTER COLUMN data_creazione SET DEFAULT CURRENT_TIMESTAMP;

CREATE TABLE users_ricerche_init_seq (id BIGINT);
INSERT INTO users_ricerche_init_seq VALUES (NEXT VALUE FOR seq_users_ricerche);




-- govway
		  		
CREATE SEQUENCE seq_db_info_console AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE db_info_console
(
	major_version INT NOT NULL,
	minor_version INT NOT NULL,
	notes VARCHAR(255) NOT NULL,
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	-- fk/pk keys constraints
	CONSTRAINT pk_db_info_console PRIMARY KEY (id)
);

CREATE TABLE db_info_console_init_seq (id BIGINT);
INSERT INTO db_info_console_init_seq VALUES (NEXT VALUE FOR seq_db_info_console);




-- govway
		  		
INSERT INTO OP2_SEMAPHORE (applicative_id) VALUES ('GestoreChaviPDND');
INSERT INTO OP2_SEMAPHORE (applicative_id) VALUES ('ServiziDigestParamsUpdate');



-- govway
			  		

INSERT INTO audit_appender (name,class) VALUES ('dbAppender','org.openspcoop2.web.lib.audit.appender.AuditDBAppender');
INSERT INTO audit_appender_prop (name,value,id_audit_appender) VALUES ('datasource','org.govway.datasource.console',(select id from audit_appender where name='dbAppender'));
INSERT INTO audit_appender_prop (name,value,id_audit_appender) VALUES ('tipoDatabase','hsql',(select id from audit_appender where name='dbAppender'));


-- govway
			  		

INSERT INTO audit_appender (name,class) VALUES ('log4jAppender','org.openspcoop2.web.lib.audit.appender.AuditLog4JAppender');
INSERT INTO audit_appender_prop (name,value,id_audit_appender) VALUES ('fileConfigurazione','console.audit.log4j2.properties',(select id from audit_appender where name='log4jAppender'));
INSERT INTO audit_appender_prop (name,value,id_audit_appender) VALUES ('nomeFileLoaderInstance','console_local.audit.log4j2.properties',(select id from audit_appender where name='log4jAppender'));
INSERT INTO audit_appender_prop (name,value,id_audit_appender) VALUES ('nomeProprietaLoaderInstance','OPENSPCOOP2_CONSOLE_AUDIT_LOG_PROPERTIES',(select id from audit_appender where name='log4jAppender'));
INSERT INTO audit_appender_prop (name,value,id_audit_appender) VALUES ('category','audit',(select id from audit_appender where name='log4jAppender'));
INSERT INTO audit_appender_prop (name,value,id_audit_appender) VALUES ('xml','true',(select id from audit_appender where name='log4jAppender'));


-- govway
			  		

INSERT INTO audit_conf (audit_engine,enabled,dump,dump_format) VALUES (1,1,0,'JSON');


-- govway
			  		
-- users
INSERT INTO users (login, password, tipo_interfaccia, permessi) VALUES ('amministratore',	'$1$.bquKJDS$yZ4EYC3354HqEjmSRL/sR0','STANDARD','SCMAU');


-- govway
			  		


INSERT INTO db_info_console (major_version,minor_version,notes) VALUES (3,4,'[3.4.1.p1] Database della Console di Gestione di GovWay');

-- **** Configurazione ****

CREATE SEQUENCE seq_registri AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE registri
(
	nome VARCHAR(255) NOT NULL,
	location VARCHAR(255) NOT NULL,
	tipo VARCHAR(255) NOT NULL,
	utente VARCHAR(255),
	password VARCHAR(255),
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	-- unique constraints
	CONSTRAINT unique_registri_1 UNIQUE (nome),
	-- fk/pk keys constraints
	CONSTRAINT pk_registri PRIMARY KEY (id)
);

CREATE TABLE registri_init_seq (id BIGINT);
INSERT INTO registri_init_seq VALUES (NEXT VALUE FOR seq_registri);



CREATE SEQUENCE seq_routing AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE routing
(
	tipo VARCHAR(255),
	nome VARCHAR(255),
	-- registro/gateway
	tiporotta VARCHAR(255) NOT NULL,
	tiposoggrotta VARCHAR(255),
	nomesoggrotta VARCHAR(255),
	-- foreign key per i registri(id)
	registrorotta BIGINT,
	is_default BIGINT NOT NULL,
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	-- fk/pk keys constraints
	CONSTRAINT pk_routing PRIMARY KEY (id)
);

CREATE TABLE routing_init_seq (id BIGINT);
INSERT INTO routing_init_seq VALUES (NEXT VALUE FOR seq_routing);



CREATE SEQUENCE seq_config_cache_regole AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE config_cache_regole
(
	status_min INT,
	status_max INT,
	fault INT,
	cache_seconds INT,
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	-- fk/pk keys constraints
	CONSTRAINT pk_config_cache_regole PRIMARY KEY (id)
);


ALTER TABLE config_cache_regole ALTER COLUMN fault SET DEFAULT 0;

CREATE TABLE config_cache_regole_init_seq (id BIGINT);
INSERT INTO config_cache_regole_init_seq VALUES (NEXT VALUE FOR seq_config_cache_regole);



CREATE SEQUENCE seq_configurazione AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE configurazione
(
	-- Cadenza inoltro Riscontri/BusteAsincrone
	cadenza_inoltro VARCHAR(255) NOT NULL,
	-- Validazione Buste
	validazione_stato VARCHAR(255),
	validazione_controllo VARCHAR(255),
	validazione_profilo VARCHAR(255),
	validazione_manifest VARCHAR(255),
	-- Validazione Contenuti Applicativi
	-- abilitato/disabilitato/warningOnly
	validazione_contenuti_stato VARCHAR(255),
	-- wsdl/interface/xsd
	validazione_contenuti_tipo VARCHAR(255),
	-- abilitato/disabilitato
	validazione_contenuti_mtom VARCHAR(255),
	-- Livello Log Messaggi Diagnostici
	msg_diag_severita VARCHAR(255) NOT NULL,
	msg_diag_severita_log4j VARCHAR(255) NOT NULL,
	-- Tracciamento Buste
	tracciamento_buste VARCHAR(255),
	tracciamento_esiti_pd VARCHAR(255),
	tracciamento_esiti VARCHAR(255),
	-- Transazione
	transazioni_tempi_pd VARCHAR(255),
	transazioni_tempi VARCHAR(255),
	transazioni_token_pd VARCHAR(255),
	transazioni_token VARCHAR(255),
	-- Dump
	dump VARCHAR(255),
	dump_bin_pd VARCHAR(255),
	dump_bin_pa VARCHAR(255),
	-- Autenticazione IntegrationManager
	auth_integration_manager VARCHAR(255),
	-- Cache per l'accesso ai registri
	statocache VARCHAR(255),
	dimensionecache VARCHAR(255),
	algoritmocache VARCHAR(255),
	idlecache VARCHAR(255),
	lifecache VARCHAR(255),
	-- Cache per l'accesso alla configurazione
	config_statocache VARCHAR(255),
	config_dimensionecache VARCHAR(255),
	config_algoritmocache VARCHAR(255),
	config_idlecache VARCHAR(255),
	config_lifecache VARCHAR(255),
	-- Cache per l'accesso ai dati di autorizzazione
	auth_statocache VARCHAR(255),
	auth_dimensionecache VARCHAR(255),
	auth_algoritmocache VARCHAR(255),
	auth_idlecache VARCHAR(255),
	auth_lifecache VARCHAR(255),
	-- Cache per l'accesso ai dati di autenticazione
	authn_statocache VARCHAR(255),
	authn_dimensionecache VARCHAR(255),
	authn_algoritmocache VARCHAR(255),
	authn_idlecache VARCHAR(255),
	authn_lifecache VARCHAR(255),
	-- Cache per la gestione dei token
	token_statocache VARCHAR(255),
	token_dimensionecache VARCHAR(255),
	token_algoritmocache VARCHAR(255),
	token_idlecache VARCHAR(255),
	token_lifecache VARCHAR(255),
	-- Cache per la gestione dei dati ottenuti da attribute authority
	aa_statocache VARCHAR(255),
	aa_dimensionecache VARCHAR(255),
	aa_algoritmocache VARCHAR(255),
	aa_idlecache VARCHAR(255),
	aa_lifecache VARCHAR(255),
	-- Cache per la gestione dei keystore
	keystore_statocache VARCHAR(255),
	keystore_dimensionecache VARCHAR(255),
	keystore_algoritmocache VARCHAR(255),
	keystore_idlecache VARCHAR(255),
	keystore_lifecache VARCHAR(255),
	keystore_crl_lifecache VARCHAR(255),
	-- connessione su cui vengono inviate le risposte
	-- reply: connessione esistente (es. http reply)
	-- new: nuova connessione
	mod_risposta VARCHAR(255),
	-- Gestione dell'indirizzo
	indirizzo_telematico VARCHAR(255),
	-- Gestione Manifest
	gestione_manifest VARCHAR(255),
	-- Routing Table
	routing_enabled VARCHAR(255) NOT NULL,
	-- Gestione errore di default per la Porta di Dominio
	-- FOREIGN KEY (id_ge_cooperazione) REFERENCES gestione_errore(id) Nota: indica un eventuale tipologia di gestione dell'errore di default durante l'utilizzo di un connettore
	id_ge_cooperazione BIGINT,
	-- FOREIGN KEY (id_ge_integrazione) REFERENCES gestione_errore(id) Nota: indica un eventuale tipologia di gestione dell'errore di default durante l'utilizzo di un connettore
	id_ge_integrazione BIGINT,
	-- Gestione MultiTenant
	multitenant_stato VARCHAR(255),
	multitenant_fruizioni VARCHAR(255),
	multitenant_erogazioni VARCHAR(255),
	-- Gestione CORS
	cors_stato VARCHAR(255),
	cors_tipo VARCHAR(255),
	cors_all_allow_origins VARCHAR(255),
	cors_all_allow_methods VARCHAR(255),
	cors_all_allow_headers VARCHAR(255),
	cors_allow_credentials VARCHAR(255),
	cors_allow_max_age INT,
	cors_allow_max_age_seconds INT,
	cors_allow_origins VARCHAR(65535),
	cors_allow_headers VARCHAR(65535),
	cors_allow_methods VARCHAR(65535),
	cors_allow_expose_headers VARCHAR(65535),
	-- Response caching
	response_cache_stato VARCHAR(255),
	response_cache_seconds INT,
	response_cache_max_msg_size BIGINT,
	response_cache_hash_url VARCHAR(255),
	response_cache_hash_query VARCHAR(255),
	response_cache_hash_query_list VARCHAR(65535),
	response_cache_hash_headers VARCHAR(255),
	response_cache_hash_hdr_list VARCHAR(65535),
	response_cache_hash_payload VARCHAR(255),
	response_cache_control_nocache INT,
	response_cache_control_maxage INT,
	response_cache_control_nostore INT,
	-- Cache per il response caching
	response_cache_statocache VARCHAR(255),
	response_cache_dimensionecache VARCHAR(255),
	response_cache_algoritmocache VARCHAR(255),
	response_cache_idlecache VARCHAR(255),
	response_cache_lifecache VARCHAR(255),
	-- Cache per la consegna agli applicativi
	consegna_statocache VARCHAR(255),
	consegna_dimensionecache VARCHAR(255),
	consegna_algoritmocache VARCHAR(255),
	consegna_idlecache VARCHAR(255),
	consegna_lifecache VARCHAR(255),
	-- Gestione CORS
	canali_stato VARCHAR(255),
	-- Cache per la gestione delle richieste
	dati_richieste_statocache VARCHAR(255),
	dati_richieste_dimensionecache VARCHAR(255),
	dati_richieste_algoritmocache VARCHAR(255),
	dati_richieste_idlecache VARCHAR(255),
	dati_richieste_lifecache VARCHAR(255),
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	-- fk/pk keys constraints
	CONSTRAINT pk_configurazione PRIMARY KEY (id)
);

CREATE TABLE configurazione_init_seq (id BIGINT);
INSERT INTO configurazione_init_seq VALUES (NEXT VALUE FOR seq_configurazione);



-- **** URLInvocazione ****

CREATE SEQUENCE seq_config_url_invocazione AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE config_url_invocazione
(
	base_url VARCHAR(255) NOT NULL,
	base_url_fruizione VARCHAR(255),
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	-- fk/pk keys constraints
	CONSTRAINT pk_config_url_invocazione PRIMARY KEY (id)
);

CREATE TABLE config_url_invocazione_init_seq (id BIGINT);
INSERT INTO config_url_invocazione_init_seq VALUES (NEXT VALUE FOR seq_config_url_invocazione);



CREATE SEQUENCE seq_config_url_regole AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE config_url_regole
(
	nome VARCHAR(255) NOT NULL,
	posizione INT NOT NULL,
	stato VARCHAR(255),
	descrizione VARCHAR(65535),
	regexpr INT NOT NULL,
	regola VARCHAR(255) NOT NULL,
	contesto_esterno VARCHAR(255) NOT NULL,
	base_url VARCHAR(255),
	protocollo VARCHAR(255),
	ruolo VARCHAR(255),
	service_binding VARCHAR(255),
	tipo_soggetto VARCHAR(255),
	nome_soggetto VARCHAR(255),
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	-- unique constraints
	CONSTRAINT unique_config_url_regole_1 UNIQUE (nome),
	-- fk/pk keys constraints
	CONSTRAINT pk_config_url_regole PRIMARY KEY (id)
);

-- index
CREATE UNIQUE INDEX index_config_url_regole_1 ON config_url_regole (nome);
CREATE TABLE config_url_regole_init_seq (id BIGINT);
INSERT INTO config_url_regole_init_seq VALUES (NEXT VALUE FOR seq_config_url_regole);



-- **** Messaggi diagnostici ****

CREATE SEQUENCE seq_msgdiag_appender AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE msgdiag_appender
(
	tipo VARCHAR(255) NOT NULL,
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	-- fk/pk keys constraints
	CONSTRAINT pk_msgdiag_appender PRIMARY KEY (id)
);

CREATE TABLE msgdiag_appender_init_seq (id BIGINT);
INSERT INTO msgdiag_appender_init_seq VALUES (NEXT VALUE FOR seq_msgdiag_appender);



CREATE SEQUENCE seq_msgdiag_appender_prop AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE msgdiag_appender_prop
(
	id_appender BIGINT NOT NULL,
	nome VARCHAR(255) NOT NULL,
	valore VARCHAR(255) NOT NULL,
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	-- unique constraints
	CONSTRAINT uniq_msgdiag_app_prop_1 UNIQUE (id_appender,nome,valore),
	-- fk/pk keys constraints
	CONSTRAINT fk_msgdiag_appender_prop_1 FOREIGN KEY (id_appender) REFERENCES msgdiag_appender(id),
	CONSTRAINT pk_msgdiag_appender_prop PRIMARY KEY (id)
);

CREATE TABLE msgdiag_appender_prop_init_seq (id BIGINT);
INSERT INTO msgdiag_appender_prop_init_seq VALUES (NEXT VALUE FOR seq_msgdiag_appender_prop);



-- **** Tracciamento ****

CREATE SEQUENCE seq_tracce_appender AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE tracce_appender
(
	tipo VARCHAR(255) NOT NULL,
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	-- fk/pk keys constraints
	CONSTRAINT pk_tracce_appender PRIMARY KEY (id)
);

CREATE TABLE tracce_appender_init_seq (id BIGINT);
INSERT INTO tracce_appender_init_seq VALUES (NEXT VALUE FOR seq_tracce_appender);



CREATE SEQUENCE seq_tracce_appender_prop AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE tracce_appender_prop
(
	id_appender BIGINT NOT NULL,
	nome VARCHAR(255) NOT NULL,
	valore VARCHAR(255) NOT NULL,
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	-- unique constraints
	CONSTRAINT uniq_tracce_app_prop_1 UNIQUE (id_appender,nome,valore),
	-- fk/pk keys constraints
	CONSTRAINT fk_tracce_appender_prop_1 FOREIGN KEY (id_appender) REFERENCES tracce_appender(id),
	CONSTRAINT pk_tracce_appender_prop PRIMARY KEY (id)
);

CREATE TABLE tracce_appender_prop_init_seq (id BIGINT);
INSERT INTO tracce_appender_prop_init_seq VALUES (NEXT VALUE FOR seq_tracce_appender_prop);



CREATE SEQUENCE seq_tracce_config AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE tracce_config
(
	proprietario VARCHAR(255) NOT NULL,
	tipo VARCHAR(255) NOT NULL,
	id_proprietario BIGINT NOT NULL,
	stato VARCHAR(255),
	filtro_esiti VARCHAR(255),
	request_in VARCHAR(255),
	request_out VARCHAR(255),
	response_out VARCHAR(255),
	response_out_complete VARCHAR(255),
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	-- fk/pk keys constraints
	CONSTRAINT pk_tracce_config PRIMARY KEY (id)
);

-- index
CREATE INDEX index_tracce_config_1 ON tracce_config (proprietario,tipo);
CREATE TABLE tracce_config_init_seq (id BIGINT);
INSERT INTO tracce_config_init_seq VALUES (NEXT VALUE FOR seq_tracce_config);



CREATE SEQUENCE seq_filetrace_config AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE filetrace_config
(
	proprietario VARCHAR(255) NOT NULL,
	id_proprietario BIGINT NOT NULL,
	config VARCHAR(255),
	dump_in_stato VARCHAR(255),
	dump_in_stato_hdr VARCHAR(255),
	dump_in_stato_body VARCHAR(255),
	dump_out_stato VARCHAR(255),
	dump_out_stato_hdr VARCHAR(255),
	dump_out_stato_body VARCHAR(255),
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	-- fk/pk keys constraints
	CONSTRAINT pk_filetrace_config PRIMARY KEY (id)
);

-- index
CREATE INDEX index_filetrace_config_1 ON filetrace_config (proprietario);
CREATE TABLE filetrace_config_init_seq (id BIGINT);
INSERT INTO filetrace_config_init_seq VALUES (NEXT VALUE FOR seq_filetrace_config);



-- **** Dump ****

CREATE SEQUENCE seq_dump_config AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE dump_config
(
	proprietario VARCHAR(255) NOT NULL,
	id_proprietario BIGINT NOT NULL,
	dump_realtime VARCHAR(255),
	id_richiesta_ingresso BIGINT NOT NULL,
	id_richiesta_uscita BIGINT NOT NULL,
	id_risposta_ingresso BIGINT NOT NULL,
	id_risposta_uscita BIGINT NOT NULL,
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	-- fk/pk keys constraints
	CONSTRAINT pk_dump_config PRIMARY KEY (id)
);

-- index
CREATE INDEX index_dump_config_1 ON dump_config (proprietario);
CREATE TABLE dump_config_init_seq (id BIGINT);
INSERT INTO dump_config_init_seq VALUES (NEXT VALUE FOR seq_dump_config);



CREATE SEQUENCE seq_dump_config_regola AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE dump_config_regola
(
	payload VARCHAR(255) NOT NULL,
	payload_parsing VARCHAR(255) NOT NULL,
	body VARCHAR(255) NOT NULL,
	attachments VARCHAR(255) NOT NULL,
	headers VARCHAR(255) NOT NULL,
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	-- fk/pk keys constraints
	CONSTRAINT pk_dump_config_regola PRIMARY KEY (id)
);

CREATE TABLE dump_config_regola_init_seq (id BIGINT);
INSERT INTO dump_config_regola_init_seq VALUES (NEXT VALUE FOR seq_dump_config_regola);



CREATE SEQUENCE seq_dump_appender AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE dump_appender
(
	tipo VARCHAR(255) NOT NULL,
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	-- fk/pk keys constraints
	CONSTRAINT pk_dump_appender PRIMARY KEY (id)
);

CREATE TABLE dump_appender_init_seq (id BIGINT);
INSERT INTO dump_appender_init_seq VALUES (NEXT VALUE FOR seq_dump_appender);



CREATE SEQUENCE seq_dump_appender_prop AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE dump_appender_prop
(
	id_appender BIGINT NOT NULL,
	nome VARCHAR(255) NOT NULL,
	valore VARCHAR(255) NOT NULL,
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	-- unique constraints
	CONSTRAINT uniq_dump_app_prop_1 UNIQUE (id_appender,nome,valore),
	-- fk/pk keys constraints
	CONSTRAINT fk_dump_appender_prop_1 FOREIGN KEY (id_appender) REFERENCES dump_appender(id),
	CONSTRAINT pk_dump_appender_prop PRIMARY KEY (id)
);

CREATE TABLE dump_appender_prop_init_seq (id BIGINT);
INSERT INTO dump_appender_prop_init_seq VALUES (NEXT VALUE FOR seq_dump_appender_prop);



-- **** Datasources dove reperire i messaggi diagnostici salvati ****

CREATE SEQUENCE seq_msgdiag_ds AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE msgdiag_ds
(
	nome VARCHAR(255) NOT NULL,
	nome_jndi VARCHAR(255) NOT NULL,
	tipo_database VARCHAR(255) NOT NULL,
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	-- unique constraints
	CONSTRAINT unique_msgdiag_ds_1 UNIQUE (nome),
	-- fk/pk keys constraints
	CONSTRAINT pk_msgdiag_ds PRIMARY KEY (id)
);

CREATE TABLE msgdiag_ds_init_seq (id BIGINT);
INSERT INTO msgdiag_ds_init_seq VALUES (NEXT VALUE FOR seq_msgdiag_ds);



CREATE SEQUENCE seq_msgdiag_ds_prop AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE msgdiag_ds_prop
(
	id_prop BIGINT NOT NULL,
	nome VARCHAR(255) NOT NULL,
	valore VARCHAR(255) NOT NULL,
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	-- unique constraints
	CONSTRAINT unique_msgdiag_ds_prop_1 UNIQUE (id_prop,nome,valore),
	-- fk/pk keys constraints
	CONSTRAINT fk_msgdiag_ds_prop_1 FOREIGN KEY (id_prop) REFERENCES msgdiag_ds(id),
	CONSTRAINT pk_msgdiag_ds_prop PRIMARY KEY (id)
);

CREATE TABLE msgdiag_ds_prop_init_seq (id BIGINT);
INSERT INTO msgdiag_ds_prop_init_seq VALUES (NEXT VALUE FOR seq_msgdiag_ds_prop);



-- **** Datasources dove reperire le tracce salvate ****

CREATE SEQUENCE seq_tracce_ds AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE tracce_ds
(
	nome VARCHAR(255) NOT NULL,
	nome_jndi VARCHAR(255) NOT NULL,
	tipo_database VARCHAR(255) NOT NULL,
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	-- unique constraints
	CONSTRAINT unique_tracce_ds_1 UNIQUE (nome),
	-- fk/pk keys constraints
	CONSTRAINT pk_tracce_ds PRIMARY KEY (id)
);

CREATE TABLE tracce_ds_init_seq (id BIGINT);
INSERT INTO tracce_ds_init_seq VALUES (NEXT VALUE FOR seq_tracce_ds);



CREATE SEQUENCE seq_tracce_ds_prop AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE tracce_ds_prop
(
	id_prop BIGINT NOT NULL,
	nome VARCHAR(255) NOT NULL,
	valore VARCHAR(255) NOT NULL,
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	-- unique constraints
	CONSTRAINT unique_tracce_ds_prop_1 UNIQUE (id_prop,nome,valore),
	-- fk/pk keys constraints
	CONSTRAINT fk_tracce_ds_prop_1 FOREIGN KEY (id_prop) REFERENCES tracce_ds(id),
	CONSTRAINT pk_tracce_ds_prop PRIMARY KEY (id)
);

CREATE TABLE tracce_ds_prop_init_seq (id BIGINT);
INSERT INTO tracce_ds_prop_init_seq VALUES (NEXT VALUE FOR seq_tracce_ds_prop);



-- **** Stato dei servizi attivi sulla Porta di Dominio ****

CREATE SEQUENCE seq_servizi_pdd AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE servizi_pdd
(
	componente VARCHAR(255) NOT NULL,
	-- Stato dei servizi attivi sulla Porta di Dominio
	stato INT,
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	-- unique constraints
	CONSTRAINT unique_servizi_pdd_1 UNIQUE (componente),
	-- fk/pk keys constraints
	CONSTRAINT pk_servizi_pdd PRIMARY KEY (id)
);

-- index
CREATE UNIQUE INDEX index_servizi_pdd_1 ON servizi_pdd (componente);

ALTER TABLE servizi_pdd ALTER COLUMN stato SET DEFAULT 1;

CREATE TABLE servizi_pdd_init_seq (id BIGINT);
INSERT INTO servizi_pdd_init_seq VALUES (NEXT VALUE FOR seq_servizi_pdd);



CREATE SEQUENCE seq_servizi_pdd_filtri AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE servizi_pdd_filtri
(
	id_servizio_pdd BIGINT NOT NULL,
	tipo_filtro VARCHAR(255) NOT NULL,
	tipo_soggetto_fruitore VARCHAR(255),
	soggetto_fruitore VARCHAR(255),
	identificativo_porta_fruitore VARCHAR(255),
	tipo_soggetto_erogatore VARCHAR(255),
	soggetto_erogatore VARCHAR(255),
	identificativo_porta_erogatore VARCHAR(255),
	tipo_servizio VARCHAR(255),
	servizio VARCHAR(255),
	versione_servizio INT,
	azione VARCHAR(255),
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	-- check constraints
	CONSTRAINT chk_servizi_pdd_filtri_1 CHECK (tipo_filtro IN ('abilitazione','disabilitazione')),
	-- fk/pk keys constraints
	CONSTRAINT fk_servizi_pdd_filtri_1 FOREIGN KEY (id_servizio_pdd) REFERENCES servizi_pdd(id),
	CONSTRAINT pk_servizi_pdd_filtri PRIMARY KEY (id)
);

CREATE TABLE servizi_pdd_filtri_init_seq (id BIGINT);
INSERT INTO servizi_pdd_filtri_init_seq VALUES (NEXT VALUE FOR seq_servizi_pdd_filtri);



-- **** PddSystemProperties ****

CREATE SEQUENCE seq_pdd_sys_props AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE pdd_sys_props
(
	nome VARCHAR(255) NOT NULL,
	valore VARCHAR(255) NOT NULL,
	enc_value VARCHAR(65535),
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	-- unique constraints
	CONSTRAINT unique_pdd_sys_props_1 UNIQUE (nome,valore),
	-- fk/pk keys constraints
	CONSTRAINT pk_pdd_sys_props PRIMARY KEY (id)
);

-- index
CREATE UNIQUE INDEX index_pdd_sys_props_1 ON pdd_sys_props (nome,valore);
CREATE TABLE pdd_sys_props_init_seq (id BIGINT);
INSERT INTO pdd_sys_props_init_seq VALUES (NEXT VALUE FOR seq_pdd_sys_props);



-- **** Proprieta Generiche ****

CREATE SEQUENCE seq_generic_properties AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE generic_properties
(
	nome VARCHAR(255) NOT NULL,
	descrizione VARCHAR(4000),
	tipologia VARCHAR(255) NOT NULL,
	tipo VARCHAR(255) NOT NULL,
	utente_richiedente VARCHAR(255),
	data_creazione TIMESTAMP,
	utente_ultima_modifica VARCHAR(255),
	data_ultima_modifica TIMESTAMP,
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	-- unique constraints
	CONSTRAINT unique_generic_properties_1 UNIQUE (tipologia,nome),
	-- fk/pk keys constraints
	CONSTRAINT pk_generic_properties PRIMARY KEY (id)
);

-- index
CREATE UNIQUE INDEX index_generic_properties_1 ON generic_properties (tipologia,nome);
CREATE TABLE generic_properties_init_seq (id BIGINT);
INSERT INTO generic_properties_init_seq VALUES (NEXT VALUE FOR seq_generic_properties);



CREATE SEQUENCE seq_generic_property AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE generic_property
(
	id_props BIGINT NOT NULL,
	nome VARCHAR(255) NOT NULL,
	valore VARCHAR(4000) NOT NULL,
	enc_value VARCHAR(65535),
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	-- unique constraints
	CONSTRAINT unique_generic_property_1 UNIQUE (id_props,nome),
	-- fk/pk keys constraints
	CONSTRAINT fk_generic_property_1 FOREIGN KEY (id_props) REFERENCES generic_properties(id),
	CONSTRAINT pk_generic_property PRIMARY KEY (id)
);

-- index
CREATE INDEX index_generic_property_1 ON generic_property (id_props);
CREATE TABLE generic_property_init_seq (id BIGINT);
INSERT INTO generic_property_init_seq VALUES (NEXT VALUE FOR seq_generic_property);



-- **** Canali ****

CREATE SEQUENCE seq_canali_configurazione AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE canali_configurazione
(
	nome VARCHAR(255) NOT NULL,
	descrizione VARCHAR(255),
	canale_default INT NOT NULL,
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	-- unique constraints
	CONSTRAINT unique_canali_configurazione_1 UNIQUE (nome),
	-- fk/pk keys constraints
	CONSTRAINT pk_canali_configurazione PRIMARY KEY (id)
);

CREATE TABLE canali_configurazione_init_seq (id BIGINT);
INSERT INTO canali_configurazione_init_seq VALUES (NEXT VALUE FOR seq_canali_configurazione);



CREATE SEQUENCE seq_canali_nodi AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE canali_nodi
(
	nome VARCHAR(255) NOT NULL,
	descrizione VARCHAR(255),
	canali VARCHAR(4000) NOT NULL,
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	-- unique constraints
	CONSTRAINT unique_canali_nodi_1 UNIQUE (nome),
	-- fk/pk keys constraints
	CONSTRAINT pk_canali_nodi PRIMARY KEY (id)
);

CREATE TABLE canali_nodi_init_seq (id BIGINT);
INSERT INTO canali_nodi_init_seq VALUES (NEXT VALUE FOR seq_canali_nodi);



-- **** Regitro Plugins ****

CREATE SEQUENCE seq_registro_plugins AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE registro_plugins
(
	nome VARCHAR(255) NOT NULL,
	posizione INT NOT NULL,
	stato VARCHAR(255),
	descrizione VARCHAR(255),
	data TIMESTAMP NOT NULL,
	compatibilita VARCHAR(65535),
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	-- unique constraints
	CONSTRAINT unique_registro_plugins_1 UNIQUE (nome),
	-- fk/pk keys constraints
	CONSTRAINT pk_registro_plugins PRIMARY KEY (id)
);

-- index
CREATE UNIQUE INDEX index_registro_plugins_1 ON registro_plugins (nome);

ALTER TABLE registro_plugins ALTER COLUMN data SET DEFAULT CURRENT_TIMESTAMP;

CREATE TABLE registro_plugins_init_seq (id BIGINT);
INSERT INTO registro_plugins_init_seq VALUES (NEXT VALUE FOR seq_registro_plugins);



CREATE SEQUENCE seq_registro_plug_jar AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE registro_plug_jar
(
	id_plugin BIGINT NOT NULL,
	nome VARCHAR(255) NOT NULL,
	sorgente VARCHAR(255) NOT NULL,
	-- In hsql 2.x usare il tipo BLOB al posto di VARBINARY
	contenuto VARBINARY(1073741823),
	url VARCHAR(4000),
	dir VARCHAR(4000),
	data TIMESTAMP NOT NULL,
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	-- unique constraints
	CONSTRAINT unique_registro_plug_jar_1 UNIQUE (id_plugin,nome),
	-- fk/pk keys constraints
	CONSTRAINT fk_registro_plug_jar_1 FOREIGN KEY (id_plugin) REFERENCES registro_plugins(id),
	CONSTRAINT pk_registro_plug_jar PRIMARY KEY (id)
);

-- index
CREATE INDEX index_registro_plug_jar_1 ON registro_plug_jar (id_plugin);

ALTER TABLE registro_plug_jar ALTER COLUMN data SET DEFAULT CURRENT_TIMESTAMP;

CREATE TABLE registro_plug_jar_init_seq (id BIGINT);
INSERT INTO registro_plug_jar_init_seq VALUES (NEXT VALUE FOR seq_registro_plug_jar);



-- **** Handlers ****

CREATE SEQUENCE seq_config_handlers AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE config_handlers
(
	tipologia VARCHAR(255) NOT NULL,
	tipo VARCHAR(255) NOT NULL,
	posizione INT NOT NULL,
	stato VARCHAR(255),
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	-- unique constraints
	CONSTRAINT unique_config_handlers_1 UNIQUE (tipologia,tipo),
	-- fk/pk keys constraints
	CONSTRAINT pk_config_handlers PRIMARY KEY (id)
);

-- index
CREATE UNIQUE INDEX index_config_handlers_1 ON config_handlers (tipologia,tipo);
CREATE TABLE config_handlers_init_seq (id BIGINT);
INSERT INTO config_handlers_init_seq VALUES (NEXT VALUE FOR seq_config_handlers);



-- **** Nodi Runtime ****

CREATE SEQUENCE seq_nodi_runtime AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE nodi_runtime
(
	hostname VARCHAR(3000) NOT NULL,
	gruppo VARCHAR(3000) NOT NULL,
	data_registrazione TIMESTAMP NOT NULL,
	data_refresh TIMESTAMP NOT NULL,
	id_numerico INT NOT NULL,
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	-- unique constraints
	CONSTRAINT unique_nodi_runtime_1 UNIQUE (hostname),
	CONSTRAINT unique_nodi_runtime_2 UNIQUE (gruppo,id_numerico),
	-- fk/pk keys constraints
	CONSTRAINT pk_nodi_runtime PRIMARY KEY (id)
);

-- index
CREATE UNIQUE INDEX index_nodi_runtime_1 ON nodi_runtime (hostname);
CREATE UNIQUE INDEX index_nodi_runtime_2 ON nodi_runtime (gruppo,id_numerico);
CREATE TABLE nodi_runtime_init_seq (id BIGINT);
INSERT INTO nodi_runtime_init_seq VALUES (NEXT VALUE FOR seq_nodi_runtime);



CREATE SEQUENCE seq_nodi_runtime_operations AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE nodi_runtime_operations
(
	descrizione VARCHAR(4000) NOT NULL,
	operazione VARCHAR(4000) NOT NULL,
	data_registrazione TIMESTAMP NOT NULL,
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	-- fk/pk keys constraints
	CONSTRAINT pk_nodi_runtime_operations PRIMARY KEY (id)
);

-- index
CREATE INDEX INDEX_NODI_OP ON nodi_runtime_operations (data_registrazione);
CREATE TABLE nodi_runtime_operations_init_seq (id BIGINT);
INSERT INTO nodi_runtime_operations_init_seq VALUES (NEXT VALUE FOR seq_nodi_runtime_operations);



-- **** Keystore ****

CREATE SEQUENCE seq_remote_store AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE remote_store
(
	nome VARCHAR(255) NOT NULL,
	data_aggiornamento TIMESTAMP NOT NULL,
	last_event VARCHAR(4000),
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	-- unique constraints
	CONSTRAINT unique_remote_store_1 UNIQUE (nome),
	-- fk/pk keys constraints
	CONSTRAINT pk_remote_store PRIMARY KEY (id)
);

-- index
CREATE UNIQUE INDEX index_remote_store_1 ON remote_store (nome);
CREATE TABLE remote_store_init_seq (id BIGINT);
INSERT INTO remote_store_init_seq VALUES (NEXT VALUE FOR seq_remote_store);



CREATE SEQUENCE seq_remote_store_key AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE remote_store_key
(
	id_remote_store BIGINT NOT NULL,
	kid VARCHAR(255) NOT NULL,
	-- In hsql 2.x usare il tipo BLOB al posto di VARBINARY
	content_key VARBINARY(1073741823) NOT NULL,
	data_registrazione TIMESTAMP NOT NULL,
	data_aggiornamento TIMESTAMP NOT NULL,
	client_id VARCHAR(255),
	client_details VARCHAR(65535),
	organization_details VARCHAR(65535),
	client_data_aggiornamento TIMESTAMP,
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	-- unique constraints
	CONSTRAINT unique_remote_store_key_1 UNIQUE (id_remote_store,kid),
	-- fk/pk keys constraints
	CONSTRAINT fk_remote_store_key_1 FOREIGN KEY (id_remote_store) REFERENCES remote_store(id),
	CONSTRAINT pk_remote_store_key PRIMARY KEY (id)
);

-- index
CREATE UNIQUE INDEX index_remote_store_key_1 ON remote_store_key (id_remote_store,kid);
CREATE INDEX REMOTE_STORE_UPDATE ON remote_store_key (data_aggiornamento);
CREATE INDEX REMOTE_STORE_CREATE ON remote_store_key (data_registrazione);
CREATE TABLE remote_store_key_init_seq (id BIGINT);
INSERT INTO remote_store_key_init_seq VALUES (NEXT VALUE FOR seq_remote_store_key);



-- **** Digests Servizi Params ****

CREATE SEQUENCE seq_servizi_digest_params AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE servizi_digest_params
(
	id_servizio BIGINT NOT NULL,
	serial_number BIGINT,
	data_registrazione TIMESTAMP NOT NULL,
	seed VARCHAR(4000) NOT NULL,
	algorithm VARCHAR(32) NOT NULL,
	lifetime INT NOT NULL,
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	-- fk/pk keys constraints
	CONSTRAINT pk_servizi_digest_params PRIMARY KEY (id)
);

-- index
CREATE INDEX INDEX_REF_SERVIZI ON servizi_digest_params (id_servizio);
CREATE TABLE servizi_digest_params_init_seq (id BIGINT);
INSERT INTO servizi_digest_params_init_seq VALUES (NEXT VALUE FOR seq_servizi_digest_params);




-- govway
		  		
-- CONTROLLO TRAFFICO

CREATE SEQUENCE seq_ct_config AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE ct_config
(
	-- Numero Massimo Richieste Simultanee
	max_threads_enabled BOOLEAN NOT NULL,
	max_threads_warning_only BOOLEAN NOT NULL,
	max_threads BIGINT NOT NULL,
	max_threads_tipo_errore VARCHAR(255) NOT NULL,
	max_threads_includi_errore BOOLEAN NOT NULL,
	-- Controllo della Congestione
	cc_enabled BOOLEAN NOT NULL,
	cc_threshold INT,
	-- Tempi di Risposta Fruizione
	pd_connection_timeout INT,
	pd_read_timeout INT,
	pd_avg_time INT,
	-- Tempi di Risposta Erogazione
	pa_connection_timeout INT,
	pa_read_timeout INT,
	pa_avg_time INT,
	-- Rate Limiting
	rt_tipo_errore VARCHAR(255) NOT NULL,
	rt_includi_errore BOOLEAN NOT NULL,
	-- Cache
	cache BOOLEAN NOT NULL,
	cache_size BIGINT,
	cache_algorithm VARCHAR(255),
	cache_idle_time BIGINT,
	cache_life_time BIGINT,
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	-- check constraints
	CONSTRAINT chk_ct_config_1 CHECK (cache_algorithm IN ('LRU','MRU')),
	-- fk/pk keys constraints
	CONSTRAINT pk_ct_config PRIMARY KEY (id)
);


ALTER TABLE ct_config ALTER COLUMN max_threads_enabled SET DEFAULT true;
ALTER TABLE ct_config ALTER COLUMN max_threads_warning_only SET DEFAULT false;
ALTER TABLE ct_config ALTER COLUMN max_threads_tipo_errore SET DEFAULT 'fault';
ALTER TABLE ct_config ALTER COLUMN max_threads_includi_errore SET DEFAULT true;
ALTER TABLE ct_config ALTER COLUMN cc_enabled SET DEFAULT false;
ALTER TABLE ct_config ALTER COLUMN rt_tipo_errore SET DEFAULT 'fault';
ALTER TABLE ct_config ALTER COLUMN rt_includi_errore SET DEFAULT true;
ALTER TABLE ct_config ALTER COLUMN cache SET DEFAULT true;

CREATE TABLE ct_config_init_seq (id BIGINT);
INSERT INTO ct_config_init_seq VALUES (NEXT VALUE FOR seq_ct_config);



CREATE SEQUENCE seq_ct_rt_props AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE ct_rt_props
(
	rt_prop_name VARCHAR(255) NOT NULL,
	rt_prop_value VARCHAR(255) NOT NULL,
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	-- unique constraints
	CONSTRAINT uniq_rt_prop_policy_1 UNIQUE (rt_prop_name),
	-- fk/pk keys constraints
	CONSTRAINT pk_ct_rt_props PRIMARY KEY (id)
);

-- index
CREATE UNIQUE INDEX idx_rt_prop_policy_1 ON ct_rt_props (rt_prop_name);
CREATE INDEX idx_rt_prop_policy_2 ON ct_rt_props (rt_prop_value);
CREATE TABLE ct_rt_props_init_seq (id BIGINT);
INSERT INTO ct_rt_props_init_seq VALUES (NEXT VALUE FOR seq_ct_rt_props);



CREATE SEQUENCE seq_ct_config_policy AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE ct_config_policy
(
	-- Dati Generali
	policy_id VARCHAR(255) NOT NULL,
	policy_built_in BOOLEAN NOT NULL,
	rt_descrizione VARCHAR(65535) NOT NULL,
	rt_risorsa VARCHAR(255) NOT NULL,
	-- Valori di Soglia
	rt_simultanee BOOLEAN NOT NULL,
	rt_valore BIGINT,
	rt_valore2 BIGINT,
	rt_bytes_type VARCHAR(255),
	rt_latency_type VARCHAR(255),
	rt_modalita_controllo VARCHAR(255),
	rt_interval_type_real VARCHAR(255),
	rt_interval_type_stat VARCHAR(255),
	rt_interval INT,
	rt_finestra VARCHAR(255),
	-- Applicabilità
	rt_applicabilita VARCHAR(255) NOT NULL,
	-- Applicabilità con Congestione in Corso
	rt_applicabilita_con_cc BOOLEAN NOT NULL,
	-- Applicabilità con Degrado Prestazionale
	rt_applicabilita_degrado BOOLEAN NOT NULL,
	degrato_modalita_controllo VARCHAR(255),
	degrado_avg_interval_type_real VARCHAR(255),
	degrado_avg_interval_type_stat VARCHAR(255),
	degrado_avg_interval INT,
	degrado_avg_finestra VARCHAR(255),
	degrado_avg_latency_type VARCHAR(255),
	-- Applicabilità con Stato Allarme
	rt_applicabilita_allarme BOOLEAN NOT NULL,
	allarme_nome VARCHAR(255),
	allarme_stato INT,
	allarme_not_stato BOOLEAN NOT NULL,
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	-- check constraints
	CONSTRAINT chk_cong_gen_policy_1 CHECK (rt_bytes_type IN ('complessiva','interna','esterna')),
	CONSTRAINT chk_cong_gen_policy_2 CHECK (rt_latency_type IN ('servizio','porta','totale')),
	CONSTRAINT chk_cong_gen_policy_3 CHECK (rt_modalita_controllo IN ('realtime','statistic')),
	CONSTRAINT chk_cong_gen_policy_4 CHECK (rt_interval_type_real IN ('secondi','minuti','orario','giornaliero')),
	CONSTRAINT chk_cong_gen_policy_5 CHECK (rt_interval_type_stat IN ('mensile','settimanale','giornaliero','orario')),
	CONSTRAINT chk_cong_gen_policy_6 CHECK (rt_finestra IN ('corrente','precedente','scorrevole')),
	CONSTRAINT chk_cong_gen_policy_7 CHECK (rt_applicabilita IN ('sempre','condizionale')),
	CONSTRAINT chk_cong_gen_policy_8 CHECK (degrato_modalita_controllo IN ('realtime','statistic')),
	CONSTRAINT chk_cong_gen_policy_9 CHECK (degrado_avg_interval_type_real IN ('secondi','minuti','orario','giornaliero')),
	CONSTRAINT chk_cong_gen_policy_10 CHECK (degrado_avg_interval_type_stat IN ('mensile','settimanale','giornaliero','orario')),
	CONSTRAINT chk_cong_gen_policy_11 CHECK (degrado_avg_finestra IN ('corrente','precedente','scorrevole')),
	CONSTRAINT chk_cong_gen_policy_12 CHECK (degrado_avg_latency_type IN ('servizio','porta','totale')),
	-- unique constraints
	CONSTRAINT uniq_cong_gen_policy_1 UNIQUE (policy_id),
	-- fk/pk keys constraints
	CONSTRAINT pk_ct_config_policy PRIMARY KEY (id)
);

-- index
CREATE UNIQUE INDEX idx_cong_gen_policy_1 ON ct_config_policy (policy_id);

ALTER TABLE ct_config_policy ALTER COLUMN policy_built_in SET DEFAULT false;
ALTER TABLE ct_config_policy ALTER COLUMN rt_simultanee SET DEFAULT false;
ALTER TABLE ct_config_policy ALTER COLUMN rt_applicabilita SET DEFAULT 'sempre';
ALTER TABLE ct_config_policy ALTER COLUMN rt_applicabilita_con_cc SET DEFAULT false;
ALTER TABLE ct_config_policy ALTER COLUMN rt_applicabilita_degrado SET DEFAULT false;
ALTER TABLE ct_config_policy ALTER COLUMN rt_applicabilita_allarme SET DEFAULT false;
ALTER TABLE ct_config_policy ALTER COLUMN allarme_not_stato SET DEFAULT false;

CREATE TABLE ct_config_policy_init_seq (id BIGINT);
INSERT INTO ct_config_policy_init_seq VALUES (NEXT VALUE FOR seq_ct_config_policy);



CREATE SEQUENCE seq_ct_active_policy AS BIGINT START WITH 1 INCREMENT BY 1 ; -- (Scommentare in hsql 2.x) NO CYCLE;

CREATE TABLE ct_active_policy
(
	-- Dati Generali
	active_policy_id VARCHAR(275) NOT NULL,
	policy_alias VARCHAR(255),
	policy_update_time TIMESTAMP NOT NULL,
	policy_posizione INT NOT NULL,
	policy_continue BOOLEAN NOT NULL,
	policy_id VARCHAR(255) NOT NULL,
	policy_enabled BOOLEAN NOT NULL,
	policy_warning BOOLEAN NOT NULL,
	-- Valori di Soglia
	policy_redefined BOOLEAN NOT NULL,
	policy_valore BIGINT,
	policy_valore2 BIGINT,
	-- Filtro
	filtro_enabled BOOLEAN NOT NULL,
	filtro_protocollo VARCHAR(255),
	filtro_ruolo VARCHAR(255),
	filtro_porta VARCHAR(2000),
	filtro_tipo_fruitore VARCHAR(255),
	filtro_nome_fruitore VARCHAR(255),
	filtro_ruolo_fruitore VARCHAR(255),
	filtro_sa_fruitore VARCHAR(255),
	filtro_tipo_erogatore VARCHAR(255),
	filtro_nome_erogatore VARCHAR(255),
	filtro_ruolo_erogatore VARCHAR(255),
	filtro_sa_erogatore VARCHAR(255),
	filtro_tag VARCHAR(255),
	filtro_tipo_servizio VARCHAR(255),
	filtro_nome_servizio VARCHAR(255),
	filtro_versione_servizio INT,
	filtro_azione VARCHAR(65535),
	filtro_token_claims VARCHAR(65535),
	-- Filtro per Chiave Applicativa
	filtro_key_enabled BOOLEAN NOT NULL,
	filtro_key_type VARCHAR(255),
	filtro_key_name VARCHAR(65535),
	filtro_key_value VARCHAR(65535),
	-- Raggruppamento
	group_enabled BOOLEAN NOT NULL,
	group_ruolo BOOLEAN NOT NULL,
	group_protocollo BOOLEAN NOT NULL,
	group_fruitore BOOLEAN NOT NULL,
	group_sa_fruitore BOOLEAN NOT NULL,
	group_id_autenticato BOOLEAN NOT NULL,
	group_token VARCHAR(65535),
	group_erogatore BOOLEAN NOT NULL,
	group_sa_erogatore BOOLEAN NOT NULL,
	group_servizio BOOLEAN NOT NULL,
	group_azione BOOLEAN NOT NULL,
	-- Raggruppamento per Chiave Applicativa
	group_key_enabled BOOLEAN NOT NULL,
	group_key_type VARCHAR(255),
	group_key_name VARCHAR(65535),
	-- fk/pk columns
	id BIGINT GENERATED BY DEFAULT AS IDENTITY (START WITH 1),
	-- check constraints
	CONSTRAINT chk_cong_att_policy_1 CHECK (filtro_ruolo IN ('delegata','applicativa','entrambi')),
	-- unique constraints
	CONSTRAINT uniq_cong_att_policy_1 UNIQUE (active_policy_id),
	-- fk/pk keys constraints
	CONSTRAINT pk_ct_active_policy PRIMARY KEY (id)
);

-- index
CREATE UNIQUE INDEX idx_cong_att_policy_1 ON ct_active_policy (active_policy_id);
CREATE INDEX idx_cong_att_policy_2 ON ct_active_policy (filtro_ruolo,filtro_porta);

ALTER TABLE ct_active_policy ALTER COLUMN policy_continue SET DEFAULT false;
ALTER TABLE ct_active_policy ALTER COLUMN policy_warning SET DEFAULT false;
ALTER TABLE ct_active_policy ALTER COLUMN filtro_enabled SET DEFAULT false;
ALTER TABLE ct_active_policy ALTER COLUMN filtro_key_enabled SET DEFAULT false;
ALTER TABLE ct_active_policy ALTER COLUMN group_enabled SET DEFAULT false;
ALTER TABLE ct_active_policy ALTER COLUMN group_ruolo SET DEFAULT false;
ALTER TABLE ct_active_policy ALTER COLUMN group_protocollo SET DEFAULT false;
ALTER TABLE ct_active_policy ALTER COLUMN group_fruitore SET DEFAULT false;
ALTER TABLE ct_active_policy ALTER COLUMN group_sa_fruitore SET DEFAULT false;
ALTER TABLE ct_active_policy ALTER COLUMN group_id_autenticato SET DEFAULT false;
ALTER TABLE ct_active_policy ALTER COLUMN group_erogatore SET DEFAULT false;
ALTER TABLE ct_active_policy ALTER COLUMN group_sa_erogatore SET DEFAULT false;
ALTER TABLE ct_active_policy ALTER COLUMN group_servizio SET DEFAULT false;
ALTER TABLE ct_active_policy ALTER COLUMN group_azione SET DEFAULT false;
ALTER TABLE ct_active_policy ALTER COLUMN group_key_enabled SET DEFAULT false;

CREATE TABLE ct_active_policy_init_seq (id BIGINT);
INSERT INTO ct_active_policy_init_seq VALUES (NEXT VALUE FOR seq_ct_active_policy);




-- govway
		  		
-- Configurazione
INSERT INTO configurazione (cadenza_inoltro, validazione_stato, validazione_controllo, msg_diag_severita, msg_diag_severita_log4j, auth_integration_manager,validazione_profilo, mod_risposta, indirizzo_telematico, routing_enabled, validazione_manifest, gestione_manifest, tracciamento_buste, dump, dump_bin_pd, dump_bin_pa, statocache,dimensionecache,algoritmocache,lifecache, config_statocache, config_dimensionecache, config_algoritmocache, config_lifecache, auth_statocache, auth_dimensionecache, auth_algoritmocache, auth_lifecache, authn_statocache, authn_dimensionecache, authn_algoritmocache, authn_lifecache, token_statocache, token_dimensionecache, token_algoritmocache, token_lifecache, aa_statocache, aa_dimensionecache, aa_algoritmocache, aa_lifecache, cors_stato, cors_tipo, cors_all_allow_origins, cors_allow_headers, cors_allow_methods, response_cache_stato, response_cache_statocache, response_cache_dimensionecache, response_cache_algoritmocache, response_cache_lifecache) VALUES( '60',       'abilitato',    'rigido', 'infoIntegration', 'infoIntegration', 'principal,ssl,basic', 'disabilitato','reply','disabilitato','disabilitato', 'abilitato', 'disabilitato', 'abilitato', 'disabilitato', 'disabilitato', 'disabilitato', 'abilitato','10000','lru','7200','abilitato','10000','lru','7200','abilitato','5000','lru','7200','abilitato','5000','lru','7200', 'abilitato','5000','lru','600','abilitato','5000','lru','7200', 'abilitato','gateway', 'abilitato', 'Authorization,Content-Type,SOAPAction','GET,PUT,POST,DELETE,PATCH','disabilitato','abilitato','10000','lru','1800');

-- Configurazione Controllo Traffico
INSERT INTO ct_config (max_threads,pd_connection_timeout,pd_read_timeout,pd_avg_time,pa_connection_timeout,pa_read_timeout,pa_avg_time,cache_size,cache_algorithm,cache_life_time) VALUES (200,10000,150000,10000,10000,120000,10000,10000,'LRU',300);

-- Init Configurazione URL Invocazione
INSERT INTO config_url_invocazione (base_url) VALUES ('http://localhost:8080/govway/');

-- Rotta di default per routing
insert INTO routing (tiporotta,registrorotta,is_default) VALUES ('registro',0,1);

-- Registro locale
insert INTO registri (nome,location,tipo) VALUES ('RegistroDB','org.govway.datasource.console','db');

-- Porta di Dominio locale
INSERT INTO pdd (nome,tipo,superuser) VALUES ('GovWay','operativo','amministratore');


-- govway
			  		
-- users
INSERT INTO users (login, password, tipo_interfaccia, permessi, soggetti_all, servizi_all) VALUES ('operatore',	'$1$.bquKJDS$yZ4EYC3354HqEjmSRL/sR0','STANDARD','DR', 1, 1);


-- govway
			  		




