/*
 * Decompiled with CFR 0.152.
 */
package org.openspcoop2.utils.oauth2;

import java.io.Serializable;
import java.net.URLEncoder;
import java.nio.charset.StandardCharsets;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.text.MessageFormat;
import java.util.Base64;
import java.util.Map;
import java.util.Properties;
import org.apache.commons.lang3.StringUtils;
import org.openspcoop2.utils.UtilsException;
import org.openspcoop2.utils.certificate.JWKSet;
import org.openspcoop2.utils.date.DateManager;
import org.openspcoop2.utils.json.JSONUtils;
import org.openspcoop2.utils.oauth2.OAuth2Token;
import org.openspcoop2.utils.oauth2.Oauth2BaseResponse;
import org.openspcoop2.utils.oauth2.Oauth2ClaimValidator;
import org.openspcoop2.utils.oauth2.Oauth2UserInfo;
import org.openspcoop2.utils.random.RandomGenerator;
import org.openspcoop2.utils.security.JOSESerialization;
import org.openspcoop2.utils.security.JWTOptions;
import org.openspcoop2.utils.security.JWTParser;
import org.openspcoop2.utils.security.JsonVerifySignature;
import org.openspcoop2.utils.transport.http.HttpRequest;
import org.openspcoop2.utils.transport.http.HttpRequestMethod;
import org.openspcoop2.utils.transport.http.HttpResponse;
import org.openspcoop2.utils.transport.http.HttpUtilities;
import org.slf4j.Logger;

public class OAuth2Utilities {
    private OAuth2Utilities() {
    }

    public static String addFirstParameter(String name, String value) {
        return "?" + OAuth2Utilities.getParameter(name, value);
    }

    public static String addParameter(String name, String value) {
        return "&" + OAuth2Utilities.getParameter(name, value);
    }

    public static String getParameter(String name, String value) {
        return name + "=" + URLEncoder.encode(value, StandardCharsets.UTF_8);
    }

    public static String generateCodeVerifier() {
        RandomGenerator randomGenerator = new RandomGenerator(true);
        byte[] code = new byte[48];
        randomGenerator.nextRandomBytes(code);
        return Base64.getUrlEncoder().withoutPadding().encodeToString(code);
    }

    public static String generateCodeChallenge(String codeVerifier) throws UtilsException {
        return OAuth2Utilities.generateCodeChallenge(codeVerifier, "S256");
    }

    public static String generateCodeChallenge(String codeVerifier, String method) throws UtilsException {
        if ("plain".equalsIgnoreCase(method)) {
            return codeVerifier;
        }
        if ("S256".equalsIgnoreCase(method)) {
            try {
                MessageDigest digest = MessageDigest.getInstance("SHA-256");
                byte[] hash = digest.digest(codeVerifier.getBytes(StandardCharsets.US_ASCII));
                return Base64.getUrlEncoder().withoutPadding().encodeToString(hash);
            }
            catch (NoSuchAlgorithmException e) {
                throw new UtilsException("SHA-256 algorithm not available for PKCE code challenge generation", e);
            }
        }
        throw new UtilsException("Unsupported PKCE method: " + method + ". Supported methods are: S256, plain");
    }

    public static boolean isPkceEnabled(Properties loginProperties) {
        String pkceEnabled = OAuth2Utilities.getProperty(loginProperties, "oauth2.pkce.enabled");
        return pkceEnabled != null && Boolean.parseBoolean(pkceEnabled);
    }

    public static String getPkceMethod(Properties loginProperties) {
        String method = OAuth2Utilities.getProperty(loginProperties, "oauth2.pkce.method");
        if (method == null || method.isEmpty()) {
            return "S256";
        }
        if ("plain".equalsIgnoreCase(method)) {
            return "plain";
        }
        return "S256";
    }

    public static String getURLLoginOAuth2(Properties loginProperties, String state) throws UtilsException {
        return OAuth2Utilities.getURLLoginOAuth2(loginProperties, state, null);
    }

    public static String getURLLoginOAuth2(Properties loginProperties, String state, String codeChallenge) throws UtilsException {
        return OAuth2Utilities.getURLLoginOAuth2(loginProperties, state, codeChallenge, null);
    }

    public static String getURLLoginOAuth2(Properties loginProperties, String state, String codeChallenge, String codeChallengeMethod) throws UtilsException {
        String authorizationEndpoint = OAuth2Utilities.checkAndReturnValue(loginProperties, "oauth2.authorization.endpoint");
        String clientId = OAuth2Utilities.checkAndReturnValue(loginProperties, "oauth2.clientId");
        String callbackUri = OAuth2Utilities.checkAndReturnValue(loginProperties, "oauth2.redirectUri");
        String scope = OAuth2Utilities.checkAndReturnValue(loginProperties, "oauth2.scope");
        String url = authorizationEndpoint + OAuth2Utilities.addFirstParameter("response_type", "code") + OAuth2Utilities.addParameter("redirect_uri", callbackUri) + OAuth2Utilities.addParameter("client_id", clientId) + OAuth2Utilities.addParameter("scope", scope) + OAuth2Utilities.addParameter("state", state);
        if (codeChallenge != null && !codeChallenge.isEmpty()) {
            url = url + OAuth2Utilities.addParameter("code_challenge", codeChallenge);
            String method = codeChallengeMethod != null ? codeChallengeMethod : "S256";
            url = url + OAuth2Utilities.addParameter("code_challenge_method", method);
        }
        return url;
    }

    private static String checkAndReturnValue(Properties loginProperties, String pName) throws UtilsException {
        String value = OAuth2Utilities.getProperty(loginProperties, pName);
        if (value == null || StringUtils.isEmpty((CharSequence)value)) {
            throw new UtilsException("Undefined property '" + pName + "'");
        }
        return value;
    }

    private static String getProperty(Properties loginProperties, String pName) {
        String value = loginProperties.getProperty(pName);
        return value != null ? value.trim() : null;
    }

    private static void injectHttpConfig(HttpRequest req, Properties loginProperties) {
        String trustAllCerts;
        String hostnameVerifier = OAuth2Utilities.getProperty(loginProperties, "oauth2.https.hostnameVerifier");
        if (hostnameVerifier != null) {
            req.setHostnameVerifier(Boolean.valueOf(hostnameVerifier));
        }
        if ((trustAllCerts = OAuth2Utilities.getProperty(loginProperties, "oauth2.https.trustAllCerts")) != null) {
            req.setTrustAllCerts(Boolean.valueOf(trustAllCerts));
        }
        req.setTrustStorePath(OAuth2Utilities.getProperty(loginProperties, "oauth2.https.trustStore"));
        req.setTrustStorePassword(OAuth2Utilities.getProperty(loginProperties, "oauth2.https.trustStore.password"));
        req.setTrustStoreType(OAuth2Utilities.getProperty(loginProperties, "oauth2.https.trustStore.type"));
        req.setCrlPath(OAuth2Utilities.getProperty(loginProperties, "oauth2.https.trustStore.crl"));
        req.setKeyStorePath(OAuth2Utilities.getProperty(loginProperties, "oauth2.https.keyStore"));
        req.setKeyStorePassword(OAuth2Utilities.getProperty(loginProperties, "oauth2.https.keyStore.password"));
        req.setKeyStoreType(OAuth2Utilities.getProperty(loginProperties, "oauth2.https.keyStore.type"));
        req.setKeyAlias(OAuth2Utilities.getProperty(loginProperties, "oauth2.https.key.alias"));
        req.setKeyPassword(OAuth2Utilities.getProperty(loginProperties, "oauth2.https.key.password"));
    }

    public static OAuth2Token getToken(Logger log, Properties loginProperties, String code) {
        return OAuth2Utilities.getToken(log, loginProperties, code, null);
    }

    public static OAuth2Token getToken(Logger log, Properties loginProperties, String code, String codeVerifier) {
        String tokenEndpoint = loginProperties.getProperty("oauth2.token.endpoint");
        String clientId = loginProperties.getProperty("oauth2.clientId");
        String callbackUri = loginProperties.getProperty("oauth2.redirectUri");
        String requestTokenBody = OAuth2Utilities.getParameter("grant_type", "authorization_code") + OAuth2Utilities.addParameter("code", code) + OAuth2Utilities.addParameter("redirect_uri", callbackUri) + OAuth2Utilities.addParameter("client_id", clientId);
        if (codeVerifier != null && !codeVerifier.isEmpty()) {
            requestTokenBody = requestTokenBody + OAuth2Utilities.addParameter("code_verifier", codeVerifier);
        }
        HttpRequest httpRequest = new HttpRequest();
        httpRequest.setMethod(HttpRequestMethod.POST);
        httpRequest.setUrl(tokenEndpoint);
        httpRequest.setContent(requestTokenBody.getBytes());
        httpRequest.setContentType("application/x-www-form-urlencoded");
        httpRequest.setFollowRedirects(false);
        httpRequest.setReadTimeout(OAuth2Utilities.getReadTimeout(log, loginProperties));
        httpRequest.setConnectTimeout(OAuth2Utilities.getConnectionTimeout(log, loginProperties));
        httpRequest.setDisconnect(false);
        httpRequest.addHeader("Accept", "application/json");
        OAuth2Utilities.injectHttpConfig(httpRequest, loginProperties);
        OAuth2Token response = new OAuth2Token();
        try {
            HttpResponse httpResponse = HttpUtilities.httpInvoke(httpRequest);
            response.setReturnCode(httpResponse.getResultHTTPOperation());
            String responseBody = new String(httpResponse.getContent());
            Map<String, Serializable> map = JSONUtils.getInstance().convertToMap(log, responseBody, responseBody);
            response.setMap(map);
            response.setRaw(responseBody);
            if (httpResponse.getResultHTTPOperation() != 200) {
                String tokenError = (String)((Object)map.get("error"));
                String tokenErrorDescription = (String)((Object)map.get("error_description"));
                OAuth2Utilities.logError(log, MessageFormat.format("Errore durante l''acquisizione del token: {0}, {1}", tokenError, tokenErrorDescription));
                response.setError(tokenError);
                response.setDescription(tokenErrorDescription);
            }
            String accessToken = (String)((Object)map.get("access_token"));
            JWTParser jwtParser = new JWTParser(accessToken);
            String kid = jwtParser.getHeaderClaim("kid");
            String alg = jwtParser.getHeaderClaim("alg");
            response.setAccessToken(accessToken);
            response.setKid(kid);
            response.setAlg(alg);
            String expireInS = (String)((Object)map.get("expires_in"));
            response.setExpiresIn(Long.parseLong(expireInS));
            response.setRefreshToken((String)((Object)map.get("refresh_token")));
            response.setScope((String)((Object)map.get("scope")));
            response.setTokenType((String)((Object)map.get("token_type")));
            response.setIdToken((String)((Object)map.get("id_token")));
            if (response.getExpiresIn() != null) {
                response.setExpiresAt(DateManager.getTimeMillis() + response.getExpiresIn() * 1000L);
            }
        }
        catch (UtilsException e) {
            OAuth2Utilities.logError(log, "Errore durante l'acquisizione del token: " + e.getMessage(), e);
            response.setReturnCode(500);
            response.setError("Errore durante l'acquisizione del token");
            response.setDescription(e.getMessage());
        }
        return response;
    }

    public static Oauth2BaseResponse getCertificati(Logger log, Properties loginProperties) {
        HttpRequest jwksHttpRequest = new HttpRequest();
        String jwksEndpoint = loginProperties.getProperty("oauth2.jwks.endpoint");
        jwksHttpRequest.setMethod(HttpRequestMethod.GET);
        jwksHttpRequest.setUrl(jwksEndpoint);
        jwksHttpRequest.setFollowRedirects(false);
        jwksHttpRequest.setReadTimeout(OAuth2Utilities.getReadTimeout(log, loginProperties));
        jwksHttpRequest.setConnectTimeout(OAuth2Utilities.getConnectionTimeout(log, loginProperties));
        jwksHttpRequest.setDisconnect(false);
        OAuth2Utilities.injectHttpConfig(jwksHttpRequest, loginProperties);
        Oauth2BaseResponse response = new Oauth2BaseResponse();
        try {
            HttpResponse jwksHttpResponse = HttpUtilities.httpInvoke(jwksHttpRequest);
            String jwksResponseBody = new String(jwksHttpResponse.getContent());
            Map<String, Serializable> jwksMap = JSONUtils.getInstance().convertToMap(log, jwksResponseBody, jwksResponseBody);
            response.setReturnCode(jwksHttpResponse.getResultHTTPOperation());
            response.setMap(jwksMap);
            response.setRaw(jwksResponseBody);
            if (jwksHttpResponse.getResultHTTPOperation() != 200) {
                String tokenError = (String)((Object)jwksMap.get("error"));
                String tokenErrorDescription = (String)((Object)jwksMap.get("error_description"));
                OAuth2Utilities.logError(log, "Errore durante la lettura dei certificati: " + tokenError + ", " + tokenErrorDescription);
                response.setError(tokenError);
                response.setDescription(tokenErrorDescription);
            }
        }
        catch (UtilsException e) {
            OAuth2Utilities.logError(log, "Errore durante la lettura dei certificati: " + e.getMessage(), e);
            response.setReturnCode(500);
            response.setError("Errore durante la lettura dei certificati");
            response.setDescription(e.getMessage());
        }
        return response;
    }

    public static Oauth2UserInfo getUserInfo(Logger log, Properties loginProperties, OAuth2Token oAuth2Token) {
        HttpRequest userInfoHttpRequest = new HttpRequest();
        String userInfoEndpoint = loginProperties.getProperty("oauth2.userInfo.endpoint");
        userInfoHttpRequest.setMethod(HttpRequestMethod.GET);
        userInfoHttpRequest.setUrl(userInfoEndpoint);
        userInfoHttpRequest.setFollowRedirects(false);
        userInfoHttpRequest.setReadTimeout(OAuth2Utilities.getReadTimeout(log, loginProperties));
        userInfoHttpRequest.setConnectTimeout(OAuth2Utilities.getConnectionTimeout(log, loginProperties));
        userInfoHttpRequest.setDisconnect(false);
        userInfoHttpRequest.addHeader("Accept", "application/json");
        userInfoHttpRequest.addHeader("Authorization", "Bearer " + oAuth2Token.getAccessToken());
        OAuth2Utilities.injectHttpConfig(userInfoHttpRequest, loginProperties);
        Oauth2UserInfo response = new Oauth2UserInfo();
        try {
            HttpResponse userInfoHttpResponse = HttpUtilities.httpInvoke(userInfoHttpRequest);
            String userInfoResponseBody = new String(userInfoHttpResponse.getContent());
            Map<String, Serializable> userInfoMap = JSONUtils.getInstance().convertToMap(log, userInfoResponseBody, userInfoResponseBody);
            response.setReturnCode(userInfoHttpResponse.getResultHTTPOperation());
            response.setMap(userInfoMap);
            response.setRaw(userInfoResponseBody);
            if (userInfoHttpResponse.getResultHTTPOperation() != 200) {
                String tokenError = (String)((Object)userInfoMap.get("error"));
                String tokenErrorDescription = (String)((Object)userInfoMap.get("error_description"));
                OAuth2Utilities.logError(log, "Errore durante la lettura delle informazioni utente: " + tokenError + ", " + tokenErrorDescription);
                response.setError(tokenError);
                response.setDescription(tokenErrorDescription);
            }
        }
        catch (UtilsException e) {
            OAuth2Utilities.logError(log, "Errore durante la lettura delle informazioni utente: " + e.getMessage(), e);
            response.setReturnCode(500);
            response.setError("Errore durante la lettura delle informazioni utente");
            response.setDescription(e.getMessage());
        }
        return response;
    }

    public static boolean isValidToken(Logger log, Oauth2BaseResponse jwksResponse, OAuth2Token oAuth2Token) {
        String accessToken = oAuth2Token.getAccessToken();
        String kid = oAuth2Token.getKid();
        String algoritm = oAuth2Token.getAlg();
        JWTOptions optionsVerify = new JWTOptions(JOSESerialization.COMPACT);
        try {
            JWKSet jwkSet = new JWKSet(jwksResponse.getRaw());
            JsonVerifySignature jsonVerify = new JsonVerifySignature(jwkSet.getJsonWebKeys(), false, kid, algoritm, optionsVerify);
            return jsonVerify.verify(accessToken);
        }
        catch (UtilsException e) {
            OAuth2Utilities.logError(log, "Errore durante la verifica del token: " + e.getMessage(), e);
            return false;
        }
    }

    public static boolean isValidToken(Logger log, Properties loginProperties, Oauth2BaseResponse jwksResponse, OAuth2Token oAuth2Token) {
        boolean signatureValid = OAuth2Utilities.isValidToken(log, jwksResponse, oAuth2Token);
        if (!signatureValid) {
            OAuth2Utilities.logError(log, "Validazione token fallita: firma JWT non valida");
            return false;
        }
        Oauth2ClaimValidator claimValidator = new Oauth2ClaimValidator(log, loginProperties);
        Oauth2ClaimValidator.ValidationResult claimResult = claimValidator.validate(oAuth2Token.getAccessToken());
        if (!claimResult.isValid()) {
            OAuth2Utilities.logError(log, "Validazione token fallita: claim non validi - " + claimResult.getErrorsAsString());
            if (log.isDebugEnabled()) {
                String debug = "Access token: " + oAuth2Token.getAccessToken();
                log.debug(debug);
            }
            return false;
        }
        if (log != null) {
            log.debug("Token validato con successo (firma + claim)");
        }
        return true;
    }

    public static OAuth2Token refreshToken(Logger log, Properties loginProperties, String refreshToken) {
        String tokenEndpoint = loginProperties.getProperty("oauth2.token.endpoint");
        String clientId = loginProperties.getProperty("oauth2.clientId");
        String refreshTokenBody = OAuth2Utilities.getParameter("grant_type", "refresh_token") + OAuth2Utilities.addParameter("refresh_token", refreshToken) + OAuth2Utilities.addParameter("client_id", clientId);
        HttpRequest httpRequest = new HttpRequest();
        httpRequest.setMethod(HttpRequestMethod.POST);
        httpRequest.setUrl(tokenEndpoint);
        httpRequest.setContent(refreshTokenBody.getBytes());
        httpRequest.setContentType("application/x-www-form-urlencoded");
        httpRequest.setFollowRedirects(false);
        httpRequest.setReadTimeout(OAuth2Utilities.getReadTimeout(log, loginProperties));
        httpRequest.setConnectTimeout(OAuth2Utilities.getConnectionTimeout(log, loginProperties));
        httpRequest.setDisconnect(false);
        httpRequest.addHeader("Accept", "application/json");
        OAuth2Utilities.injectHttpConfig(httpRequest, loginProperties);
        OAuth2Token response = new OAuth2Token();
        try {
            HttpResponse httpResponse = HttpUtilities.httpInvoke(httpRequest);
            response.setReturnCode(httpResponse.getResultHTTPOperation());
            String responseBody = new String(httpResponse.getContent());
            Map<String, Serializable> map = JSONUtils.getInstance().convertToMap(log, responseBody, responseBody);
            response.setMap(map);
            response.setRaw(responseBody);
            if (httpResponse.getResultHTTPOperation() != 200) {
                String tokenError = (String)((Object)map.get("error"));
                String tokenErrorDescription = (String)((Object)map.get("error_description"));
                OAuth2Utilities.logError(log, "Errore durante l'acquisizione del token: " + tokenError + ", " + tokenErrorDescription);
                response.setError(tokenError);
                response.setDescription(tokenErrorDescription);
            }
            String accessToken = (String)((Object)map.get("access_token"));
            JWTParser jwtParser = new JWTParser(accessToken);
            String kid = jwtParser.getHeaderClaim("kid");
            String alg = jwtParser.getHeaderClaim("alg");
            response.setAccessToken(accessToken);
            response.setKid(kid);
            response.setAlg(alg);
            String expireInS = (String)((Object)map.get("expires_in"));
            response.setExpiresIn(Long.parseLong(expireInS));
            response.setRefreshToken((String)((Object)map.get("refresh_token")));
            response.setScope((String)((Object)map.get("scope")));
            response.setTokenType((String)((Object)map.get("token_type")));
            response.setIdToken((String)((Object)map.get("id_token")));
            if (response.getExpiresIn() != null) {
                response.setExpiresAt(DateManager.getTimeMillis() + response.getExpiresIn() * 1000L);
            }
        }
        catch (UtilsException e) {
            OAuth2Utilities.logError(log, "Errore durante il refresh del token: " + e.getMessage(), e);
            response.setReturnCode(500);
            response.setError("Errore durante il refresh del token");
            response.setDescription(e.getMessage());
        }
        return response;
    }

    public static void logError(Logger log, String msg) {
        OAuth2Utilities.logError(log, msg, null);
    }

    public static void logError(Logger log, String msg, Throwable e) {
        if (e != null) {
            log.error(msg, e);
        } else {
            log.error(msg);
        }
    }

    private static int getReadTimeout(Logger log, Properties loginProperties) {
        String readTimeout = loginProperties.getProperty("oauth2.readTimeout");
        if (readTimeout != null && !readTimeout.isEmpty()) {
            try {
                return Integer.parseInt(readTimeout);
            }
            catch (NumberFormatException e) {
                OAuth2Utilities.logError(log, "Errore nel parsing del read timeout: " + e.getMessage(), e);
            }
        }
        return 15000;
    }

    private static int getConnectionTimeout(Logger log, Properties loginProperties) {
        String connectTimeout = loginProperties.getProperty("oauth2.connectTimeout");
        if (connectTimeout != null && !connectTimeout.isEmpty()) {
            try {
                return Integer.parseInt(connectTimeout);
            }
            catch (NumberFormatException e) {
                OAuth2Utilities.logError(log, "Errore nel parsing del connection timeout: " + e.getMessage(), e);
            }
        }
        return 10000;
    }

    public static String creaUrlLogout(String idToken, String oauth2LogoutUrl, String redirPageUrl) {
        return oauth2LogoutUrl + OAuth2Utilities.addFirstParameter("id_token_hint", idToken) + OAuth2Utilities.addParameter("post_logout_redirect_uri", redirPageUrl);
    }
}

